/*
 * oauth2-oidc-sdk
 *
 * Copyright 2012-2024, Connect2id Ltd and contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use
 * this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

package com.nimbusds.oauth2.sdk.dpop.verifiers;


import com.nimbusds.oauth2.sdk.id.JWTID;
import net.jcip.annotations.Immutable;

import java.util.Objects;


/**
 * DPoP proof use.
 */
@Immutable
public final class DPoPProofUse {


        private final DPoPIssuer iss;

        private final JWTID jti;

        private final long maxAge;


        /**
         * Creates new DPoP proof use.
         *
         * @param iss    The DPoP proof issuer. Must not be {@code null}.
         * @param jti    The DPoP proof JWT ID. Must not be {@code null}.
         * @param maxAge The maximum accepted DPoP proof "iat" age relative to
         *               the current system time, in seconds. Intended to limit
         *               replay by bounding how long a proof is valid after
         *               issue.
         */
        public DPoPProofUse(final DPoPIssuer iss, final JWTID jti, final long maxAge) {
                this.jti = Objects.requireNonNull(jti);
                this.iss = Objects.requireNonNull(iss);
                if (maxAge < 0) {
                        throw new IllegalArgumentException("The DPoP proof max age must be non-negative");
                }
                this.maxAge = maxAge;
        }


        /**
         * Returns the DPoP proof issuer.
         *
         * @return The issuer.
         */
        public DPoPIssuer getIssuer() {
                return iss;
        }


        /**
         * Returns the DPoP proof JWT ID.
         *
         * @return The JWT ID.
         */
        public JWTID getJWTID() {
                return jti;
        }


        /**
         * Returns the maximum accepted DPoP proof "iat" age relative to the
         * current system time, in seconds. Intended to limit replay by
         * bounding how long a proof is valid after issue.
         *
         * @return The maximum accepted DPoP proof {@code iat} age.
         */
        public long getMaxAge() {
                return maxAge;
        }
}
