package com.newrelic.bootstrap;

import com.newrelic.agent.config.IBMUtils;
import com.newrelic.agent.config.JavaVersionUtils;

import java.lang.instrument.Instrumentation;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public class BootstrapAgent {

    public static final String AGENT_CLASS_NAME = "com.newrelic.agent.Agent";
    private static final String WS_SERVER_JAR = "ws-server.jar";
    private static final String WS_LOG_MANAGER = "com.ibm.ws.kernel.boot.logging.WsLogManager";
    private static final String IBM_VENDOR = "IBM";
    private static final String JDK_INTERNAL_LOADER = "jdk.internal.loader";
    private static final String JAVA_LANG = "java.lang";

    public static URL getAgentJarUrl() {
        return BootstrapAgent.class.getProtectionDomain().getCodeSource().getLocation();
    }

    /**
     * A wrapper around the Agent's main method that makes sure the bootstrap classes are available.
     *
     * @param args
     */
    public static void main(String[] args) {
        try {
            Collection<URL> urls = BootstrapLoader.getJarURLs();
            urls.add(getAgentJarUrl());
            urls.add(BootstrapAgent.class.getProtectionDomain().getCodeSource().getLocation());

            ClassLoader classLoader = new URLClassLoader(urls.toArray(new URL[0]), null);
            Class<?> agentClass = classLoader.loadClass(AGENT_CLASS_NAME);
            Method main = agentClass.getDeclaredMethod("main", String[].class);
            main.invoke(null, new Object[] { args });
        } catch (Throwable t) {
            System.err.println(MessageFormat.format("Error invoking the New Relic command: {0}", t));
            t.printStackTrace();
        }
    }

    /**
     * This is called via the Java 1.5 Instrumentation startup sequence (JSR 163). Boot up the agent.
     * <p>
     * Thanks Mr. Cobb! ;)
     */
    public static void premain(String agentArgs, Instrumentation inst) {
        String javaSpecVersion = JavaVersionUtils.getJavaSpecificationVersion();
        if (!JavaVersionUtils.isAgentSupportedJavaSpecVersion(javaSpecVersion)) {
            System.err.println("----------");
            System.err.println(JavaVersionUtils.getUnsupportedAgentJavaSpecVersionMessage(javaSpecVersion));
            System.err.println("----------");
            return;
        }
        checkAndApplyIBMLibertyProfileLogManagerWorkaround();
        startAgent(agentArgs, inst);
    }

    private static void checkAndApplyIBMLibertyProfileLogManagerWorkaround() {
        String javaVendor = System.getProperty("java.vendor");
        if (javaVendor != null && javaVendor.startsWith(IBM_VENDOR)) {
            String javaClassPath = System.getProperty("java.class.path");
            // WS_SERVER_JAR is characteristic of a Liberty Profile installation
            if (javaClassPath != null && javaClassPath.contains(WS_SERVER_JAR)) {
                if (System.getProperty("java.util.logging.manager") == null) {
                    try {
                        // Check for the existence of WsLogManager (without initializing it)
                        // before attempting the logging manager workaround
                        Class.forName(WS_LOG_MANAGER, false, BootstrapAgent.class.getClassLoader());

                        // This property is used in java.util.logging.LogManager during initialization
                        // and allows us to properly set the logger hierarchy for Liberty Profile.
                        System.setProperty("java.util.logging.manager", WS_LOG_MANAGER);
                    } catch (Exception e) {
                        // WSLogManager was not found, this must not be Liberty
                    }
                }
            }
        }
    }

    static void startAgent(String agentArgs, Instrumentation inst) {
        try {

            // Premain start time will be recorded starting from this point
            long startTime = System.currentTimeMillis();

            BootstrapLoader.load(inst);

            // Check for the IBM workaround
            boolean ibmWorkaround = IBMUtils.getIbmWorkaroundDefault();
            if (System.getProperty("ibm_iv25688_workaround") != null) {
                ibmWorkaround = Boolean.parseBoolean(System.getProperty("ibm_iv25688_workaround"));
            }

            ClassLoader classLoader;
            if (ibmWorkaround) {
                // For the IBM workaround lets just use the System ClassLoader
                classLoader = ClassLoader.getSystemClassLoader();
            } else {
                String javaVersion = System.getProperty("java.version", "");
                ClassLoader agentClassLoaderParent = null;

                try {
                    Class classLoaderClass = ClassLoader.class;
                    Method getPlatformClassLoader = classLoaderClass.getDeclaredMethod("getPlatformClassLoader");
                    agentClassLoaderParent = (ClassLoader) getPlatformClassLoader.invoke(null);
                } catch (Exception e) {
                }

                // Create a new URLClassLoader instance for the agent to use instead of relying on the System ClassLoader
                URL[] codeSource;
                if (javaVersion.startsWith("9") || javaVersion.startsWith("10") || javaVersion.startsWith("11")) {
                    // We want to add the agent-bridge-datastore jar to the agent classloader on Java 9+
                    // See com.newrelic.bootstrap.BootstrapLoader.load()
                    URL url = BootstrapLoader.getDatastoreJarURL();
                    codeSource = new URL[] { BootstrapAgent.class.getProtectionDomain().getCodeSource().getLocation(), url };
                } else {
                    codeSource = new URL[] { BootstrapAgent.class.getProtectionDomain().getCodeSource().getLocation() };
                }

                classLoader = new JVMAgentClassLoader(codeSource, agentClassLoaderParent);

                if (javaVersion.startsWith("9") || javaVersion.startsWith("10") || javaVersion.startsWith("11")) {
                    Class<?> objectClass = Object.class.getClass();
                    Method getModuleMethod = objectClass.getDeclaredMethod("getModule");
                    Object baseModule = getModuleMethod.invoke(objectClass);

                    Class<?> instClass = Instrumentation.class;
                    Method redefineModule = instClass.getDeclaredMethod("redefineModule", baseModule.getClass(),
                            Set.class, Map.class, Map.class, Set.class, Map.class);

                    Class<? extends ClassLoader> bootLoaderClass = classLoader.getClass();
                    Method getUnnamedModule = bootLoaderClass.getMethod("getUnnamedModule");
                    Object bootUnnamedModule = getUnnamedModule.invoke(classLoader);
                    Set<Object> bootUnnamedModuleSet = new HashSet<Object>();
                    bootUnnamedModuleSet.add(bootUnnamedModule);

                    Map<String, Set<Object>> extraOpensAndExports = new HashMap<String, Set<Object>>();
                    extraOpensAndExports.put(JDK_INTERNAL_LOADER, bootUnnamedModuleSet);
                    extraOpensAndExports.put(JAVA_LANG, bootUnnamedModuleSet);

                    redefineModule.invoke(inst, baseModule, Collections.emptySet(), extraOpensAndExports,
                            extraOpensAndExports, Collections.emptySet(), Collections.emptyMap());
                }
            }

            Class<?> agentClass = classLoader.loadClass(AGENT_CLASS_NAME);
            Method premain = agentClass.getDeclaredMethod("premain", String.class, Instrumentation.class, long.class);
            premain.invoke(null, agentArgs, inst, startTime);
        } catch (Throwable t) {
            System.err.println(MessageFormat.format("Error bootstrapping New Relic agent: {0}", t));
            t.printStackTrace();
        }
    }

    private static class JVMAgentClassLoader extends URLClassLoader {
        static {
            try {
                // This is so gross, I'm really sorry (please don't hate me)
                Method method = ClassLoader.class.getDeclaredMethod("registerAsParallelCapable");
                method.setAccessible(true);

                // We need to invoke "registerAsParallelCapable" via reflection in order to properly support JDK6
                // which does not have the registerAsParallelCapable method yet (introduced in JDK7). This allows us
                // to compile our agent with a bootstrap classpath of JDK6 for protection against version compat issues
                method.invoke(null);
            } catch (Throwable t) {
                // Not supported in < JDK7. Ignore errors associated with it
            }
        }

        public JVMAgentClassLoader(URL[] urls, ClassLoader parent) {
            super(urls, parent);
        }
    }

}
