package com.newrelic.api.agent.tracing;

import com.newrelic.agent.Agent;
import com.newrelic.agent.DistributedTracePayloadImpl;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.trace.TransactionGuidFactory;
import com.newrelic.agent.tracing.DistributedTraceUtil;
import com.newrelic.api.agent.DistributedTracePayload;
import com.newrelic.api.agent.NewRelic;

import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

public class SpanProxy {
    private volatile long timestamp;
    private volatile long transportDurationInMillis;

    private final AtomicReference<DistributedTracePayloadImpl> inboundPayloadData = new AtomicReference<>(null);
    private final AtomicReference<DistributedTracePayloadImpl> outboundPayloadData = new AtomicReference<>(null);

    public SpanProxy() {
    }

    public DistributedTracePayload createDistributedTracePayload() {
        String guid = TransactionGuidFactory.generateGuid();
        return createDistributedTracePayload(inboundPayloadData.get() == null ? guid : inboundPayloadData.get().guid,
                DistributedTraceUtil.getPriority().get(), guid, null);
    }

    public DistributedTracePayload createDistributedTracePayload(String traceId, Float priority, String guid, String txnId) {
        try {
            DistributedTracePayloadImpl payload = DistributedTracePayloadImpl.createDistributedTracePayload(inboundPayloadData.get(), traceId, guid, txnId,
                    priority);
            if (payload == null) {
                return null;
            }

            if (Agent.LOG.isFinestEnabled()) {
                Agent.LOG.log(Level.FINEST, "Created distributed trace payload: {0} for transaction: {1}", payload, this);
            }
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_CREATE_PAYLOAD_SUCCESS);

            // This only records the data the first time that we generate "outbound" data
            this.outboundPayloadData.compareAndSet(null, payload);

            return payload;
        } catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Unable to create distributed trace payload");
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_CREATE_PAYLOAD_EXCEPTION);
            return null;
        }
    }

    public boolean acceptDistributedTracePayload(String payload) {
        try {
            DistributedTracePayloadImpl parsedPayload = DistributedTracePayloadImpl.parseDistributedTracePayload(outboundPayloadData.get(), payload);
            return parsedPayload != null && acceptDistributedTracePayloadImpl(parsedPayload);
        } catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Unable to accept distributed trace payload: {0}", payload);
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_EXCEPTION);
        }
        return false;
    }

    public boolean acceptDistributedTracePayload(DistributedTracePayload payload) {
        try {
            // record supportability error if someone called createDistributedTracePayload already
            if (outboundPayloadData.get() != null) {
                Agent.LOG.log(Level.WARNING,
                        "Error: createDistributedTracePayload was called before acceptDistributedTracePayload. Ignoring Call");
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_CREATE_BEFORE_ACCEPT);
                return false;
            }

            if (payload instanceof DistributedTracePayloadImpl) {
                return acceptDistributedTracePayloadImpl((DistributedTracePayloadImpl) payload);
            } else {
                Agent.LOG.log(Level.FINEST, "Unable to accept distributed trace payload. Incorrect type: {0}",
                        payload.getClass().getName());
            }
        } catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Unable to accept distributed trace payload: {0}", payload);
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_EXCEPTION);
        }

        return false;
    }

    private boolean acceptDistributedTracePayloadImpl(DistributedTracePayloadImpl payload) {
        // First payload "accepted" wins
        if (this.inboundPayloadData.compareAndSet(null, payload)) {
            // We'd like to get the transaction start time in milliseconds since the epoch.
            this.transportDurationInMillis = timestamp - payload.timestamp; // check positive

            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_SUCCESS);
            return true;
        } else {
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_MULTIPLE_ACCEPT);
            return false;
        }
    }

    public DistributedTracePayloadImpl getInboundDistributedTracePayload() {
        return inboundPayloadData.get();
    }

    public DistributedTracePayloadImpl getOutboundDistributedTracePayload() {
        return outboundPayloadData.get();
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public long getTransportDurationInMillis() {
        return transportDurationInMillis;
    }

}
