package com.newrelic.agent.xray;

import com.newrelic.agent.Agent;
import com.newrelic.agent.HarvestListener;
import com.newrelic.agent.IRPMService;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Maintains the active collection of X-Ray sessions. This object registers itself to be notified after every
 * harvest cycle, and at that time it expires out old sessions. The processSessionsList method is called from
 * the processEnabled method of StartXRayCommand, which gets triggered from the harvest cycle if
 * get_agent_commands returns any X-Ray commands.
 *
 * This object is not thread-safe.
 */
public class XRaySessionService extends AbstractService implements HarvestListener, IXRaySessionService {

    private final Map<Long, XRaySession> sessions = new HashMap<>();
    private final boolean enabled;
    private final List<XRaySessionListener> listeners = new CopyOnWriteArrayList<>();
    public static final int MAX_SESSION_COUNT = 50; // put a ceiling on the number of X-Ray sessions to avoid
    // overloading the customer server
    public static final long MAX_SESSION_DURATION_SECONDS = (long) (60 * 60 * 24); // 24 hours, in seconds
    public static final long MAX_TRACE_COUNT = 100L;

    public XRaySessionService() {
        super(XRaySessionService.class.getSimpleName());
        enabled = ServiceFactory.getConfigService().getDefaultAgentConfig().isXraySessionEnabled();
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    @Override
    protected void doStart() throws Exception {
        addCommands();
        ServiceFactory.getHarvestService().addHarvestListener(this);
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getHarvestService().removeHarvestListener(this);
    }

    private void addCommands() {
        ServiceFactory.getCommandParser().addCommands(new StartXRayCommand(this));
    }

    private void addSession(XRaySession newSession) {
        if (listeners.size() < MAX_SESSION_COUNT) {
            Agent.LOG.info("Adding X-Ray session: " + newSession.toString());
            this.sessions.put(newSession.getxRayId(), newSession);
            for (XRaySessionListener listener : listeners) {
                listener.xraySessionCreated(newSession);
            }
        } else {
            Agent.LOG.error("Unable to add X-Ray Session because this would exceed the maximum number of concurrent X-Ray Sessions allowed.  Max allowed is "
                    + MAX_SESSION_COUNT);
        }
    }

    private void removeSession(Long sessionId) {
        XRaySession session = sessions.remove(sessionId);
        if (null == session) {
            Agent.LOG.info("Tried to remove X-Ray session " + sessionId + " but no such session exists.");
        } else {
            Agent.LOG.info("Removing X-Ray session: " + session.toString());
            for (XRaySessionListener listener : listeners) {
                listener.xraySessionRemoved(session);
            }
        }
    }

    @Override
    public void beforeHarvest(String appName, StatsEngine statsEngine) {
    }

    @Override
    public void afterHarvest(String appName) {
        Set<Long> expired = new HashSet<>();
        for (XRaySession session : sessions.values()) {
            if (session.sessionHasExpired()) {
                expired.add(session.getxRayId());
                Agent.LOG.debug("Identified X-Ray session for expiration: " + session.toString());
            }
        }
        for (Long key : expired) {
            XRaySession session = sessions.get(key);
            if (null != session) {
                Agent.LOG.info("Expiring X-Ray session: " + session.getxRaySessionName());
                this.removeSession(key);
            }
        }
    }

    void setupSession(Map<?, ?> sessionMap, String applicationName) {
        Long xRayId = null;
        Boolean runProfiler = null;
        String keyTransactionName = null;
        Double samplePeriod = null;
        String xRaySessionName = null;
        Long duration = null;
        Long requestedTraceCount = null;

        Object x_ray_id = sessionMap.remove("x_ray_id");
        if (x_ray_id instanceof Long) {
            xRayId = (Long) x_ray_id;
        }
        Object run_profiler = sessionMap.remove("run_profiler");
        if (run_profiler instanceof Boolean) {
            runProfiler = (Boolean) run_profiler;
        }
        Object key_transaction_name = sessionMap.remove("key_transaction_name");
        if (key_transaction_name instanceof String) {
            keyTransactionName = (String) key_transaction_name;
        }
        Object sample_period = sessionMap.remove("sample_period");
        if (sample_period instanceof Double) {
            samplePeriod = (Double) sample_period;
        }
        Object xray_session_name = sessionMap.remove("xray_session_name");
        if (xray_session_name instanceof String) {
            xRaySessionName = (String) xray_session_name;
        }
        Object duration_obj = sessionMap.remove("duration");
        if (duration_obj instanceof Long) {
            duration = (Long) duration_obj;
            if (duration < 0) {
                duration = 0L; // no negative durations allowed
                Agent.LOG.error("Tried to create an X-Ray Session with negative duration, setting duration to 0");
            } else if (duration > MAX_SESSION_DURATION_SECONDS) {
                Agent.LOG.error("Tried to create an X-Ray session with a duration (" + duration + ") longer than "
                        + MAX_SESSION_DURATION_SECONDS + " seconds.  Setting the duration to "
                        + MAX_SESSION_DURATION_SECONDS + " seconds");
                duration = MAX_SESSION_DURATION_SECONDS;
            }
        }
        Object requested_trace_count = sessionMap.remove("requested_trace_count");
        if (requested_trace_count instanceof Long) {
            requestedTraceCount = (Long) requested_trace_count;
            if (requestedTraceCount > MAX_TRACE_COUNT) {
                Agent.LOG.error("Tried to create an X-Ray session with a requested trace count (" + requestedTraceCount
                        + ") larger than " + MAX_TRACE_COUNT + ".  Setting the max trace count to " + MAX_TRACE_COUNT);
                requestedTraceCount = MAX_TRACE_COUNT;
            } else if (requestedTraceCount < 0) {
                Agent.LOG.error("Tried to create an X-Ray Session with negative trace count, setting trace count to 0");
                requestedTraceCount = 0L;
            }
        }
        XRaySession newSession = new XRaySession(xRayId, runProfiler, keyTransactionName, samplePeriod,
                xRaySessionName, duration, requestedTraceCount, applicationName);
        addSession(newSession);
    }

    /**
     * This method accepts a list of Longs which are assumed to be X-Ray session IDs. It does the necessary adds and
     * deletes to make our local store match that new list of active X-Ray session IDs. For new sessions being added,
     * the metadata is fetched from RPM in order to populate details about the session via a call to get_xray_metadata
     * on the collector.
     *
     * @param incomingList
     * @param rpmService
     * @return result value to be returned from startXrayCommand.processRunning()
     */
    public Map<?, ?> processSessionsList(List<Long> incomingList, IRPMService rpmService) {
        Set<Long> sessionIdsToAdd = new HashSet<>();
        Set<Long> sessionIdsToRemove = new HashSet<>();

        String applicationName = rpmService.getApplicationName();

        // items which are in incomingList but are not in the sessions map need to be added
        for (Long id : incomingList) {
            if (!sessions.keySet().contains(id)) {
                // new one, add it top the list to pass to the collector query
                sessionIdsToAdd.add(id);
            }
        }

        // items which are in the sessions map but not in incomingList need to be removed from sessions map
        for (long id : sessions.keySet()) {
            if (!incomingList.contains(id)) {
                // needs to be removed, do the needful. Don't manipulate the sessions map while also iterating over it.
                Agent.LOG.debug("Identified " + id + " for removal from the active list of X-Ray sessions");
                sessionIdsToRemove.add(id);
            }
        }

        // do removes
        if (sessionIdsToRemove.size() > 0) {
            Agent.LOG.debug("Removing " + sessionIdsToRemove + " from the active list of X-Ray sessions");
            for (Long id : sessionIdsToRemove) {
                removeSession(id);
            }
        }

        // do adds: query the collector for the detailed metadata for the new sessions
        if (sessionIdsToAdd.size() > 0) {
            Agent.LOG.debug("Fetching details for " + sessionIdsToAdd + " to add to the active list of X Ray Sessions");

            Collection<?> newSessionDetails;
            try {
                newSessionDetails = rpmService.getXRaySessionInfo(sessionIdsToAdd);
            } catch (Exception e) {
                // HttpError/LicenseException handled here
                Agent.LOG.error("Unable to fetch X-Ray session details from RPM" + e.getMessage());
                return Collections.EMPTY_MAP;

            }
            for (Object newSession : newSessionDetails) {
                if (newSession instanceof Map) {
                    // if it isn't something has gone wrong with our JSON fetch
                    Map<?, ?> newSessionMap = (Map<?, ?>) newSession;
                    // adds it to our collection and notifies our listeners
                    this.setupSession(newSessionMap, applicationName);
                } else {
                    Agent.LOG.error("Unable to read X-Ray session details: " + newSession);
                }

            }
        }
        Agent.LOG.debug("Resulting collection of X-Ray sessions: " + sessions);
        return Collections.EMPTY_MAP;
    }

    public void addListener(XRaySessionListener listener) {
        listeners.add(listener);
    }

    public void removeListener(XRaySessionListener listener) {
        listeners.remove(listener);
    }

}
