/**
 * 
 */
package com.newrelic.agent.xray;

import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Contains metadata about an X-Ray session. Managed by XRaySessionService.
 * 
 * This object is thread safe.
 * 
 */
public class XRaySession {
    private final Long xRayId;
    private final boolean runProfiler;
    private final String keyTransactionName;
    private final long samplePeriodMillis;
    private final String xRaySessionName;
    private final long durationMilliseconds;
    private final long requestedTraceCount;
    private final String applicationName;

    private final long sessionEndTimeInNanos;
    private final AtomicInteger collectedTraceCount = new AtomicInteger(0);
    private final String endTimeForDisplay;

    public XRaySession(Long xRayId, boolean runProfiler, String keyTransactionName, double samplePeriod,
            String xRaySessionName, Long durationSeconds, Long requestedTraceCount, String applicationName) {
        super();
        this.xRayId = xRayId;
        this.runProfiler = runProfiler;
        this.keyTransactionName = keyTransactionName;
        this.samplePeriodMillis = (long) (samplePeriod * 1000);
        this.xRaySessionName = xRaySessionName;
        this.durationMilliseconds = TimeUnit.SECONDS.toMillis(durationSeconds);
        this.requestedTraceCount = requestedTraceCount;
        this.sessionEndTimeInNanos = System.nanoTime() + TimeUnit.SECONDS.toNanos(durationSeconds);
        // only used in toString but there's no point in computing it every time
        endTimeForDisplay = new Date(TimeUnit.NANOSECONDS.toMillis(sessionEndTimeInNanos)).toString();
        this.applicationName = applicationName;
    }

    public void incrementCount() {
        this.collectedTraceCount.incrementAndGet();
    }

    public Long getxRayId() {
        return xRayId;
    }

    public boolean isRunProfiler() {
        return runProfiler;
    }

    public String getKeyTransactionName() {
        return keyTransactionName;
    }

    public long getSamplePeriodMilliseconds() {
        return samplePeriodMillis;
    }

    public String getxRaySessionName() {
        return xRaySessionName;
    }

    public long getDurationMilliseconds() {
        return durationMilliseconds;
    }

    public long getRequestedTraceCount() {
        return requestedTraceCount;
    }

    public long getSessionEndTimeInNanos() {
        return sessionEndTimeInNanos;
    }

    public long getCollectedTraceCount() {
        return this.collectedTraceCount.get();
    }

    public String getApplicationName() {
        return applicationName;
    }

    public boolean sessionHasExpired() {
        return ((this.collectedTraceCount.get() >= this.requestedTraceCount) || (System.nanoTime() > this.sessionEndTimeInNanos));
    }

    @Override
    public String toString() {
        return String.format(
                "XRaySession [xRayId=%s, applicationName=%s, runProfiler=%s, keyTransactionName=%s, samplePeriodMilliseconds=%s, xRaySessionName=%s, durationMilliseconds=%s, requestedTraceCount=%s, sessionEndTimeInMillis=%s, collectedTraceCount=%s, derived end time=%s]",
                xRayId, applicationName, runProfiler, keyTransactionName, samplePeriodMillis, xRaySessionName,
                durationMilliseconds, requestedTraceCount, sessionEndTimeInNanos, collectedTraceCount,
                endTimeForDisplay);
    }

}
