package com.newrelic.agent.utilization;

import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsWorks;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.ssl.StrictHostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

import java.io.IOException;

import static com.google.common.base.CharMatcher.ascii;

public class CloudUtility {

    // Spec: All characters should be in the following character class: over U+007F
    private static final int MIN_CHAR_CODEPOINT = "\u007F".codePointAt(0);

    private CloudUtility() {
    }

    public static String makeHttpRequest(String url, int requestTimeoutMillis, String... headers) throws IOException {
        CloseableHttpClient httpclient = null;
        try {
            httpclient = configureHttpClient(requestTimeoutMillis);

            HttpGet httpGet = new HttpGet(url);
            for (String header : headers) {
                String[] parts = header.split(":");
                httpGet.addHeader(parts[0].trim(), parts[1].trim());
            }

            CloseableHttpResponse response = httpclient.execute(httpGet);
            // status code should be in the 200s
            if (response.getStatusLine().getStatusCode() <= HttpStatus.SC_MULTI_STATUS) {
                return EntityUtils.toString(response.getEntity(), "UTF-8");
            }
        } finally {
            if (httpclient != null) {
                try {
                    httpclient.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    private static CloseableHttpClient configureHttpClient(int requestTimeoutInMillis) {
        HttpClientBuilder builder = HttpClientBuilder.create();
        builder.setDefaultSocketConfig(SocketConfig.custom().setSoTimeout(requestTimeoutInMillis).setSoKeepAlive(true).build());
        RequestConfig.Builder requestBuilder = RequestConfig.custom().setConnectTimeout(requestTimeoutInMillis)
                .setConnectionRequestTimeout(requestTimeoutInMillis).setSocketTimeout(requestTimeoutInMillis);
        builder.setDefaultRequestConfig(requestBuilder.build());
        builder.setHostnameVerifier(new StrictHostnameVerifier());
        return builder.build();
    }

    public static void recordError(String metricName) {
        ServiceFactory.getStatsService().doStatsWork(StatsWorks.getIncrementCounterWork(metricName, 1));
    }

    /**
     * @param value ASCII String.
     * @return true if all chars in value are ASCII, false otherwise.
     */
    static boolean isAscii(String value) {
        return ascii().matchesAllOf(value);
    }

    /**
     * @param value UTF-8 encoded String.
     * @return true if the value is invalid, false if it is valid.
     */
    static boolean isInvalidValue(String value) {
        /*
         * No value should be longer than 255 bytes when encoded as a UTF-8 string.
         *
         * characters should be in the following character class: over U+007F or [0-9a-zA-Z_ ./-]. Notice that the space
         * character is in that character class, and is thus allowed.
         */
        if (value == null) {
            return true;
        }

        if (value.getBytes().length > 255) {
            return true;
        }

        for (int i = 0; i < value.length(); i++) {
            char c = value.charAt(i);

            if (c >= '0' && c <= '9') {
                continue;
            }

            if (c >= 'a' && c <= 'z') {
                continue;
            }

            if (c >= 'A' && c <= 'Z') {
                continue;
            }

            if (c == ' ' || c == '_' || c == '.' || c == '/' || c == '-') {
                continue;
            }

            if (c > MIN_CHAR_CODEPOINT) {
                continue;
            }

            // Invalid character
            return true;
        }

        return false;
    }

}
