package com.newrelic.agent.utilization;

import com.newrelic.agent.Agent;
import com.newrelic.agent.MetricNames;
import org.apache.http.conn.ConnectTimeoutException;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;

import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;

public class AWS implements CloudVendor {
    static String PROVIDER = "aws";

    private static final String INSTANCE_DOCUMENT_URL = "http://169.254.169.254/2016-09-02/dynamic/instance-identity/document";
    private static int REQUEST_TIMEOUT_MILLIS = 100;

    // AWS request strings
    private static final String AWS_INSTANCE_ID_REQUEST = "instanceId";
    private static final String AWS_INSTANCE_TYPE_REQUEST = "instanceType";
    private static final String AWS_AVAILABILITY_ZONE_REQUEST = "availabilityZone";

    // AWS map keys. These are the keys that will be added to the vendor hash in the JSON generated by the agent.
    private static final String AWS_INSTANCE_ID_KEY = "instanceId";
    private static final String AWS_INSTANCE_TYPE_KEY = "instanceType";
    private static final String AWS_AVAILABILITY_ZONE_KEY = "availabilityZone";

    /**
     * Query the AWS API to get metadata
     *
     * @return {@link AwsData} with values for instanceId, instanceType, and availabilityZone
     */
    public AwsData getData() {
        try {
            String unparsedResult = getAwsValues();

            JSONParser parser = new JSONParser();
            JSONObject result = null;

            if (unparsedResult != null) {
                result = (JSONObject) parser.parse(unparsedResult);
            }

            // not on AWS
            if (result == null || result.isEmpty()) {
                return AwsData.EMPTY_DATA;
            }

            String type = (String) result.get(AWS_INSTANCE_TYPE_REQUEST);
            String id = (String) result.get(AWS_INSTANCE_ID_REQUEST);
            String zone = (String) result.get(AWS_AVAILABILITY_ZONE_REQUEST);

            if (CloudUtility.isInvalidValue(type) || CloudUtility.isInvalidValue(id)
                    || CloudUtility.isInvalidValue(zone)) {
                Agent.LOG.log(Level.WARNING, "Failed to validate AWS value");
                recordAwsError();
                return AwsData.EMPTY_DATA;
            }

            AwsData data = new AwsData(id, type, zone);
            Agent.LOG.log(Level.FINEST, "Found {0}", data);
            return data;
        } catch (Exception e) {
            return AwsData.EMPTY_DATA;
        }
    }

    protected String getAwsValues() {
        try {
            return CloudUtility.makeHttpRequest(INSTANCE_DOCUMENT_URL, REQUEST_TIMEOUT_MILLIS);
        } catch (ConnectTimeoutException e) {
            // Request timed out. This is not AWS.
        } catch (UnknownHostException e) {
            // Unknown environment. This is not AWS.
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Error occurred trying to get AWS values. {0}", t));
            recordAwsError();
        }
        return null;
    }

    private static void recordAwsError() {
        CloudUtility.recordError(MetricNames.SUPPORTABILITY_AWS_ERROR);
    }

    protected static class AwsData implements CloudData {
        private final String instanceId;
        private final String instanceType;
        private final String availabilityZone;

        static final AwsData EMPTY_DATA = new AwsData();

        private AwsData() {
            instanceId = null;
            instanceType = null;
            availabilityZone = null;
        }

        protected AwsData(String id, String type, String zone) {
            instanceId = id;
            instanceType = type;
            availabilityZone = zone;
        }

        public String getInstanceId() {
            return instanceId;
        }

        public String getInstanceType() {
            return instanceType;
        }

        public String getAvailabilityZone() {
            return availabilityZone;
        }

        public Map<String, String> getValueMap() {
            Map<String, String> aws = new HashMap<String, String>();

            if (instanceType == null || instanceId == null || availabilityZone == null) {
                return aws;
            } else {
                aws.put(AWS_INSTANCE_TYPE_KEY, instanceType);
                aws.put(AWS_INSTANCE_ID_KEY, instanceId);
                aws.put(AWS_AVAILABILITY_ZONE_KEY, availabilityZone);
            }
            return aws;
        }

        public String getProvider() {
            return PROVIDER;
        }

        public boolean isEmpty() {
            return this == EMPTY_DATA;
        }

        @Override
        public String toString() {
            return "AwsData{" +
                    "instanceId='" + instanceId + '\'' +
                    ", instanceType='" + instanceType + '\'' +
                    ", availabilityZone='" + availabilityZone + '\'' +
                    '}';
        }
    }

}
