package com.newrelic.agent.util;

import java.io.UnsupportedEncodingException;

import org.apache.commons.codec.binary.Base64;

/**
 * This class obfuscates and deobfuscates strings using a given key. It is used by the real user monitoring feature to
 * obfuscate transaction names inserted into web pages, and used in response headers to obfuscate the app data returned
 * to the calling agent.
 * 
 * @author sdaubin
 */
public class Obfuscator {

    private Obfuscator() {
    }

    /**
     * Obfuscates a name using the given key. Note that when the RUM feature calls this api it only sends the first 13
     * characters of the browser monitoring key, while the cross process callers expect the full key to be used.
     * 
     * @throws UnsupportedEncodingException
     */
    public static String obfuscateNameUsingKey(String name, String key) throws UnsupportedEncodingException {
        byte[] encodedBytes = name.getBytes("UTF-8");
        byte[] keyBytes = key.getBytes("UTF-8");
        return Base64.encodeBase64String(encode(encodedBytes, keyBytes));
    }

    private static byte[] encode(byte[] bytes, byte[] keyBytes) {
        for (int i = 0; i < bytes.length; i++) {
            bytes[i] = (byte) (bytes[i] ^ keyBytes[i % keyBytes.length]);
        }
        return bytes;
    }

    /**
     * Deobfuscates a name using the given key.
     * 
     * @throws UnsupportedEncodingException
     */
    public static String deobfuscateNameUsingKey(String name, String key) throws UnsupportedEncodingException {
        byte[] bytes = Base64.decodeBase64(name);
        byte[] keyBytes = key.getBytes("UTF-8");
        return new String(encode(bytes, keyBytes), "UTF-8");
    }
}
