package com.newrelic.agent.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Pattern;

import org.objectweb.asm.ClassReader;

import com.newrelic.agent.bridge.AgentBridge;
import org.reflections.Reflections;
import org.reflections.util.ConfigurationBuilder;

import com.google.common.collect.Sets;
import com.newrelic.agent.Agent;
import com.newrelic.agent.config.AgentJarHelper;
import com.newrelic.agent.config.JarResource;
import com.newrelic.agent.util.asm.ClassStructure;
import com.newrelic.weave.utils.Streams;

public class Annotations {

    private static Reflections loaded;

    private Annotations() {
    }

    /**
     * Returns a collection of classes that bear the mark of the given annotation class.
     * 
     * @param annotationClass
     * @param packageSearchPath
     * @return
     */
    public static Collection<Class<?>> getAnnotationClasses(Class<?> annotationClass, String packageSearchPath) {
        String pointcutAnnotation = 'L' + annotationClass.getName().replace('.', '/') + ';';
        if (!packageSearchPath.endsWith("/")) {
            packageSearchPath = packageSearchPath + "/";
        }
        Pattern pattern = Pattern.compile(packageSearchPath + "(.*).class");
        ClassLoader classLoader = AgentBridge.getAgent().getClass().getClassLoader();
        JarResource agentJarFile = AgentJarHelper.getAgentJarResource();
        try {
            Collection<String> fileNames = AgentJarHelper.findAgentJarFileNames(pattern);
            Collection<Class<?>> classes = new ArrayList<Class<?>>(fileNames.size());
            for (String fileName : fileNames) {
                int size = (int) agentJarFile.getSize(fileName);
                ByteArrayOutputStream out = new ByteArrayOutputStream(size);
                try {
                    Streams.copy(agentJarFile.getInputStream(fileName), out, size, true);
                    ClassReader cr = new ClassReader(out.toByteArray());
                    ClassStructure structure = ClassStructure.getClassStructure(cr, ClassStructure.CLASS_ANNOTATIONS);
                    Collection<String> annotations = structure.getClassAnnotations().keySet();
                    if (annotations.contains(pointcutAnnotation)) {
                        String className = fileName.replace('/', '.');
                        int index = className.indexOf(".class");
                        if (index > 0) {
                            className = className.substring(0, index);
                        }
                        Class<?> clazz = classLoader.loadClass(className);
                        classes.add(clazz);
                    }
                } catch (Exception e) {
                    Agent.LOG.log(Level.FINEST, e, e.toString());
                }
            }
            return classes;
        } finally {
            try {
                agentJarFile.close();
            } catch (IOException e) {
                Agent.LOG.log(Level.FINEST, e, e.toString());
            }
        }

    }

    /**
     * Returns a collection of classes that bear the mark of the given annotation class.
     *
     * @param annotationClass
     * @param packageSearchPath
     * @return
     */
    public static Collection<Class<?>> getAnnotationClassesFromManifest(Class<? extends Annotation> annotationClass,
            String packageSearchPath) {

        if (loaded == null) {
            JarResource agentJarFile = AgentJarHelper.getAgentJarResource();

            try {
                ConfigurationBuilder builder = new ConfigurationBuilder().setSerializer(
                        new EncodingAwareJsonSerializer());
                builder.setClassLoaders(new ClassLoader[] { AgentBridge.getAgent().getClass().getClassLoader() });
                Reflections loader = new Reflections(builder);
                loaded = loader.collect(agentJarFile.getInputStream("newrelic-manifest.json"));
            } catch (Exception e) {
                return Collections.emptySet();
            } finally {
                try {
                    agentJarFile.close();
                } catch (IOException e) {
                    Agent.LOG.log(Level.FINEST, e, e.toString());
                }
            }
        }

        Set<Class<?>> annotationClasses = loaded.getTypesAnnotatedWith(annotationClass);
        packageSearchPath = packageSearchPath.replaceAll("/", ".");
        Set<Class<?>> filteredAnnotationClasses = Sets.newHashSetWithExpectedSize(annotationClasses.size());
        for (Class<?> annotationClassValue : annotationClasses) {
            if (annotationClassValue.getName().startsWith(packageSearchPath)) {
                filteredAnnotationClasses.add(annotationClassValue);
            }
        }
        return filteredAnnotationClasses;
    }
}
