/**
 * 
 */
package com.newrelic.agent.trace;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import com.newrelic.agent.TransactionData;
import com.newrelic.agent.xray.XRaySession;

/**
 * Sampler which gathers up TTs corresponding to an existing X-Ray session.
 * 
 * This class also has the ability to examine a TT and associate it with our X-Ray session if it was gathered up by a
 * different sampler. This scenario would happen if the XRayTransactionSampler for a given X-Ray session reached it's
 * limit for a given harvest cycle, but then a different sampler picked up a TT for a transaction which was interesting
 * in some way. If that TT was one that this sampler would have picked up if it hadn't reached it's quota we can
 * retroactively associate it with this X-Ray session. That could result in the X-Ray session going over it's quota, but
 * it's better to pick up a few extra TTs than to have transactions that we know are relevant to an X-Ray session but we
 * don't associate them with the X-Ray session.
 * 
 * 
 * @author ddelany
 * 
 */
public class XRayTransactionSampler implements ITransactionSampler {

    static final int TRACES_TO_KEEP = 10; // not private so that unit test can read it

    private final XRaySession session;
    private final String applicationName;
    private final List<TransactionData> data = new CopyOnWriteArrayList<TransactionData>();

    public XRayTransactionSampler(XRaySession session) {
        this.session = session;
        this.applicationName = session.getApplicationName();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.newrelic.agent.trace.ITransactionSampler#noticeTransaction(com.newrelic.agent.TransactionData)
     */
    @Override
    public boolean noticeTransaction(TransactionData td) {
        if (session.sessionHasExpired()) {
            return false;
        }
        if (data.size() >= TRACES_TO_KEEP) {
            return false;
        }
        String appName = td.getApplicationName();
        String transactionName = td.getBlameOrRootMetricName();
        if (isTransactionOfInterest(appName, transactionName)) {
            data.add(td);
            session.incrementCount();
            return true;
        } else {
            return false;
        }
    }

    boolean isTransactionOfInterest(String appName, String transactionName) {
        if (!(applicationName.equals(appName))) {
            return false;
        }
        if (!transactionName.equals(session.getKeyTransactionName())) {
            return false;
        }
        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.newrelic.agent.trace.ITransactionSampler#harvest(java.lang.String)
     */
    @Override
    public List<TransactionTrace> harvest(String appName) {
        List<TransactionTrace> tracesToReturn = new ArrayList<TransactionTrace>();
        for (TransactionData td : data) {
            TransactionTrace trace = TransactionTrace.getTransactionTrace(td);
            trace.setXraySessionId(session.getxRayId());
            tracesToReturn.add(trace);
        }
        data.clear();
        return tracesToReturn;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.newrelic.agent.trace.ITransactionSampler#stop()
     */
    @Override
    public void stop() {
        // TODO Auto-generated method stub

    }

}
