package com.newrelic.agent.service.module;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Array;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;

import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.TypePath;

import com.newrelic.agent.Agent;
import com.newrelic.bootstrap.BootstrapLoader;
import com.newrelic.weave.utils.Streams;
import com.newrelic.weave.utils.WeaveUtils;

public class ClassMetadataFiles {
    /**
     * We stick the ClassInfo annotation onto class metadata files.  The sha1Checksum attribute of 
     * the annotation contains the checksum of the original class file. 
     */
    static final String CLASS_INFO_ANNOTATION = "Lcom/newrelic/agent/ClassInfo;";

    private ClassMetadataFiles() {
    }

    /**
     * Takes a url pointing to a jar file and writes the class metadata (annotations on the class,
     * class methods, annotations on methods, etc) into a new class metadata file (cmd). 
     * The new cmd file has all of the structural information about the classes but none of the code.
     */
    public static File createClassMetadataFile(URL url) throws IOException {
        File outFile = File.createTempFile("struct", ".cmd", BootstrapLoader.getTempDir());
        outFile.deleteOnExit();

        JarInputStream in = new JarInputStream(EmbeddedJars.getInputStream(url));
        try {
            Manifest manifest = in.getManifest();
            if (null == manifest) {
                manifest = new Manifest();
            }
            OutputStream out = new BufferedOutputStream(new FileOutputStream(outFile));
            JarOutputStream jarOut = new JarOutputStream(out, manifest);
            try {

                for (JarEntry entry = in.getNextJarEntry(); entry != null; entry = in.getNextJarEntry()) {

                    // only copy class files, skipping package-info classes
                    if (entry.getName().endsWith(".class") && !entry.getName().endsWith("/package-info.class")) {
                        JarEntry newEntry = new JarEntry(entry.getName());
                        jarOut.putNextEntry(newEntry);

                        try {
                            ByteArrayOutputStream classBytes = new ByteArrayOutputStream(Streams.DEFAULT_BUFFER_SIZE);
                            Streams.copy(in, classBytes, false);

                            writeClassMetadata(classBytes.toByteArray(), jarOut);
                        } finally {
                            jarOut.closeEntry();
                        }
                    }
                }
            } finally {
                jarOut.close();
            }

            return outFile;
        } finally {
            in.close();
        }
    }

    private static void writeClassMetadata(byte[] classBytes, OutputStream out) throws IOException {

        // compute the sha1checksum of the original file. we can use this to find the file in public repositories.
        // we can also use it to detect when classes change across deploys.
        String sha1checksum = null;
        try {
            sha1checksum = ShaChecksums.computeSha(new ByteArrayInputStream(classBytes));
        } catch (NoSuchAlgorithmException e) {
            Agent.LOG.log(Level.FINEST, e, "{0}", e.getMessage());
        }

        ClassReader cr = new ClassReader(classBytes);
        ClassWriter cw = new ClassWriter(ClassWriter.COMPUTE_FRAMES);

        ClassVisitor cv = new ClassVisitor(WeaveUtils.ASM_API_LEVEL, cw) {

            @Override
            public FieldVisitor visitField(int access, String name, String desc, String signature, Object value) {
                // keep fields but strip out value
                return super.visitField(access, name, desc, signature, null);
            }

        };

        if (sha1checksum != null) {
            final String checksum = sha1checksum;
            cv = new ClassVisitor(WeaveUtils.ASM_API_LEVEL, cv) {

                @Override
                public void visit(int version, int access, String name, String signature, String superName,
                        String[] interfaces) {
                    super.visit(version, access, name, signature, superName, interfaces);

                    AnnotationVisitor av = super.visitAnnotation(CLASS_INFO_ANNOTATION, true);

                    av.visit(JarCollectorServiceProcessor.SHA1_CHECKSUM_KEY, checksum);
                    av.visitEnd();
                }

                @Override
                public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
                    return new TruncatingAnnotationVisitor(super.visitAnnotation(desc, visible));
                }

                @Override
                public AnnotationVisitor visitTypeAnnotation(int typeRef, TypePath typePath, String desc,
                        boolean visible) {
                    return new TruncatingAnnotationVisitor(super.visitTypeAnnotation(typeRef, typePath, desc, visible));
                }

                @Override
                public MethodVisitor visitMethod(int access, String name, String desc, String signature,
                        String[] exceptions) {
                    return new MethodVisitor(WeaveUtils.ASM_API_LEVEL, super.visitMethod(access, name, desc, signature, exceptions)) {

                        @Override
                        public AnnotationVisitor visitAnnotationDefault() {
                            return new TruncatingAnnotationVisitor(super.visitAnnotationDefault());
                        }

                        @Override
                        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
                            return new TruncatingAnnotationVisitor(super.visitAnnotation(desc, visible));
                        }
                        
                    };
                }
                
                
            };
        }

        // skipping the code with the reader is critical
        cr.accept(cv, ClassReader.SKIP_CODE);
        out.write(cw.toByteArray());
    }

    static final int ANNOTATION_VALUE_LIMIT = 255; 
    static Object truncateAnnotationValue(Object value) {
        if (value instanceof String) {
            if (value.toString().length() > ANNOTATION_VALUE_LIMIT) {
                value = value.toString().substring(0, ANNOTATION_VALUE_LIMIT);
            }
        } else if (value.getClass().isArray()) {
            if (value.getClass().getComponentType().isPrimitive() && byte.class.isAssignableFrom(value.getClass().getComponentType())) {
                // throw away byte arrays.  c'mon scala!
                value = new byte[0];
            } else if (Array.getLength(value) > ANNOTATION_VALUE_LIMIT) {
                try {
                    Object newArray = Array.newInstance(value.getClass().getComponentType(), ANNOTATION_VALUE_LIMIT);
                    System.arraycopy(value, 0, newArray, 0, ANNOTATION_VALUE_LIMIT);
                    value = newArray;
                } catch (Exception e) {
                    Agent.LOG.log(Level.FINEST, e, e.getMessage());
                }
            }
        }
        return value;
    }

    private static class TruncatingAnnotationVisitor extends AnnotationVisitor {

        public TruncatingAnnotationVisitor(AnnotationVisitor annotationVisitor) {
            super(WeaveUtils.ASM_API_LEVEL, annotationVisitor);
        }

        @Override
        public void visit(String name, Object value) {
            value = truncateAnnotationValue(value);
            super.visit(name, value);
        }
        
    }
}
