package com.newrelic.agent.service.analytics;

import com.newrelic.agent.Harvestable;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsWork;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.transport.DataSenderImpl;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class TransactionEventHarvestableImpl implements Harvestable {

    public TransactionEventsService transactionEventsService;
    public String appName;
    private long lastHarvest;

    TransactionEventHarvestableImpl(TransactionEventsService transactionEventsService, String appName) {
        this.transactionEventsService = transactionEventsService;
        this.appName = appName;
        this.lastHarvest = System.nanoTime();
    }

    @Override
    public String getEndpointMethodName() {
        return DataSenderImpl.ANALYTIC_DATA_METHOD;
    }

    @Override
    public void harvest() {
        recordIntervalMetric();
        transactionEventsService.harvestEvents(appName);
    }

    private void recordIntervalMetric() {
        long startTimeInNanos = System.nanoTime();
        final long harvestIntervalInNanos = startTimeInNanos - lastHarvest;
        lastHarvest = startTimeInNanos;

        ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
            @Override
            public void doWork(StatsEngine statsEngine) {
                if (harvestIntervalInNanos > 0) {
                    statsEngine.getResponseTimeStats(
                            MetricNames.SUPPORTABILITY_TRANSACTION_EVENT_SERVICE_EVENT_HARVEST_INTERVAL).recordResponseTime(
                            harvestIntervalInNanos, TimeUnit.NANOSECONDS);
                }
            }

            @Override
            public String getAppName() {
                return appName;
            }
        });
    }

    @Override
    public String getAppName() {
        return appName;
    }

    @Override
    public void configure(Map<String, Object> configuration) {
        /*
         * There are two ways to configure the max number of transaction events we store. One is
         * transaction_events.max_samples_stored, the other one is max_samples_stored that may come down on connect in
         * the data_methods.analytic_event_data block. We grab the min of these two values.
         *
         * See: https://source.datanerd.us/agents/agent-specs/blob/master/Transaction-Events-PORTED.md
         * https://source.datanerd.us/agents/agent-specs/blob/master/Connect-LEGACY.md#data-methods
         */
        int maxSamplesConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getTransactionEventsConfig().getMaxSamplesStored();
        Number maxSamplesCollector = (Number) configuration.get(InsightsService.MAX_SAMPLES_STORED_KEY_NAME);

        int maxSamples = maxSamplesCollector != null ?
                Math.min(maxSamplesCollector.intValue(), maxSamplesConfig) :
                maxSamplesConfig;

        Number reportPeriod = (Number) configuration.get(Harvestable.REPORT_PERIOD_IN_SECONDS_KEY_NAME);

        ServiceFactory.getStatsService().doStatsWork(StatsWorks.getRecordMetricWork(
                MetricNames.SUPPORTABILITY_TRANSACTION_EVENT_SERVICE_REPORT_PERIOD_IN_SECONDS, reportPeriod.intValue()));

        if (maxSamplesConfig != maxSamples) {
            transactionEventsService.updateMaxSamplesStored(maxSamples);
            transactionEventsService.harvestPendingEvents();
            transactionEventsService.clearReservoirForApp();
        }
    }

}
