package com.newrelic.agent.service.analytics;

import com.newrelic.agent.attributes.AttributeSender;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.database.SqlObfuscator;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.api.agent.DatastoreParameters;
import com.newrelic.api.agent.ExternalParameters;
import com.newrelic.api.agent.HttpParameters;
import com.newrelic.api.agent.SlowQueryDatastoreParameters;
import org.json.simple.JSONArray;

import java.io.IOException;
import java.io.Writer;
import java.net.URI;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class SpanEvent extends AnalyticsEvent {

    private static final String SPAN = "Span";
    private static final String SPAN_KIND = "client";

    // Truncate `db.statement` at 2000 characters
    private static final int DB_STATEMENT_TRUNCATE_LENGTH = 2000;

    private final String appName;
    private final Map<String, Object> intrinsics;
    private final Map<String, Object> userAttributes;
    private final Map<String, Object> agentAttributes;
    private final boolean decider;

    private SpanEvent(String appName, float priority, Map<String, Object> intrinsics, Map<String, Object> userAttributes, Map<String, Object> agentAttributes,
            boolean decider) {
        super(SPAN, System.currentTimeMillis(), priority);
        this.appName = appName;
        this.intrinsics = intrinsics;
        this.userAttributes = userAttributes;
        this.agentAttributes = agentAttributes;
        this.decider = decider;
    }

    public Map<String, Object> getIntrinsics() {
        return intrinsics;
    }

    public String getAppName() {
        return appName;
    }

    public Map<String, Object> getUserAttributes() {
        return userAttributes;
    }

    public Map<String, Object> getAgentAttributes() {
        return agentAttributes;
    }

    public boolean isDecider() {
        return decider;
    }

    @SuppressWarnings("unchecked")
    @Override
    public void writeJSONString(Writer out) throws IOException {
        Map<String, ? extends Object> filteredUserAtts = getUserFilteredMap(userAttributes);
        Map<String, ? extends Object> filteredAgentAtts = getFilteredMap(agentAttributes);
        JSONArray.writeJSONString(Arrays.asList(intrinsics, filteredUserAtts, filteredAgentAtts), out);
    }

    private Map<String, ? extends Object> getFilteredMap(Map<String, Object> input) {
        return ServiceFactory.getAttributesService().filterEventAttributes(appName, input);
    }

    private Map<String, ? extends Object> getUserFilteredMap(Map<String, Object> input) {
        // user attributes should have already been filtered for high security - this is just extra protection
        // high security is per an account - meaning it can not be different for various application names within a
        // JVM - so we can just check the default agent config
        if (!ServiceFactory.getConfigService().getDefaultAgentConfig().isHighSecurity()) {
            return getFilteredMap(input);
        } else {
            return Collections.emptyMap();
        }
    }

    public static SpanEventBuilder builder() {
        return SpanEventBuilder.builder();
    }

    public String getTraceId() {
        return (String) intrinsics.get("traceId");
    }

    public String getGuid() {
        return (String) intrinsics.get("guid");
    }

    public String getParentId() {
        return (String) intrinsics.get("parentId");
    }

    public String getName() {
        return (String) intrinsics.get("name");
    }

    public float getDuration() {
        return (Float) intrinsics.get("duration");
    }

    public String getTransactionId() {
        return (String) intrinsics.get("transactionId");
    }

    public SpanCategory getCategory() {
        return SpanCategory.fromString((String) intrinsics.get("category"));
    }

    public static final class SpanEventBuilder {
        private String appName;
        private float priority;
        private Map<String, Object> userAttributes;
        private Map<String, Object> agentAttributes;
        private Map<String, Object> intrinsics;
        private boolean decider;
        private boolean isRoot;

        private SpanEventBuilder() {
            intrinsics = new HashMap<String, Object>();
            intrinsics.put("type", SPAN);
            intrinsics.put("category", SpanCategory.generic.name());
        }

        private static SpanEventBuilder builder() {
            return new SpanEventBuilder();
        }

        public SpanEventBuilder setPriority(float priority) {
            this.priority = priority;
            intrinsics.put("priority", priority);
            return this;
        }

        public SpanEventBuilder setParentType(String parentType) {
            if (parentType != null) {
                intrinsics.put("parent.type", parentType);
            }
            return this;
        }

        public SpanEventBuilder setParentAccount(String parentAccount) {
            if (parentAccount != null) {
                intrinsics.put("parent.account", parentAccount);
            }
            return this;
        }

        public SpanEventBuilder setParentAppId(String parentAppId) {
            if (parentAppId != null) {
                intrinsics.put("parent.app", parentAppId);
            }
            return this;
        }

        public SpanEventBuilder setParentId(String parentId) {
            if (parentId != null) {
                intrinsics.put("parentId", parentId);
            }
            return this;
        }

        public SpanEventBuilder setGuid(String guid) {
            if (guid != null) {
                intrinsics.put("guid", guid);
            }
            return this;
        }

        public SpanEventBuilder setTraceId(String traceId) {
            if (traceId != null) {
                intrinsics.put("traceId", traceId);
            }
            return this;
        }

        public SpanEventBuilder setSampled(boolean sampled) {
            intrinsics.put("sampled", sampled);
            return this;
        }

        public SpanEventBuilder setSpanTimestamp(long spanTimestamp) {
            intrinsics.put("timestamp", spanTimestamp);
            return this;
        }

        public SpanEventBuilder setParentTransportType(String parentTransportType) {
            intrinsics.put("parent.transportType", parentTransportType);
            return this;
        }

        public SpanEventBuilder setParentTransportDuration(float parentTransportDuration) {
            intrinsics.put("parent.transportDuration", parentTransportDuration);
            return this;
        }

        public SpanEventBuilder setDurationInSeconds(float duration) {
            intrinsics.put("duration", duration);
            return this;
        }

        public SpanEventBuilder setName(String name) {
            if (name != null) {
                intrinsics.put("name", name);
            }
            return this;
        }

        public SpanEventBuilder setAppName(String appName) {
            if (appName != null) {
                this.appName = appName;
            }
            return this;
        }

        public SpanEventBuilder setUserAttributes(Map<String, Object> userAttributes) {
            this.userAttributes = userAttributes;
            return this;
        }

        public SpanEventBuilder setAgentAttributes(Map<String, Object> agentAttributes) {
            this.agentAttributes = agentAttributes;
            return this;
        }

        public SpanEventBuilder setTransactionId(String rootId) {
            if (rootId != null) {
                intrinsics.put("transactionId", rootId);
            }
            return this;
        }

        public SpanEventBuilder setTimestamp(long startTime) {
            intrinsics.put("timestamp", startTime);
            return this;
        }

        public SpanEventBuilder setCategory(SpanCategory category) {
            if (category != null) {
                intrinsics.put("category", category.name());
            }
            return this;
        }

        public SpanEventBuilder setKind() {
            intrinsics.put("span.kind", userAttributes != null && userAttributes.containsKey("span.kind") ? userAttributes.get("span.kind") : SPAN_KIND);
            return this;
        }

        // http parameter
        public SpanEventBuilder setUri(URI uri) {
            if (uri != null) {
                intrinsics.put("http.url", uri.toString());
            }
            return this;
        }

        // http parameter
        public SpanEventBuilder setHttpMethod(String method) {
            if (method != null) {
                intrinsics.put("http.method", method);
            }
            return this;
        }

        // http parameter
        public SpanEventBuilder setHttpComponent(String component) {
            if (component != null) {
                intrinsics.put("component", component);
            }
            return this;
        }

        // datastore parameter
        public SpanEventBuilder setDatabaseName(String databaseName) {
            if (databaseName != null) {
                intrinsics.put("db.instance", databaseName);
            }
            return this;
        }

        // datastore parameter
        public SpanEventBuilder setDatastoreComponent(String component) {
            if (component != null) {
                intrinsics.put("component", component);
            }
            return this;
        }

        // datastore parameter
        public SpanEventBuilder setHostName(String host) {
            if (host != null) {
                intrinsics.put("peer.hostname", host);
            }
            return this;
        }

        // datastore parameter
        public SpanEventBuilder setAddress(String hostName, String portPathOrId) {
            if (portPathOrId != null && hostName != null) {
                String address = MessageFormat.format("{0}:{1}", hostName, portPathOrId);
                intrinsics.put("peer.address", address);
            }
            return this;
        }

        // datastore parameter
        public SpanEventBuilder setDatabaseStatement(String query) {
            if (query != null) {
                intrinsics.put("db.statement", truncateWithEllipsis(query, DB_STATEMENT_TRUNCATE_LENGTH));
            }
            return this;
        }

        private String truncateWithEllipsis(String value, int maxLengthWithEllipsis) {
            if (value.length() > maxLengthWithEllipsis) {
                int maxLengthWithoutEllipsis = maxLengthWithEllipsis - 3;
                return AttributeSender.truncateString(value, maxLengthWithoutEllipsis) + "...";
            }
            return value;
        }

        public SpanEventBuilder setDecider(boolean decider) {
            this.decider = decider;
            return this;
        }

        public SpanEventBuilder setIsRootSpanEvent(boolean isRoot) {
            if (isRoot) {
                intrinsics.put("nr.entryPoint", true);
            }
            return this;
        }

        public SpanEventBuilder setExternalParameterAttributes(ExternalParameters parameters) {
            if (parameters instanceof HttpParameters) {
                HttpParameters httpParameters = (HttpParameters) parameters;
                setCategory(SpanCategory.http);
                setUri(httpParameters.getUri());
                setHttpMethod(httpParameters.getProcedure());
                setHttpComponent((httpParameters).getLibrary());
                setKind();
            } else if (parameters instanceof DatastoreParameters) {
                DatastoreParameters datastoreParameters = (DatastoreParameters) parameters;
                setCategory(SpanCategory.datastore);
                setDatastoreComponent(datastoreParameters.getProduct());
                setDatabaseName(datastoreParameters.getDatabaseName());
                setHostName(datastoreParameters.getHost());
                setKind();
                if (datastoreParameters instanceof SlowQueryDatastoreParameters) {
                    SlowQueryDatastoreParameters queryDatastoreParameters = (SlowQueryDatastoreParameters) datastoreParameters;
                    setDatabaseStatement(determineObfuscationLevel(queryDatastoreParameters));
                }
                if (datastoreParameters.getPort() != null) {
                    setAddress(datastoreParameters.getHost(), String.valueOf(datastoreParameters.getPort()));
                } else {
                    setAddress(datastoreParameters.getHost(), datastoreParameters.getPathOrId());
                }
            } else {
                setCategory(SpanCategory.generic);
            }
            return this;
        }

        private String determineObfuscationLevel(SlowQueryDatastoreParameters slowQueryDatastoreParameters) {
            AgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();
            if (config.isHighSecurity() ||
                    config.getTransactionTracerConfig().getRecordSql().equals(SqlObfuscator.OFF_SETTING)) {
                return null;
            } else if (config.getTransactionTracerConfig().getRecordSql().equals(SqlObfuscator.RAW_SETTING)) {
                return slowQueryDatastoreParameters.getQueryConverter()
                        .toRawQueryString(slowQueryDatastoreParameters.getRawQuery());
            } else {
                return slowQueryDatastoreParameters.getQueryConverter()
                        .toObfuscatedQueryString(slowQueryDatastoreParameters.getRawQuery());
            }
        }

        public SpanEvent build() {
            return new SpanEvent(appName, priority, intrinsics, userAttributes, agentAttributes, decider);
        }
    }

}
