package com.newrelic.agent.service.analytics;

import com.newrelic.agent.MetricNames;
import com.newrelic.agent.TransactionData;
import com.newrelic.agent.attributes.AttributeNames;
import com.newrelic.agent.attributes.AttributeSender;
import com.newrelic.agent.attributes.AttributesUtils;
import com.newrelic.agent.config.DistributedTracingConfig;
import com.newrelic.agent.datastore.DatastoreMetrics;
import com.newrelic.agent.environment.AgentIdentity;
import com.newrelic.agent.environment.Environment;
import com.newrelic.agent.environment.EnvironmentService;
import com.newrelic.agent.errors.DeadlockTraceError;
import com.newrelic.agent.errors.TracedError;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.ServiceManager;
import com.newrelic.agent.stats.AbstractStats;
import com.newrelic.agent.stats.CountStats;
import com.newrelic.agent.stats.StatsBase;
import com.newrelic.agent.stats.TransactionStats;
import com.newrelic.agent.tracing.DistributedTraceService;
import com.newrelic.agent.util.TimeConversion;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class ErrorEvent extends AnalyticsEvent {
    static final float UNASSIGNED = Float.NEGATIVE_INFINITY;
    static final int UNASSIGNED_INT = Integer.MIN_VALUE;

    /**
     * Required.
     */
    static final String TYPE = "TransactionError";

    /**
     * Required.
     */
    final String errorClass;

    /**
     * Required.
     */
    final String errorMessage;

    /**
     * Required.
     */
    boolean errorExpected = false;

    /**
     * Required. Full metric name of the Transaction, or "Unknown" if not in a transaction.
     */
    String transactionName = "Unknown";

    /**
     * Optional, requires Transaction. Duration of this transaction. Does not include queue time.
     */
    float duration = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'WebFrontend/QueueTime' metric.
     */
    float queueDuration = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'External/all' metric.
     */
    float externalDuration = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'Datastore/all' metric.
     */
    float databaseDuration = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'GC/cumulative' metric. Time spent in garbage collection across all
     * transactions during the timespan of this transaction.
     */
    float gcCumulative = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'Datastore/all' call count.
     */
    float databaseCallCount = UNASSIGNED;

    /**
     * Optional, requires Transaction. Equivalent to 'External/all' call count.
     */
    float externalCallCount = UNASSIGNED;

    /**
     * Optional, requires Transaction
     */
    String transactionGuid;

    /**
     * Optional, requires Transaction
     */
    String referringTransactionGuid;

    /**
     * Optional, requires Transaction
     */
    String syntheticsResourceId;

    /**
     * Optional, requires Transaction
     */
    String syntheticsMonitorId;

    /**
     * Optional, requires Transaction
     */
    String syntheticsJobId;

    /**
     * Optional.
     */
    int port = UNASSIGNED_INT;

    /**
     * Optional.
     */
    String timeoutCause;

    /**
     * Better CAT trip ID
     */
    String tripId;

    /**
     * Event priority
     */
    float priority = UNASSIGNED;

    /**
     * Better CAT intrinsics
     */
    Map<String, Object> distributedTraceIntrinsics;

    /**
     * Optional. Agent custom parameters. This includes request parameters.
     */
    Map<String, Object> agentAttributes;

    /**
     * Required.
     */
    String appName;

    public ErrorEvent(String appName, TracedError tracedError) {
        super(TYPE, tracedError.getTimestampInMillis());
        this.appName = appName;
        this.errorClass = tracedError.getExceptionClass();
        this.errorMessage = truncateIfNecessary(tracedError.getMessage());
        assignPortUsingServiceManagerIfPossible();
        if (!tracedError.incrementsErrorMetric() && !(tracedError instanceof DeadlockTraceError)) {
            this.errorExpected = true;
        }
        distributedTraceIntrinsics = Collections.emptyMap();
        tripId = null;
        this.userAttributes = new HashMap<String, Object>(tracedError.getErrorAtts());
    }

    private void assignPortUsingServiceManagerIfPossible() {
        // Who needs DI?
        ServiceManager serviceManager = ServiceFactory.getServiceManager();
        if (serviceManager != null) {
            EnvironmentService environmentService = serviceManager.getEnvironmentService();
            if (environmentService != null) {
                Environment environment = environmentService.getEnvironment();
                if (environment != null) {
                    AgentIdentity agentIdentity = environment.getAgentIdentity();
                    if (agentIdentity != null) {
                        Integer serverPort = agentIdentity.getServerPort();
                        if (serverPort != null) {
                            this.port = serverPort;
                        }
                    }
                }
            }
        }
    }

    public ErrorEvent(String appName, TracedError tracedError, TransactionData transactionData,
            TransactionStats transactionStats) {
        super(TYPE, tracedError.getTimestampInMillis(), transactionData.getPriority());
        this.appName = appName;
        this.errorClass = tracedError.getExceptionClass();
        this.errorMessage = truncateIfNecessary(tracedError.getMessage());
        this.transactionName = transactionData.getPriorityTransactionName().getName();
        this.duration = (float) transactionData.getDurationInMillis() / TimeConversion.MILLISECONDS_PER_SECOND;
        this.queueDuration = retrieveMetricIfExists(transactionStats, MetricNames.QUEUE_TIME).getTotal();
        this.externalDuration = retrieveMetricIfExists(transactionStats, MetricNames.EXTERNAL_ALL).getTotal();
        this.databaseDuration = retrieveMetricIfExists(transactionStats, DatastoreMetrics.ALL).getTotal();
        this.gcCumulative = retrieveMetricIfExists(transactionStats, MetricNames.GC_CUMULATIVE).getTotal();
        this.databaseCallCount = retrieveMetricIfExists(transactionStats, DatastoreMetrics.ALL).getCallCount();
        this.externalCallCount = retrieveMetricIfExists(transactionStats, MetricNames.EXTERNAL_ALL).getCallCount();
        this.transactionGuid = transactionData.getGuid();
        this.referringTransactionGuid = transactionData.getReferrerGuid();
        this.syntheticsResourceId = transactionData.getSyntheticsResourceId();
        this.syntheticsJobId = transactionData.getSyntheticsJobId();
        this.syntheticsMonitorId = transactionData.getSyntheticsMonitorId();
        this.timeoutCause = transactionData.getTimeoutCause() == null ? null : transactionData.getTimeoutCause().cause;
        if (!tracedError.incrementsErrorMetric() && !(tracedError instanceof DeadlockTraceError)) {
            this.errorExpected = true;
        }

        assignPortUsingServiceManagerIfPossible();

        if (ServiceFactory.getAttributesService().isAttributesEnabledForErrors(appName)) {
            // trans events take user and agent atts - any desired intrinsics should have already been grabbed
            this.userAttributes = transactionData.getUserAttributes();
            this.userAttributes.putAll(transactionData.getErrorAttributes());
            this.agentAttributes = transactionData.getAgentAttributes();
            // request/message parameters are sent up in the same bucket as agent attributes
            this.agentAttributes.putAll(AttributesUtils.appendAttributePrefixes(transactionData.getPrefixedAttributes()));
        }

        this.priority = transactionData.getPriority();

        DistributedTracingConfig distributedTracingConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig();
        if (distributedTracingConfig.isEnabled()) {
            // Better CAT
            DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();
            this.distributedTraceIntrinsics = distributedTraceService.getIntrinsics(
                    transactionData.getInboundDistributedTracePayload(), transactionData.getGuid(),
                    transactionData.getTripId(), transactionData.getTransportType(),
                    transactionData.getTransportDurationInMillis(),
                    transactionData.getLargestTransportDurationInMillis(),
                    transactionData.getParentId(), transactionData.getParentSpanId(),
                    transactionData.getPriority());
            this.tripId = transactionData.getTripId();
        }
    }

    public String getErrorClass() {
        return errorClass;
    }

    public String getErrorMessage() {
        return errorMessage;
    }

    public String getTransactionName() {
        return transactionName;
    }

    public Map<String, Object> getDistributedTraceIntrinsics() {
        return this.distributedTraceIntrinsics;
    }

    public String getTransactionGuid() {
        return transactionGuid;
    }

    /*
     * The data should go up as 3 hashes. Example: [ { "databaseDuration":value, "timestamp":value, "name":"value",
     * "duration":value, "type":"value" }, { "user_param1":"value", "user_param2":value, }, { "agent_param1": "value",
     * "agent_param2": value }
     */
    @SuppressWarnings("unchecked")
    @Override
    public void writeJSONString(Writer out) throws IOException {
        JSONObject obj = new JSONObject();
        obj.put("type", type);
        obj.put("error.class", errorClass);
        obj.put("error.message", errorMessage);
        obj.put("timestamp", timestamp);
        obj.put("transactionName", transactionName);
        obj.put(AttributeNames.ERROR_EXPECTED, errorExpected);

        if (duration != UNASSIGNED) {
            obj.put("duration", duration);
        }
        if (queueDuration != UNASSIGNED) {
            obj.put("queueDuration", queueDuration);
        }
        if (externalDuration != UNASSIGNED) {
            obj.put("externalDuration", externalDuration);
        }
        if (databaseDuration > 0) {
            obj.put("databaseDuration", databaseDuration);
        }
        if (gcCumulative != UNASSIGNED) {
            obj.put("gcCumulative", gcCumulative);
        }
        if (databaseCallCount > 0) {
            obj.put("databaseCallCount", databaseCallCount);
        }
        if (externalCallCount > 0) {
            obj.put("externalCallCount", externalCallCount);
        }
        if (transactionGuid != null) {
            obj.put("nr.transactionGuid", transactionGuid); // prefixed to be hidden by Insights
        }
        if (referringTransactionGuid != null) {
            obj.put("nr.referringTransactionGuid", referringTransactionGuid); // prefixed to be hidden by Insights
        }
        if (this.syntheticsResourceId != null) {
            obj.put("nr.syntheticsResourceId", this.syntheticsResourceId);
        }
        if (this.syntheticsMonitorId != null) {
            obj.put("nr.syntheticsMonitorId", this.syntheticsMonitorId);
        }
        if (this.syntheticsJobId != null) {
            obj.put("nr.syntheticsJobId", this.syntheticsJobId);
        }
        if (port != UNASSIGNED_INT) {
            obj.put("port", port);
        }
        if (timeoutCause != null) {
            obj.put(AttributeNames.TIMEOUT_CAUSE, timeoutCause);
        }
        if (distributedTraceIntrinsics != null && !distributedTraceIntrinsics.isEmpty()) {
            obj.putAll(distributedTraceIntrinsics);
        }
        if (tripId != null) {
            obj.put("nr.tripId", tripId); // prefixed to be hidden by Insights
        }
        if (priority != UNASSIGNED) {
            obj.put("priority", priority);
        }

        Map<String, ? extends Object> filteredUserAtts = getUserFilteredMap(userAttributes);
        Map<String, ? extends Object> filteredAgentAtts = getFilteredMap(agentAttributes);
        if (filteredAgentAtts.isEmpty()) {
            if (filteredUserAtts.isEmpty()) {
                JSONArray.writeJSONString(Arrays.asList(obj), out);
            } else {
                JSONArray.writeJSONString(Arrays.asList(obj, filteredUserAtts), out);
            }
        } else {
            JSONArray.writeJSONString(Arrays.asList(obj, filteredUserAtts, filteredAgentAtts), out);
        }
    }

    private Map<String, ? extends Object> getFilteredMap(Map<String, Object> input) {
        return ServiceFactory.getAttributesService().filterErrorAttributes(appName, input);
    }

    private Map<String, ? extends Object> getUserFilteredMap(Map<String, Object> input) {
        // user attributes should have already been filtered for high security - this is just extra protection
        // high security is per an account - meaning it can not be different for various application names within a
        // JVM - so we can just check the default agent config
        if (!ServiceFactory.getConfigService().getDefaultAgentConfig().isHighSecurity()) {
            return getFilteredMap(input);
        } else {
            return Collections.emptyMap();
        }
    }

    @Override
    public boolean isValid() {
        // We don't need to validate the type "Error" every time.
        return true;
    }

    private CountStats retrieveMetricIfExists(TransactionStats transactionStats, String metricName) {
        if (!transactionStats.getUnscopedStats().getStatsMap().containsKey(metricName)) {
            return NoCallCountStats.NO_STATS;
        }
        return transactionStats.getUnscopedStats().getResponseTimeStats(metricName);
    }

    private String truncateIfNecessary(String value) {
        int maxUserParameterSize = ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameterSize();
        try {
            if (value.getBytes("UTF-8").length > maxUserParameterSize) {
                return AttributeSender.truncateString(value, maxUserParameterSize);
            }
        } catch (UnsupportedEncodingException e) {
        }
        return value;
    }

    private static class NoCallCountStats extends AbstractStats {
        static final NoCallCountStats NO_STATS = new NoCallCountStats();

        @Override
        public float getTotal() {
            return ErrorEvent.UNASSIGNED;
        }

        @Override
        public float getTotalExclusiveTime() {
            return ErrorEvent.UNASSIGNED;
        }

        @Override
        public float getMinCallTime() {
            return ErrorEvent.UNASSIGNED;
        }

        @Override
        public float getMaxCallTime() {
            return ErrorEvent.UNASSIGNED;
        }

        @Override
        public double getSumOfSquares() {
            return ErrorEvent.UNASSIGNED;
        }

        @Override
        public boolean hasData() {
            return false;
        }

        @Override
        public void reset() {
        }

        @Override
        public void merge(StatsBase stats) {
        }

        @Override
        public Object clone() throws CloneNotSupportedException {
            return NO_STATS;
        }
    }
}