package com.newrelic.agent.service.analytics;

import com.google.common.collect.Maps;
import com.newrelic.agent.Agent;
import org.json.simple.JSONStreamAware;

import java.io.IOException;
import java.io.Writer;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Map;
import java.util.Random;
import java.util.logging.Level;
import java.util.regex.Pattern;

public abstract class AnalyticsEvent implements JSONStreamAware, PriorityAware {
    private static final Pattern TYPE_VALID = Pattern.compile("^[a-zA-Z0-9:_ ]{1,255}$");
    private static final Random RANDOM = new Random();

    // Instantiate a new DecimalFormat instance as it is not thread safe:
    // http://jonamiller.com/2015/12/21/decimalformat-is-not-thread-safe/
    private static final ThreadLocal<DecimalFormat> FORMATTER =
            new ThreadLocal<DecimalFormat>() {
                @Override
                protected DecimalFormat initialValue() {
                    DecimalFormat format = new DecimalFormat();
                    DecimalFormatSymbols symbols = format.getDecimalFormatSymbols();
                    if (symbols.getDecimalSeparator() == ',') {
                        format.applyLocalizedPattern("#,######");
                    } else {
                        format.applyPattern("#.######");
                    }
                    return format;
                }
            };

    final float priority;

    /**
     * Required. Must match /^[a-zA-Z0-9:_ ]+$/ and be less than 256 chars.<br>
     * Validate and discard invalid events before reporting.
     */
    final String type;

    /**
     * Required. Start time of the transaction (UTC timestamp)
     */
    final long timestamp;

    /**
     * Optional. User custom parameters.
     */
    Map<String, Object> userAttributes;

    public AnalyticsEvent(String type, long timestamp) {
        this(type, timestamp, nextTruncatedFloat());
    }

    public AnalyticsEvent(String type, long timestamp, float priority) {
        super();
        this.type = type;
        this.timestamp = timestamp;
        this.priority = priority;
    }

    // implementation must match com.newrelic.agent.tracing.DistributedTraceServiceImpl
    static float nextTruncatedFloat() {
        float next = 0.0f;
        try {
            next = Float.parseFloat(FORMATTER.get().format(RANDOM.nextFloat()).replace(',', '.'));
        } catch (NumberFormatException e) {
            Agent.LOG.log(Level.WARNING, "Unable to create priority value", e);
        }
        return next;
    }

    @Override
    public abstract void writeJSONString(Writer out) throws IOException;

    public static boolean isValidType(String type) {
        return type != null && TYPE_VALID.matcher(type).matches();
    }

    public boolean isValid() {
        return isValidType(type);
    }

    public String getType() {
        return type;
    }

    @Override
    public boolean decider() {
        return false;
    }

    @Override
    public float getPriority() {
        return priority;
    }

    public Map<String, Object> getAttributesCopy() {
        if (userAttributes == null) {
            return null;
        }
        return Maps.newHashMap(userAttributes);
    }
}
