package com.newrelic.agent.reinstrument;

import java.lang.instrument.UnmodifiableClassException;
import java.text.MessageFormat;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.collect.Sets;
import com.newrelic.agent.Agent;
import com.newrelic.agent.PrivateApiImpl;
import com.newrelic.agent.service.ServiceFactory;

/**
 * Uses the SamplerService thread to periodically request retransform for classes added to the queue.
 * 
 * @see PrivateApiImpl(java.lang.reflect.Method, String)
 * @author tyler
 */
public class PeriodicRetransformer implements Retransformer, Runnable {
    private static final int FREQUENCY_IN_SECONDS = 10;

    private final AtomicReference<ConcurrentLinkedQueue<Class<?>>> classesToRetransform =
            new AtomicReference<ConcurrentLinkedQueue<Class<?>>>(new ConcurrentLinkedQueue<Class<?>>());

    private final AtomicBoolean scheduled = new AtomicBoolean(false);

    public static final Retransformer INSTANCE = new PeriodicRetransformer();

    private PeriodicRetransformer() {

    }

    @Override
    public void run() {
        ConcurrentLinkedQueue<Class<?>> classList = classesToRetransform.getAndSet(new ConcurrentLinkedQueue<Class<?>>());
        if (classList.isEmpty()) {
            return;
        }
        Set<Class<?>> classSet = Sets.newHashSet(classList);
        try {
            ServiceFactory.getAgent().getInstrumentation().retransformClasses(classSet.toArray(new Class[] {}));
        } catch (UnmodifiableClassException e) {
            Agent.LOG.fine(MessageFormat.format("Unable to retransform class: {0}", e.getMessage()));
        }
    }

    @Override
    public void queueRetransform(Set<Class<?>> classesToRetransform) {
        this.classesToRetransform.get().addAll(classesToRetransform);
        if (!scheduled.get() && !scheduled.getAndSet(true)) {
            ServiceFactory.getSamplerService().addSampler(this, FREQUENCY_IN_SECONDS, TimeUnit.SECONDS);
        }
    }
}