package com.newrelic.agent.logging;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import com.google.common.io.Resources;

import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentConfigImpl;
import com.newrelic.agent.config.AgentJarHelper;

import ch.qos.logback.classic.Level;

class LogbackLogManager implements IAgentLogManager {
    private final static String AGENT_JAR_LOGBACK_CONFIG_FILE = "/META-INF/logging/logback.xml";

    /**
     * This needs to get set to keep errors from occurring.
     */
    private static final String CONFIG_FILE_PROP = "logback.configurationFile";
    private static final String CONTEXT_SELECT_PROP = "logback.ContextSelector";
    private static final String STATUS_LIST_PROP = "logback.statusListenerClass";

    private final LogbackLogger rootLogger;
    private volatile String logFilePath;

    private LogbackLogManager(String name) {
        rootLogger = initializeRootLogger(name);
    }

    private LogbackLogger createRootLogger(String name) {
        LogbackLogger logger = LogbackLogger.create(name, true);
        String logLevel = getStartupLogLevel();
        logger.setLevel(logLevel);
        logger.addConsoleAppender();
        return logger;
    }

    private String getStartupLogLevel() {
        String propName = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.STARTUP_LOG_LEVEL;
        String logLevel = System.getProperty(propName);
        if (logLevel == null) {
            return Level.INFO.levelStr.toLowerCase();
        } else {
            return logLevel.toLowerCase();
        }
    }

    private LogbackLogger initializeRootLogger(String name) {
        LogbackLogger logger = null;
        Map<String, String> systemProps = new HashMap<String, String>();

        try {
            String jarFileName = AgentJarHelper.getAgentJarFileName();
            if (jarFileName == null) {
                logger = LogbackLogger.create(name, true);
            } else {
                // If they already set the config location, we want to clear it before configuring our logger.
                // and we want to clear any other logback system properties
                clearAllLogbackSystemProperties(systemProps);

                URL logbackConfigXmlUrl = null;
                if (jarFileName.endsWith(".jar")) {
                    // it isn't enough to specify the jar, we have to specify the path within the jar
                    logbackConfigXmlUrl = new URL(new StringBuilder("jar:file:")
                            .append(jarFileName)
                            .append("!")
                            .append(AGENT_JAR_LOGBACK_CONFIG_FILE)
                            .toString());
                } else {
                    // we likely have received a path to a set of class files (this happens when running tests)
                    try {
                        // guava's Resources class is usually smart enough to figure out where to find the logback.xml file
                        logbackConfigXmlUrl = Resources.getResource(this.getClass(), AGENT_JAR_LOGBACK_CONFIG_FILE);
                    } catch (IllegalArgumentException iae) {
                        // fallback on path
                        logbackConfigXmlUrl = new File(jarFileName).toURI().toURL();
                    }
                }

                System.setProperty(CONFIG_FILE_PROP, logbackConfigXmlUrl.toString());
                try {
                    logger = createRootLogger(name);
                } finally {
                    // Our logger is configured, restore back to original setting.
                    // be sure to remove our property
                    System.getProperties().remove(CONFIG_FILE_PROP);
                    applyOriginalSystemProperties(systemProps, logger);
                }
            }
        } catch (Exception e) {
            if (logger == null) {
                logger = createRootLogger(name);
            }
            String msg = MessageFormat.format("Error setting logback.configurationFile property: {0}", e);
            logger.warning(msg);
        }
        return logger;
    }

    private void clearAllLogbackSystemProperties(Map<String, String> storedSystemProps) {
        clearLogbackSystemProperty(CONFIG_FILE_PROP, storedSystemProps);
        clearLogbackSystemProperty(CONTEXT_SELECT_PROP, storedSystemProps);
        clearLogbackSystemProperty(STATUS_LIST_PROP, storedSystemProps);
    }

    private void clearLogbackSystemProperty(String prop, Map<String, String> storedSystemProps) {
        String old = System.clearProperty(prop);
        if (old != null) {
            storedSystemProps.put(prop, old);
        }
    }

    private void applyOriginalSystemProperties(Map<String, String> storedSystemProps, LogbackLogger logger) {
        for (Entry<String, String> currentProp : storedSystemProps.entrySet()) {
            try {
                System.setProperty(currentProp.getKey(), currentProp.getValue());
            } catch (Exception e) {
                String msg = MessageFormat.format("Error setting logback property {0} back to {1}. Error: {2}",
                        currentProp.getKey(), currentProp.getValue(), e);
                logger.warning(msg);
            }
        }
    }

    @Override
    public IAgentLogger getRootLogger() {
        return rootLogger;
    }

    @Override
    public String getLogFilePath() {
        return logFilePath;
    }

    @Override
    public void configureLogger(AgentConfig pAgentConfig) {
        configureLogLevel(pAgentConfig);
        configureConsoleHandler(pAgentConfig);
        configureFileHandler(pAgentConfig);
    }

    private void configureFileHandler(AgentConfig agentConfig) {
        String logFileName = getLogFileName(agentConfig);
        if (logFileName == null) {
            return;
        }
        try {
            configureFileHandler(logFileName, agentConfig);
            logFilePath = logFileName;
            String msg = MessageFormat.format("Writing to New Relic log file: {0}", logFileName);
            rootLogger.info(msg);
            rootLogger.info(MessageFormat.format("JRE vendor {0} version {1}", System.getProperty("java.vendor"),
                    System.getProperty("java.version")));
            rootLogger.info(MessageFormat.format("JVM vendor {0} {1} version {2}",
                    System.getProperty("java.vm.vendor"), System.getProperty("java.vm.name"),
                    System.getProperty("java.vm.version")));
            rootLogger.fine(MessageFormat.format("JVM runtime version {0}", System.getProperty("java.runtime.version")));
            rootLogger.info(MessageFormat.format("OS {0} version {1} arch {2}", System.getProperty("os.name"),
                    System.getProperty("os.version"), System.getProperty("os.arch")));
        } catch (IOException e) {
            String msg = MessageFormat.format("Unable to configure newrelic log file: {0}", logFileName);
            rootLogger.error(msg);
            addConsoleHandler();
        }
    }

    private String getLogFileName(AgentConfig agentConfig) {
        File logFile = LogFileHelper.getLogFile(agentConfig);
        return logFile == null ? null : logFile.getPath();
    }

    private void configureLogLevel(AgentConfig agentConfig) {

        if (agentConfig.isDebugEnabled()) {
            rootLogger.setLevel(Level.TRACE.levelStr.toLowerCase());
        } else {
            rootLogger.setLevel(agentConfig.getLogLevel());
        }

    }

    private void configureConsoleHandler(AgentConfig agentConfig) {
        if (agentConfig.isDebugEnabled() || agentConfig.isLoggingToStdOut()) {
            addConsoleHandler();
        } else {
            rootLogger.removeConsoleAppender();
        }
    }

    private String configureFileHandler(String logFileName, AgentConfig agentConfig) throws IOException {
        rootLogger.addConsoleAppender();
        // write logging success or failure to stdout
        if (canWriteLogFile(logFileName)) {
            rootLogger.info(MessageFormat.format("New Relic Agent: Writing to log file: {0}", logFileName));
        } else {
            rootLogger.warning(MessageFormat.format(
                    "New Relic Agent: Unable to write log file: {0}. Please check permissions on the file and directory.",
                    logFileName));
        }
        rootLogger.removeConsoleAppender();

        int limit = agentConfig.getLogLimit() * 1024;
        int fileCount = Math.max(1, agentConfig.getLogFileCount());
        boolean isDaily = agentConfig.isLogDaily();
        rootLogger.addFileAppender(logFileName, limit, fileCount, isDaily);
        return logFileName;
    }

    private boolean canWriteLogFile(String logFileName) {
        try {
            File logFile = new File(logFileName);
            if (!logFile.exists()) {
                if (null != logFile.getParentFile()) {
                    logFile.getParentFile().mkdirs();
                }
                logFile.createNewFile();
            }
            return logFile.canWrite();
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public void addConsoleHandler() {
        rootLogger.addConsoleAppender();

    }

    @Override
    public void setLogLevel(String pLevel) {
        rootLogger.setLevel(pLevel);

    }

    @Override
    public String getLogLevel() {
        return rootLogger.getLevel();
    }

    public static LogbackLogManager create(String name) {
        return new LogbackLogManager(name);
    }

}
