package com.newrelic.agent.jmx;

import java.text.MessageFormat;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import javax.management.Attribute;
import javax.management.ObjectInstance;

import com.ibm.websphere.management.statistics.BoundaryStatistic;
import com.ibm.websphere.management.statistics.CountStatistic;
import com.ibm.websphere.management.statistics.JCAConnectionPoolStats;
import com.ibm.websphere.management.statistics.JCAConnectionStats;
import com.ibm.websphere.management.statistics.JCAStats;
import com.ibm.websphere.management.statistics.JDBCConnectionPoolStats;
import com.ibm.websphere.management.statistics.JDBCConnectionStats;
import com.ibm.websphere.management.statistics.JDBCStats;
import com.ibm.websphere.management.statistics.JMSConnectionStats;
import com.ibm.websphere.management.statistics.JMSSessionStats;
import com.ibm.websphere.management.statistics.JMSStats;
import com.ibm.websphere.management.statistics.RangeStatistic;
import com.ibm.websphere.management.statistics.Statistic;
import com.ibm.websphere.management.statistics.Stats;
import com.ibm.websphere.management.statistics.TimeStatistic;
import com.newrelic.agent.Agent;
import com.newrelic.agent.stats.StatsEngine;

@SuppressWarnings("deprecation")
public class WebSphereStatsAttributeProcessor extends AbstractStatsAttributeProcessor {

    // Called by the JMXService through reflection.
    public WebSphereStatsAttributeProcessor() {
    }

    @Override
    public boolean process(StatsEngine statsEngine, ObjectInstance instance, Attribute attribute, String metricName,
            Map<String, Float> values) {

        Object value = attribute.getValue();
        if (value instanceof Stats) {
            boolean isBuiltInMetric = isBuiltInMetric(metricName);
            if (value instanceof JDBCStats) {
                pullJDBCStats(statsEngine, (JDBCStats) value, attribute, metricName, values, isBuiltInMetric);
            } else if (value instanceof JCAStats) {
                pullJCAStats(statsEngine, (JCAStats) value, attribute, metricName, values, isBuiltInMetric);
            } else if (value instanceof JMSStats) {
                pullJMSStats(statsEngine, (JMSStats) value, attribute, metricName, values, isBuiltInMetric);
            } else {
                Stats jmxStats = (Stats) value;

                for (Statistic statistic : jmxStats.getStatistics()) {
                    if (isBuiltInMetric) {
                        addJmxValue(attribute, statistic, values);
                    } else {
                        processStatistic(statsEngine, metricName, statistic);
                    }
                }
            }
            return true;
        }
        return false;
    }

    private static void pullJMSStats(StatsEngine statsEngine, JMSStats jmsStats, Attribute attribute,
            String metricName, Map<String, Float> values, boolean isBuiltInMetric) {
        for (JMSConnectionStats connStats : jmsStats.getConnections()) {
            for (JMSSessionStats current : connStats.getSessions()) {
                grabBaseStats(statsEngine, current, attribute, metricName, values, isBuiltInMetric);
            }
        }
    }

    private static void pullJDBCStats(StatsEngine statsEngine, JDBCStats jdbcStats, Attribute attribute,
            String metricName, Map<String, Float> values, boolean isBuiltInMetric) {

        if (jdbcStats.getConnectionPools() != null) {
            for (JDBCConnectionPoolStats current : jdbcStats.getConnectionPools()) {
                grabBaseStats(statsEngine, current, attribute, metricName, values, isBuiltInMetric);
            }
        }
        if (jdbcStats.getConnections() != null) {
            for (JDBCConnectionStats current : jdbcStats.getConnections()) {
                grabBaseStats(statsEngine, current, attribute, metricName, values, isBuiltInMetric);
            }
        }
    }

    private static void pullJCAStats(StatsEngine statsEngine, JCAStats jcaStats, Attribute attribute,
            String metricName, Map<String, Float> values, boolean isBuiltInMetric) {

        if (jcaStats.getConnectionPools() != null) {
            for (JCAConnectionPoolStats current : jcaStats.getConnectionPools()) {
                grabBaseStats(statsEngine, current, attribute, metricName, values, isBuiltInMetric);
            }
        }
        if (jcaStats.getConnections() != null) {
            for (JCAConnectionStats current : jcaStats.getConnections()) {
                grabBaseStats(statsEngine, current, attribute, metricName, values, isBuiltInMetric);
            }
        }
    }

    private static void grabBaseStats(StatsEngine statsEngine, Stats jmxStats, Attribute attribute, String metricName,
            Map<String, Float> values, boolean isBuiltInMetric) {
        for (Statistic statistic : jmxStats.getStatistics()) {
            if (isBuiltInMetric) {
                // this will only pull the desired metric from the stats object
                if (addJmxValue(attribute, statistic, values)) {
                    break;
                }
            } else {
                // this will pull all metrics from the stats object
                processStatistic(statsEngine, metricName, statistic);
            }
        }
    }

    static void processStatistic(StatsEngine statsEngine, String metricName, Statistic statistic) {
        String fullMetricName = metricName + '/' + statistic.getName();
        if (statistic instanceof CountStatistic) {
            CountStatistic stat = (CountStatistic) statistic;
            statsEngine.getStats(fullMetricName).recordDataPoint(stat.getCount());
        } else if (statistic instanceof RangeStatistic) {
            RangeStatistic stat = (RangeStatistic) statistic;
            statsEngine.getStats(fullMetricName).recordDataPoint(stat.getCurrent());
        } else if (statistic instanceof BoundaryStatistic) {
            BoundaryStatistic stat = (BoundaryStatistic) statistic;
            statsEngine.getStats(fullMetricName).recordDataPoint(stat.getLowerBound());
            statsEngine.getStats(fullMetricName).recordDataPoint(stat.getUpperBound());
        } else if (statistic instanceof TimeStatistic) {
            TimeStatistic stat = (TimeStatistic) statistic;
            TimeUnit unit = getTimeUnit(stat.getUnit());
            statsEngine.getResponseTimeStats(fullMetricName).recordResponseTime((int) stat.getCount(),
                    stat.getTotalTime(), stat.getMinTime(), stat.getMaxTime(), unit);
        }
    }

    static boolean addJmxValue(Attribute attribute, Statistic statistic, Map<String, Float> values) {
        if (attribute.getName().contains(statistic.getName())) {
            if (statistic instanceof CountStatistic) {
                CountStatistic stat = (CountStatistic) statistic;
                values.put(attribute.getName(), Float.valueOf(stat.getCount()));
                return true;
            } else if (statistic instanceof RangeStatistic) {
                RangeStatistic stat = (RangeStatistic) statistic;
                values.put(attribute.getName(), Float.valueOf(stat.getCurrent()));
                return true;
            } else if (statistic instanceof BoundaryStatistic) {
                BoundaryStatistic stat = (BoundaryStatistic) statistic;
                values.put(attribute.getName(), Float.valueOf((stat.getLowerBound() + stat.getUpperBound()) / 2));
                return true;
            } else if (statistic instanceof TimeStatistic) {
                TimeStatistic stat = (TimeStatistic) statistic;
                values.put(attribute.getName(), Float.valueOf(stat.getTotalTime() / stat.getCount()));
                return true;
            }
        } else {
            Agent.LOG.log(Level.FINEST, MessageFormat.format(
                    "Not recording stat {0} because it does not match the attribute name {1}.", statistic.getName(),
                    attribute.getName()));
        }
        return false;
    }

}
