package com.newrelic.agent.instrumentation.pointcuts.play;

import java.util.ArrayDeque;

import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionActivity;
import com.newrelic.agent.TransactionStateImpl;
import com.newrelic.agent.instrumentation.pointcuts.play.PlayDispatcherPointCut.PlayHttpRequest;
import com.newrelic.agent.tracers.ClassMethodSignature;
import com.newrelic.agent.tracers.MethodExitTracerNoSkip;
import com.newrelic.agent.tracers.Tracer;
import com.newrelic.agent.tracers.TracerFactory;

/**
 * This class handles the special requirements of asynchronous processing in Play.
 * 
 * Play uses the Apache Commons Javaflow project to store the state of the stack in a Continuation object which can be
 * suspended and resumed. When Play processes a web request, a continuation object is started. Asynchronous programming
 * is typically done by creating a Future object and passing it to the Controller await method. The await method
 * suspends the Continuation. This saves the state of the stack and causes execution to continue from the point the
 * Continuation was started. The Continuation is then stored in the web request, and the thread is freed to process
 * another request. When the Future is ready, the web request is placed back on the execution queue and is processed by
 * a possibly different thread. Play notices the request has a continuation, and resumes it. This restores the state of
 * the stack, and processing continues at the point the continuation was suspended.
 * 
 * When the continuation is suspended each tracer on the stack finishes. This is because Agent instrumentation adds a
 * try/finally block to the method after Javaflow has instrumented it. Moreover, when the thread is freed up for another
 * web request, the root tracer finishes which causes the transaction to finish. So when the continuation is resumed, a
 * new transaction is started. This is bad because a single web request ends up spanning multiple transactions, and the
 * time spent waiting for asynchronous processing is not recorded.
 * 
 * This class is responsible for preventing tracers from finishing when the Continuation is suspended, and to return the
 * suspended tracers when the Continuation is resumed. It is also saves the transaction in the Play request and clears
 * the current transaction to free the thread for another request.
 * 
 * This class also supports the previous mechanism of throwing a suspend exception in the Controller await method. The
 * exception is caught in the ActionInvoker class and the web request is later retried by invoking a callback method. In
 * this scenario, only the root tracer is prevented from finishing.
 * 
 * @{see PlayDispatcherPointCut}
 * @{see PlayContinuationPointCut}
 * @{see PlayControllerPointCut}
 * 
 */
public class PlayTransactionStateImpl extends TransactionStateImpl {

    private static final TransactionActivity NULL_TRANSACTION_ACTIVITY = null;

    private static final Tracer NULL_TRACER = new MethodExitTracerNoSkip(null, NULL_TRANSACTION_ACTIVITY) {
        @Override
        protected void doFinish(int opcode, Object returnValue) {
        }
    };

    private final PlayHttpRequest request;
    private State state = State.RUNNING;
    private final ArrayDeque<Tracer> tracers = new ArrayDeque<Tracer>();
    private ArrayDeque<Tracer> suspendedTracers = new ArrayDeque<Tracer>();

    public PlayTransactionStateImpl(PlayHttpRequest request) {
        this.request = request;
    }

    @Override
    public Tracer getTracer(Transaction tx, TracerFactory tracerFactory, ClassMethodSignature signature, Object object,
            Object... args) {
        if (state == State.RESUMING) {
            Tracer tracer = removeSuspendedTracer();
            if (tracer == NULL_TRACER) {
                return null;
            }
            if (tracer != null) {
                return tracer;
            }
            state = State.RUNNING;
        }
        Tracer tracer = super.getTracer(tx, tracerFactory, signature, object, args);
        addTracer(tracer == null ? NULL_TRACER : tracer);
        return tracer;
    }

    @Override
    public Tracer getTracer(Transaction tx, final Object invocationTarget, final ClassMethodSignature sig,
            final String metricName, final int flags) {
        if (state == State.RESUMING) {
            Tracer tracer = removeSuspendedTracer();
            if (tracer == NULL_TRACER) {
                return null;
            }
            if (tracer != null) {
                return tracer;
            }
            state = State.RUNNING;
        }
        Tracer tracer = super.getTracer(tx, invocationTarget, sig, metricName, flags);
        addTracer(tracer == null ? NULL_TRACER : tracer);
        return tracer;
    }

    private Tracer removeSuspendedTracer() {
        return suspendedTracers.pollFirst();
    }

    private void removeTracer(Tracer tracer) {
        Tracer lastTracer = tracers.peekLast();
        while (lastTracer == NULL_TRACER) {
            tracers.pollLast();
            lastTracer = tracers.peekLast();
        }
        if (lastTracer == tracer) {
            tracers.pollLast();
        }
    }

    private void addTracer(Tracer tracer) {
        tracers.addLast(tracer);
    }

    @Override
    public void resume() {
        state = State.RESUMING;
    }

    @Override
    public void suspend() {
        state = State.SUSPENDING;
    }

    @Override
    public void suspendRootTracer() {
        state = State.SUSPENDING_ROOT_TRACER;
    }

    @Override
    public boolean finish(Transaction tx, Tracer tracer) {
        if (state == State.SUSPENDING) {
            if (tracer == tx.getRootTracer()) {
                saveTransaction(tx);
            }
            return false;
        }
        if (state == State.SUSPENDING_ROOT_TRACER) {
            if (tracer == tx.getRootTracer()) {
                saveTransaction(tx);
                return false;
            }
        }
        if (state == State.RESUMING) {
            suspendedTracers.clear();
            state = State.RUNNING;
        }
        removeTracer(tracer);
        return true;
    }

    private void saveTransaction(Transaction tx) {
        suspendedTracers = new ArrayDeque<Tracer>(tracers);
        request._nr_setTransaction(tx);
        Transaction.clearTransaction();
    }

    private enum State {

        RESUMING, RUNNING, SUSPENDING, SUSPENDING_ROOT_TRACER
    }

}
