package com.newrelic.agent.instrumentation.pointcuts.play;

import java.text.MessageFormat;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Map;
import java.util.logging.Level;

import com.newrelic.agent.Agent;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionState;
import com.newrelic.agent.bridge.TransactionNamePriority;
import com.newrelic.agent.instrumentation.PointCutClassTransformer;
import com.newrelic.agent.instrumentation.PointCutConfiguration;
import com.newrelic.agent.instrumentation.TracerFactoryPointCut;
import com.newrelic.agent.instrumentation.classmatchers.ClassMatcher;
import com.newrelic.agent.instrumentation.classmatchers.ExactClassMatcher;
import com.newrelic.agent.instrumentation.methodmatchers.ExactMethodMatcher;
import com.newrelic.agent.instrumentation.methodmatchers.MethodMatcher;
import com.newrelic.agent.instrumentation.pointcuts.FieldAccessor;
import com.newrelic.agent.instrumentation.pointcuts.InterfaceMixin;
import com.newrelic.agent.instrumentation.pointcuts.PointCut;
import com.newrelic.agent.tracers.ClassMethodSignature;
import com.newrelic.agent.tracers.RetryException;
import com.newrelic.agent.tracers.Tracer;
import com.newrelic.agent.tracers.metricname.ClassMethodMetricNameFormat;
import com.newrelic.agent.tracers.metricname.MetricNameFormat;
import com.newrelic.agent.tracers.metricname.SimpleMetricNameFormat;
import com.newrelic.agent.tracers.servlet.BasicRequestRootTracer;
import com.newrelic.agent.transaction.TransactionNamingPolicy;
import com.newrelic.api.agent.ExtendedRequest;
import com.newrelic.api.agent.HeaderType;
import com.newrelic.api.agent.Request;
import com.newrelic.api.agent.Response;

@PointCut
public class PlayDispatcherPointCut extends TracerFactoryPointCut {

    public static final String PLAY_INSTRUMENTATION_GROUP_NAME = "play_instrumentation";
    public static final boolean DEFAULT_ENABLED = true;

    private static final String POINT_CUT_NAME = PlayDispatcherPointCut.class.getName();
    private static final String ACTION_INVOKER_CLASS = "play/mvc/ActionInvoker";
    private static final String SCOPE_PARAMS_CLASS = "play/mvc/Scope$Params";
    private static final String HTTP_COOKIE_CLASS = "play/mvc/Http$Cookie";
    private static final String HTTP_HEADER_CLASS = "play/mvc/Http$Header";
    private static final String HTTP_REQUEST_CLASS = "play/mvc/Http$Request";
    private static final String HTTP_RESPONSE_CLASS = "play/mvc/Http$Response";
    private static final String INVOKE_METHOD_NAME = "invoke";
    private static final String INVOKE_METHOD_DESC = "(Lplay/mvc/Http$Request;Lplay/mvc/Http$Response;)V";
    public static final String UNKNOWN_CONTROLLER_ACTION = "UNKNOWN";
    public static final String PLAY_CONTROLLER_ACTION = "PlayControllerAction";

    public PlayDispatcherPointCut(PointCutClassTransformer classTransformer) {
        super(createPointCutConfig(), createClassMatcher(), createMethodMatcher());
    }

    private static PointCutConfiguration createPointCutConfig() {
        return new PointCutConfiguration(POINT_CUT_NAME, PLAY_INSTRUMENTATION_GROUP_NAME, DEFAULT_ENABLED);
    }

    private static ClassMatcher createClassMatcher() {
        return new ExactClassMatcher(ACTION_INVOKER_CLASS);
    }

    private static MethodMatcher createMethodMatcher() {
        return new ExactMethodMatcher(INVOKE_METHOD_NAME, INVOKE_METHOD_DESC);
    }

    @Override
    public boolean isDispatcher() {
        return true;
    }

    @Override
    public final Tracer doGetTracer(Transaction tx, ClassMethodSignature sig, Object object, Object[] args) {
        Tracer rootTracer = tx.getRootTracer();
        if (rootTracer != null) {
            return null;
        }
        PlayHttpRequest request = (PlayHttpRequest) args[0];
        Transaction savedTx = getAndClearSavedTransaction(request);
        if (savedTx != null) {
            resumeTransaction(savedTx);
            // tell the TracerService to retry the getTracer call with the new current transaction
            throw new RetryException();
        }
        TransactionState transactionState = tx.getTransactionState();
        if (!(transactionState instanceof PlayTransactionStateImpl)) {
            transactionState = new PlayTransactionStateImpl(request);
            tx.setTransactionState(transactionState);
            // tell the TracerService to retry the getTracer call with the new transaction state
            throw new RetryException();
        }
        Tracer tracer = createTracer(tx, sig, object, args);
        if (tracer != null) {
            setTransactionName(tx, request);
        }
        return tracer;
    }

    /**
     * Get the saved transaction from the Play request.
     */
    private Transaction getAndClearSavedTransaction(PlayHttpRequest request) {
        Transaction savedTx = (Transaction) request._nr_getTransaction();
        if (savedTx == null) {
            return null;
        }
        request._nr_setTransaction(null);
        return savedTx;
    }

    /**
     * Set the current transaction to the saved transaction, and tell the @link{TransactionState} that the saved
     * transaction is about to be resumed.
     */
    private void resumeTransaction(Transaction savedTx) {
        TransactionState transactionState = savedTx.getTransactionState();
        transactionState.resume();
        Transaction.clearTransaction();
        Transaction.setTransaction(savedTx);
    }

    private Tracer createTracer(Transaction tx, ClassMethodSignature sig, Object object, Object[] args) {
        try {
            return new BasicRequestRootTracer(tx, sig, object, getRequest(tx, sig, object, args),
                                              getResponse(tx, sig, object, args), getMetricNameFormat(tx, sig, object, args)) {
                @Override
                protected final void doFinish(Throwable throwable) {
                    getTransactionActivity().getTransaction().addOutboundResponseHeaders();
                    super.doFinish(throwable);
                }

                @Override
                protected void doFinish(int opcode, Object returnValue) {
                    getTransactionActivity().getTransaction().addOutboundResponseHeaders();
                    super.doFinish(opcode, returnValue);
                }
            };
        } catch (Exception e) {
            String msg = MessageFormat.format("Unable to create request dispatcher tracer: {0}", e);
            if (Agent.LOG.isFinestEnabled()) {
                Agent.LOG.log(Level.WARNING, msg, e);
            } else {
                Agent.LOG.warning(msg);
            }
            return null;
        }
    }

    private MetricNameFormat getMetricNameFormat(Transaction transaction, ClassMethodSignature sig, Object object,
            Object[] args) {
        return new SimpleMetricNameFormat(MetricNames.REQUEST_DISPATCHER, ClassMethodMetricNameFormat.getMetricName(
                sig, object, MetricNames.REQUEST_DISPATCHER));
    }

    private Response getResponse(Transaction tx, ClassMethodSignature sig, Object object, Object[] args)
            throws Exception {
        return DelegatingPlayHttpResponse.create((PlayHttpResponse) args[1]);
    }

    private Request getRequest(Transaction tx, ClassMethodSignature sig, Object object, Object[] args) throws Exception {
        return DelegatingPlayHttpRequest.create((PlayHttpRequest) args[0]);
    }

    private void setTransactionName(Transaction tx, PlayHttpRequest request) {
        if (!tx.isTransactionNamingEnabled()) {
            return;
        }
        String action = request._nr_getAction();
        action = action == null ? UNKNOWN_CONTROLLER_ACTION : action;
        setTransactionName(tx, action);
    }

    private void setTransactionName(Transaction tx, String action) {
        TransactionNamingPolicy policy = TransactionNamingPolicy.getHigherPriorityTransactionNamingPolicy();
        if (Agent.LOG.isLoggable(Level.FINER)) {
            if (policy.canSetTransactionName(tx, TransactionNamePriority.FRAMEWORK_LOW)) {
                String msg = MessageFormat.format("Setting transaction name to \"{0}\" using Play controller action",
                        action);
                Agent.LOG.finer(msg);
            }
        }
        policy.setTransactionName(tx, action, PLAY_CONTROLLER_ACTION, TransactionNamePriority.FRAMEWORK_LOW);
    }

    @InterfaceMixin(originalClassName = HTTP_HEADER_CLASS)
    public interface PlayHttpHeader {

        String value();
    }

    @InterfaceMixin(originalClassName = SCOPE_PARAMS_CLASS)
    public interface PlayScopeParams {

        String[] getAll(String key);

        Map<String, String[]> all();
    }

    @InterfaceMixin(originalClassName = HTTP_COOKIE_CLASS)
    public interface PlayHttpCookie {

        @FieldAccessor(fieldName = "value", existingField = true)
        String _nr_getValue();
    }

    @InterfaceMixin(originalClassName = HTTP_REQUEST_CLASS)
    public interface PlayHttpRequest {

        @FieldAccessor(fieldName = "transaction")
        void _nr_setTransaction(Object tx);

        @FieldAccessor(fieldName = "transaction")
        Object _nr_getTransaction();

        @FieldAccessor(fieldName = "headers", existingField = true)
        Map<?, ?> _nr_getHeaders();

        @FieldAccessor(fieldName = "cookies", existingField = true)
        Map<?, ?> _nr_getCookies();

        @FieldAccessor(fieldName = "url", existingField = true)
        String _nr_getUrl();

        @FieldAccessor(fieldName = "action", existingField = true)
        String _nr_getAction();

        @FieldAccessor(fieldName = "method", existingField = true)
        String _nr_getMethod();

        @FieldAccessor(fieldName = "params", fieldDesc = "Lplay/mvc/Scope$Params;", existingField = true)
        Object _nr_getParams();

    }

    @InterfaceMixin(originalClassName = HTTP_RESPONSE_CLASS)
    public interface PlayHttpResponse {

        void setHeader(String name, String value);

        @FieldAccessor(fieldName = "status", existingField = true)
        Integer _nr_getResponseStatus();

        @FieldAccessor(fieldName = "contentType", existingField = true)
        String _nr_getContentType();
    }

    private static class DelegatingPlayHttpRequest extends ExtendedRequest {

        private final PlayHttpRequest delegate;

        private DelegatingPlayHttpRequest(PlayHttpRequest delegate) {
            this.delegate = delegate;
        }

        @Override
        public HeaderType getHeaderType() {
            return HeaderType.HTTP;
        }

        @Override
        public Enumeration<?> getParameterNames() {
            PlayScopeParams playScopeParams = getScopeParams();
            if (playScopeParams == null) {
                return null;
            }
            Map<String, String[]> params = playScopeParams.all();
            return Collections.enumeration(params.keySet());
        }

        @Override
        public String[] getParameterValues(String name) {
            PlayScopeParams playScopeParams = getScopeParams();
            if (playScopeParams == null) {
                return new String[0];
            }
            return playScopeParams.getAll(name);
        }

        /**
         * Play has no equivalent.
         */
        @Override
        public Object getAttribute(String name) {
            return null;
        }

        @Override
        public String getRequestURI() {
            return delegate._nr_getUrl();
        }

        @Override
        public String getHeader(String name) {
            if (name == null) {
                return null;
            }
            Map<?, ?> headers = delegate._nr_getHeaders();
            PlayHttpHeader header = (PlayHttpHeader) headers.get(name);
            if (header != null) {
                return header.value();
            }
            // Play stores headers as lower case.
            // For example: "X-NewRelic-ID" is stored as "x-newrelic-id"
            header = (PlayHttpHeader) headers.get(name.toLowerCase());
            return header == null ? null : header.value();
        }

        /**
         * Play has no equivalent.
         */
        @Override
        public String getRemoteUser() {
            return null;
        }

        @Override
        public String getCookieValue(String name) {
            if (name == null) {
                return null;
            }
            Map<?, ?> cookies = delegate._nr_getCookies();
            PlayHttpCookie cookie = (PlayHttpCookie) cookies.get(name);
            return cookie == null ? null : cookie._nr_getValue();
        }

        @Override
        public String getMethod() {
            return delegate._nr_getMethod();
        }

        private PlayScopeParams getScopeParams() {
            Object scopeParams = delegate._nr_getParams();
            if (scopeParams instanceof PlayScopeParams) {
                return (PlayScopeParams) scopeParams;
            }
            return null;
        }

        static Request create(PlayHttpRequest delegate) {
            return new DelegatingPlayHttpRequest(delegate);
        }

    }

    private static class DelegatingPlayHttpResponse implements Response {

        private final PlayHttpResponse delegate;

        private DelegatingPlayHttpResponse(PlayHttpResponse delegate) {
            this.delegate = delegate;
        }

        @Override
        public HeaderType getHeaderType() {
            return HeaderType.HTTP;
        }

        /**
         * Play has no equivalent.
         */
        @Override
        public String getStatusMessage() {
            return null;
        }

        @Override
        public void setHeader(String name, String value) {
            delegate.setHeader(name, value);
        }

        @Override
        public int getStatus() {
            Integer status = delegate._nr_getResponseStatus();
            return status == null ? 0 : status.intValue();
        }

        @Override
        public String getContentType() {
            return delegate._nr_getContentType();
        }

        static Response create(PlayHttpResponse delegate) {
            return new DelegatingPlayHttpResponse(delegate);
        }
    }
}
