package com.newrelic.agent.instrumentation.pointcuts;

import java.text.MessageFormat;

import com.newrelic.agent.Agent;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.instrumentation.PointCutClassTransformer;
import com.newrelic.agent.instrumentation.PointCutConfiguration;
import com.newrelic.agent.instrumentation.TracerFactoryPointCut;
import com.newrelic.agent.instrumentation.classmatchers.InterfaceMatcher;
import com.newrelic.agent.tracers.ClassMethodSignature;
import com.newrelic.agent.tracers.OtherRootTracer;
import com.newrelic.agent.tracers.Tracer;
import com.newrelic.agent.tracers.metricname.ClassMethodMetricNameFormat;

/**
 * Instruments Quartz jobs.
 * 
 * @author sdaubin
 * 
 */
@PointCut
public class QuartzJobPointCut extends TracerFactoryPointCut {

    public QuartzJobPointCut(PointCutClassTransformer classTransformer) {
        super(new PointCutConfiguration("quartz_job"), new InterfaceMatcher("org/quartz/Job"),
                createExactMethodMatcher("execute", "(Lorg/quartz/JobExecutionContext;)V"));
    }

    @Override
    public Tracer doGetTracer(Transaction transaction, ClassMethodSignature sig, Object job, Object[] args) {
        return new QuartzJobTracer(transaction, sig, job, args[0]);
    }

    @Override
    public boolean isDispatcher() {
        return true;
    }

    private static class QuartzJobTracer extends OtherRootTracer {

        public QuartzJobTracer(Transaction transaction, ClassMethodSignature sig, Object job, Object context) {
            super(transaction, sig, job, new ClassMethodMetricNameFormat(sig, job, MetricNames.OTHER_TRANSACTION_JOB));
            try {
                Object jobDetail = context.getClass().getMethod("getJobDetail").invoke(context);
                setAttribute("name", jobDetail.getClass().getMethod("getFullName").invoke(jobDetail));
            } catch (Throwable e) {
                Agent.LOG.finer(MessageFormat.format("An error occurred getting a Quartz job name", e.toString()));
            }
            if (Agent.isDebugEnabled()) {
                Agent.LOG.fine("Quartz job started");
            }
        }

    }

}
