package com.newrelic.agent.install;

import java.io.File;

import com.newrelic.agent.util.EditableFile;

public class TomcatSelfInstaller extends SelfInstaller {

    // Define constants and regular expressions

    // Tomcat 5.x, 6.x, 7.0
    private final String scriptPath = "/bin/catalina";
    private final String altScriptPath = "/bin/catalina.50";

    // Script snippets to look for, or add
    private final String unixAgentSet = "^(export )?CATALINA_OPTS=(.*)\\-javaagent:(.*)newrelic.jar(.*)$";
    private final String unixLocator = "^(export )?CATALINA_OPTS=(.*)$";
    private final String altUnixLocator = "^# OS specific support(.*)$";
    private final String windowsAgentSet = "^(SET|set) CATALINA_OPTS=%CATALINA_OPTS% \\-javaagent:(.*)$";
    private final String winLocator = "^rem Guess CATALINA_HOME if not defined(.*)$";
    private final String altWinLocator = "^rem Suppress Terminate batch job(.*)$";

    private String rootDir;

    /**
     * Adds text like this to Tomcat's catalina start script:
     * 
     * # ---- New Relic switch automatically added to start command on Aug 13 2010, 20:06:55
     * NR_JAR=/usr/tomcat/6.0.29/newrelic/newrelic.jar; export NR_JAR CATALINA_OPTS="$CATALINA_OPTS -javaagent:$NR_JAR";
     * export CATALINA_OPTS
     * 
     * Separating the export statement is intentional. Some Solaris environments do not support the syntax: export
     * VAR=value
     * 
     */
    @Override
    public boolean backupAndEditStartScript(String appServerRootDir) {

        rootDir = appServerRootDir;

        // Always edit catalina.ext
        boolean result = backupAndEdit(appServerRootDir + getStartScript());

        // And catalina.50.ext if it's there
        File catalina50 = new File(appServerRootDir + getAlternateStartScript());
        if (catalina50.exists()) {
            result &= backupAndEdit(catalina50.toString());
        }

        return result;

    }

    private boolean backupAndEdit(String fullPathToScript) {
        try {
            EditableFile file = new EditableFile(fullPathToScript);

            String fullSwitch = lineSep + getCommentForAgentSwitch(file.comment) + lineSep + getAgentSettings();

            // Install the javaagent switch, if it is not already set.
            if (!file.contains(getAgentAlreadySetExpr())) {

                if (file.contains(getLocator())) {
                    backup(file);
                    if (osIsWindows) {
                        file.insertBeforeLocator(getLocator(), fullSwitch, true);
                    } else {
                        file.insertAfterLocator(getLocator(), fullSwitch, true);
                    }
                    System.out.println("* Added agent switch to start script " + file.getLocation() + lineSep);
                } else if (file.contains(getAlternateLocator())) {
                    backup(file);
                    file.insertBeforeLocator(getAlternateLocator(), fullSwitch, true);
                    System.out.println("* Added agent switch to start script " + file.getLocation() + lineSep);
                } else {
                    // FIXME do a real check to make sure the file exists
                    System.out.println("* Did not locate Tomcat start script. No edit performed" + lineSep);
                    return false;
                }

            } else {
                System.out.println("* Did not edit start script " + file.getLocation() + " because:");
                System.out.println("The agent switch is already set" + SelfInstaller.lineSep);
            }

            return true;

        } catch (Exception e) {
            System.out.println(e.getMessage());
            return false;
        }
    }

    @Override
    public String getStartScript() {
        String path = scriptPath;
        if (osIsWindows) {
            path = path.replaceAll("/", "\\\\");
        }
        return path + (osIsWindows ? DOTBAT : DOTSH);
    }

    @Override
    public String getAlternateStartScript() {
        return altScriptPath + (osIsWindows ? DOTBAT : DOTSH);
    }

    @Override
    public String getLocator() {
        return osIsWindows ? winLocator : unixLocator;
    }

    @Override
    public String getAlternateLocator() {
        return osIsWindows ? altWinLocator : altUnixLocator;
    }

    @Override
    public String getAgentSettings() {
        // TODO Update this Unix switch to handle spaces in the newrelic path.
        String unixSwitch = "NR_JAR=" + rootDir + "/newrelic/newrelic.jar; export NR_JAR" + lineSep
                + "CATALINA_OPTS=\"\\$CATALINA_OPTS -javaagent:\\$NR_JAR\"; export CATALINA_OPTS" + lineSep;

        String winSwitch = "set CATALINA_OPTS=%CATALINA_OPTS% -javaagent:\"" + rootDir.replaceAll("\\\\", "/")
                + "/newrelic/newrelic.jar\"" + lineSep;

        return osIsWindows ? winSwitch : unixSwitch;
    }

    @Override
    public String getAgentAlreadySetExpr() {
        return osIsWindows ? windowsAgentSet : unixAgentSet;
    }

}
