package com.newrelic.agent.install;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.newrelic.agent.util.EditableFile;

/**
 * A SelfInstaller can back up and edit an app server start script.
 * 
 * Implement one SelfInstaller for each supported appserver type.
 * 
 * @author nlitt
 * 
 */
public abstract class SelfInstaller {

    public static final String lineSep = System.getProperty("line.separator");
    public static final String fileSep = System.getProperty("file.separator");
    static final String DOTBAT = ".bat";
    static final String DOTSH = ".sh";
    static final String DOTCONF = ".conf";

    public enum OS {
        MAC, UNIX, WINDOWS
    }

    public OS os;
    public boolean osIsMac;
    public boolean osIsUnix;
    public boolean osIsWindows;

    private DateFormat df;

    public SelfInstaller() {
        df = new SimpleDateFormat("yyyy MMM dd, HH:mm:ss");
        os = getOS();
        if (os == OS.MAC) {
            osIsMac = true;
            osIsUnix = false;
            osIsWindows = false;
        } else if (os == OS.UNIX) {
            osIsMac = false;
            osIsUnix = true;
            osIsWindows = false;
        } else if (os == OS.WINDOWS) {
            osIsMac = false;
            osIsUnix = false;
            osIsWindows = true;
        }
    }

    // Methods delegated to the various app server types

    /**
     * Returns true if the backup/edit operation was successful, false otherwise.
     */
    public abstract boolean backupAndEditStartScript(String appServerRootDir);

    /**
     * Returns the path to the appserver start script to edit during install. The path is relative to the appserver
     * root.
     */
    public abstract String getStartScript();

    /**
     * Returns an alternate path to the appserver start script to edit during install. The path is relative to the
     * appserver root.
     * 
     */
    public abstract String getAlternateStartScript();

    /**
     * Returns a regular expression for locating the point in the start script where the agent settings should be
     * inserted.
     * 
     */
    public abstract String getLocator();

    /**
     * Returns an alternate regular expression for locating the point in the start script where the agent settings
     * should be inserted.
     * 
     */
    public abstract String getAlternateLocator();

    /**
     * Returns the text that, when inserted into the appserver start script, will activate the agent.
     */
    public abstract String getAgentSettings();

    /**
     * Returns a regular expression for testing whether the agent settings are already present in the start script.
     */
    public abstract String getAgentAlreadySetExpr();

    // Common utility methods

    public OS getOS() {
        String osName = System.getProperty("os.name");
        if (osName.toLowerCase().startsWith("windows")) {
            return OS.WINDOWS;
        } else if (osName.toLowerCase().startsWith("mac")) {
            return OS.MAC;
        } else {
            return OS.UNIX;
        }
    }

    /**
     * Returns a commented line that we can use to mark the agent options in the start switch.
     * 
     */
    public String getCommentForAgentSwitch(String commentChars) {
        return commentChars + " ---- New Relic switch automatically added to start command on " + df.format(new Date());
    }

    /**
     * Backs up the start script and prints a nice message
     * 
     * @param file
     */
    public void backup(EditableFile file) {
        String backedUpFile = file.backup();
        if (!backedUpFile.equals("")) {
            System.out.println("* Backed up start script to " + backedUpFile + lineSep);
        }
    }

}
