package com.newrelic.agent.install;

import java.io.File;

import com.newrelic.agent.util.EditableFile;

public class JettySelfInstaller extends SelfInstaller {

    // Define constants and regular expressions

    // Jetty 6.1, 7.0, 8.0
    private final String scriptPath = "/bin/jetty";
    private final String altScriptPath = "/bin/jetty-cygwin";

    // Script snippets to look for, or add
    private final String agentAlreadySet = "^(export )?JAVA_OPTIONS=(.*)\\-javaagent:(.*)newrelic.jar(.*)$";
    private final String winlinLocator = "[# \r\n]+ This is how the Jetty server will be started([ \t]+)?";

    private String rootDir;

    /**
     * Adds text like this to Jetty's jetty.sh start script:
     * 
     * # ---- New Relic switch automatically added to start command on Aug 13 2010, 20:06:55
     * NR_JAR=/usr/jetty/6.1.25/newrelic/newrelic.jar; export NR_JAR JAVA_OPTIONS="${JAVA_OPTIONS} -javaagent:$NR_JAR";
     * export JAVA_OPTIONS
     * 
     * Separating the export statement is intentional. Some Solaris environments do not support the syntax: export
     * VAR=value
     * 
     */
    @Override
    public boolean backupAndEditStartScript(String appServerRootDir) {

        rootDir = appServerRootDir;

        // Always edit jetty.sh
        boolean result = backupAndEdit(appServerRootDir + getStartScript());

        // And jetty-cygwin.sh if it's there
        File jettyCygwin = new File(appServerRootDir + getAlternateStartScript());
        if (jettyCygwin.exists()) {
            result &= backupAndEdit(jettyCygwin.toString());
        }

        return result;

    }

    private boolean backupAndEdit(String fullPathToScript) {
        try {
            EditableFile file = new EditableFile(fullPathToScript);

            String fullSwitch = lineSep + lineSep + getCommentForAgentSwitch(file.comment) + lineSep
                    + getAgentSettings();

            // Install the javaagent switch, if it is not already set.
            if (!file.contains(getAgentAlreadySetExpr())) {

                if (file.contains(getLocator())) {
                    backup(file);
                    file.insertBeforeLocator(getLocator(), fullSwitch, false);
                    System.out.println("* Added agent switch to start script " + file.getLocation() + lineSep);
                } else {
                    System.out.println("* Did not locate Jetty start script. No edit performed" + lineSep);
                }

            } else {
                System.out.println("* Did not edit start script " + file.getLocation() + " because:");
                System.out.println("The agent switch is already set" + lineSep);
            }

            return true;

        } catch (Exception e) {
            System.out.println(e.getMessage());
            return false;
        }
    }

    @Override
    public String getStartScript() {
        String path = scriptPath;
        if (osIsWindows) {
            path = path.replaceAll("/", "\\\\");
        }
        return path + DOTSH; // it's sh on both Windows and Unix
    }

    @Override
    public String getAlternateStartScript() {
        String path = altScriptPath;
        if (osIsWindows) {
            path = path.replaceAll("/", "\\\\");
        }
        return path + DOTSH;
    }

    @Override
    public String getLocator() {
        return winlinLocator;
    }

    @Override
    public String getAlternateLocator() {
        return getLocator();
    }

    @Override
    public String getAgentSettings() {
        String switchPath = rootDir;
        if (osIsWindows) {
            switchPath = switchPath.replaceAll("\\\\", "/");
        }
        return "NR_JAR=" + switchPath + "/newrelic/newrelic.jar; export NR_JAR" + lineSep
                + "JAVA_OPTIONS=\"\\$\\{JAVA_OPTIONS\\} -javaagent:\\$NR_JAR\"; export JAVA_OPTIONS";
    }

    @Override
    public String getAgentAlreadySetExpr() {
        return agentAlreadySet;
    }

}
