package com.newrelic.agent.install;

import java.io.File;

/**
 * Inspects a specified root directory to determine whether it contains a known app server type.
 * 
 * SRH - added support for Glassfish
 * 
 * @author natasha
 * 
 */
public class AppServerIdentifier {

    public enum AppServerType {
        TOMCAT("Tomcat"), JETTY("Jetty"), JBOSS("JBoss"), JBOSS7("JBoss7"), GLASSFISH("Glassfish"), UNKNOWN("Unknown");

        private final String name;

        AppServerType(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }

    }

    // Marker files that signify the presence of a particular app server.

    private static String tomcat5Marker = "/server/lib/catalina.jar";
    private static String tomcat67Marker = "/lib/catalina.jar";

    // TODO the marker should be a /start.jar containing an org.mortbay namespace
    private static String jettyMarker = "/bin/jetty.sh";

    private static String jBoss4Marker = "/lib/jboss-common.jar";
    private static String jBoss56Marker = "/lib/jboss-common-core.jar";
    private static String jBoss7Marker = "/jboss-modules.jar";

    // SRH - This seems like a pretty generic file, but there is nothing in the directory
    // that will more accurately identify Glassfish.
    private static String glassfish3Marker = "/config/domain.xml";

    public static AppServerType getAppServerType(String path) throws Exception {
        return getAppServerType(new File(path));
    }

    /**
     * Returns the #AppServerType living in the specified root directory, or type UNKNOWN if no app server is
     * recognized. Throws an exception if the specified root dir is invalid or unreadable.
     * 
     */
    public static AppServerType getAppServerType(File rootDir) throws Exception {

        if (!rootDir.exists()) {
            throw new Exception("App server root " + rootDir.toString() + "does not exist on filesystem.");
        }
        if (!rootDir.isDirectory()) {
            throw new Exception("App server root " + rootDir.toString() + "is not a directory.");
        }

        if (isTomcat(rootDir.toString())) {
            return AppServerType.TOMCAT;
        }

        if (isJetty(rootDir.toString())) {
            return AppServerType.JETTY;
        }

        if (isJBoss(rootDir.toString())) {
            return AppServerType.JBOSS;
        }

        if (isJBoss7(rootDir.toString())) {
            return AppServerType.JBOSS7;
        }

        // SRH - Check for Glassfish
        if (isGlassfish(rootDir.toString())) {
            return AppServerType.GLASSFISH;
        }

        return AppServerType.UNKNOWN;
    }

    /**
     * Returns true if the specified root dir contains a Tomcat install.
     */
    private static boolean isTomcat(String rootDir) {
        return (markerFileExists(rootDir + tomcat5Marker) || markerFileExists(rootDir + tomcat67Marker));
    }

    /**
     * Returns true if the specified root dir contains a Jetty install.
     */
    private static boolean isJetty(String rootDir) {
        return markerFileExists(rootDir + jettyMarker);
    }

    /**
     * Returns true if the specified root dir contains a JBoss install.
     */
    private static boolean isJBoss(String rootDir) {
        return (markerFileExists(rootDir + jBoss4Marker) || markerFileExists(rootDir + jBoss56Marker));
    }

    /**
     * Returns true if the specified root dir contains a JBoss install.
     */
    private static boolean isJBoss7(String rootDir) {
        return markerFileExists(rootDir + jBoss7Marker);
    }

    /**
     * SRH - Returns true if the specified root dir contains a Glassfish install.
     */
    private static boolean isGlassfish(String rootDir) {
        return (markerFileExists(rootDir + glassfish3Marker));
    }

    /**
     * Returns true if the specified string is an existing file.
     */
    private static boolean markerFileExists(String path) {
        File markerFile = new File(path);
        if (markerFile.exists() && markerFile.isFile()) {
            return true;
        }
        return false;
    }

}
