package com.newrelic.agent.heap;

import com.newrelic.agent.HarvestListener;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.config.ClassHistogramConfig;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.analytics.ClassHistogramRowEvent;
import com.newrelic.agent.service.analytics.ClassHistogramStatsEvent;
import com.newrelic.agent.service.analytics.InternalCustomEventService;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsService;
import com.newrelic.agent.stats.StatsWorks;

import java.lang.management.ManagementFactory;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ClassHistogramService extends AbstractService implements HarvestListener {

    private final ClassHistogramConfig classHistogramConfig;
    private final Map<String, ClassHistogramRowEvent> lastHistogram;
    private int pid;

    public ClassHistogramService() {
        super(ClassHistogramService.class.getSimpleName());
        classHistogramConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getClassHistogramConfig();
        lastHistogram = new HashMap<String, ClassHistogramRowEvent>();
    }

    @Override
    public void beforeHarvest(String appName, StatsEngine statsEngine) {
    }

    @Override
    public void afterHarvest(String appName) {
        List<String> output = ClassHistogramHelper.runJmapCommand(classHistogramConfig.jmapPath(), pid);
        if (output != null) {
            sendHistogramData(appName, output);
            sendHistogramStats(appName, output);
        }
    }

    @Override
    protected void doStart() throws Exception {
        if (isEnabled() && ClassHistogramHelper.jmapExists(classHistogramConfig.jmapPath())) {
            String name = ManagementFactory.getRuntimeMXBean().getName();
            if (name != null && !name.isEmpty()) {
                String[] pidString = name.split("@");
                if (pidString.length > 0 && !pidString[0].isEmpty()) {
                    try {
                        Integer parsedPid = Integer.parseInt(pidString[0]);
                        if (parsedPid != null && parsedPid > 0) {
                            pid = parsedPid;
                        }
                    } catch (NumberFormatException e) {
                    }
                }
            }

            // only add the listener if the service is enabled and we have a valid process id
            if (pid > 0) {
                ServiceFactory.getHarvestService().addHarvestListener(this);
            }
        }
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getHarvestService().removeHarvestListener(this);
    }

    @Override
    public boolean isEnabled() {
        return classHistogramConfig.enabled();
    }

    /**
     * The object histogram can be thousands of rows, so we only keep the top X classes as measured by the third
     * column, total size in bytes of instantiated objects. The output is already sorted in descending order by
     * the third column. It Ignores all the agent newrelic classes so customers don't see tons of ASM and other
     * miscellaneous objects and dependencies that would confuse them.
     */
    private void sendHistogramData(String appName, List<String> output) {
        InternalCustomEventService eventService = ServiceFactory.getServiceManager().getInternalCustomEventService();

        Map<String, ClassHistogramRowEvent> currentHistogram = new HashMap<String, ClassHistogramRowEvent>();

        int countOther = 0;
        for (String row : output) {
            if (countOther >= classHistogramConfig.classesPerHistogram()) {
                break;
            }

            // typical row looks like: "1:  278744  13311808  [C" where the first value is meaningless
            String[] parts = ClassHistogramHelper.split(row);
            if (parts.length == 4) {
                Long classInstances = Long.parseLong(parts[1]);
                Long classBytes = Long.parseLong(parts[2]);
                String className = parts[3].trim().replaceAll(";", "");
                String classType = ClassHistogramHelper.getClassLabel(className);

                // ignore newrelic java agent classes
                if (!ClassHistogramHelper.isNewRelicJavaAgentClass(className)) {
                    long classInstancesDelta = 0;
                    long classBytesDelta = 0;
                    ClassHistogramRowEvent lastEvent = lastHistogram.get(className);
                    if (lastEvent != null) {
                        classInstancesDelta = classInstances - lastEvent.getClassInstances();
                        classBytesDelta = classBytes - lastEvent.getClassBytes();
                    }

                    ClassHistogramRowEvent rowEvent = new ClassHistogramRowEvent(appName, classInstances, classBytes,
                            classInstancesDelta, classBytesDelta, className, classType);

                    currentHistogram.put(className, rowEvent);

                    eventService.addInternalCustomEvent(rowEvent);
                    countOther++;
                }
            }
        }

        lastHistogram.clear();
        lastHistogram.putAll(currentHistogram);
    }

    /**
     * Loops through every class in the histogram output and counts the total number of objects and total size of
     * those objects. Also reports the values as metrics.
     *
     * @param output the amount and size of objects in the heap
     */
    private void sendHistogramStats(String appName, List<String> output) {
        long totalObjects = 0;
        long totalSize = 0;

        for (String row : output) {
            String[] parts = ClassHistogramHelper.split(row);
            if (parts.length == 4) {
                totalObjects += Long.parseLong(parts[1]);
                totalSize += Long.parseLong(parts[2]);
            }
        }

        ClassHistogramStatsEvent statsEvent = new ClassHistogramStatsEvent(appName, totalObjects, totalSize);
        ServiceFactory.getServiceManager().getInternalCustomEventService().addInternalCustomEvent(statsEvent);

        // reporting these 2 as metrics makes it easier to create alerts
        StatsService statsService = ServiceFactory.getStatsService();
        statsService.doStatsWork(StatsWorks.getRecordMetricWork(MetricNames.CLASS_HISTOGRAM_HEAP_TOTAL_SIZE, totalSize));
        statsService.doStatsWork(StatsWorks.getRecordMetricWork(MetricNames.CLASS_HISTOGRAM_HEAP_TOTAL_OBJECTS, totalObjects));
    }

}
