package com.newrelic.agent.heap;

import com.google.common.collect.Sets;
import com.newrelic.agent.Agent;
import com.newrelic.agent.config.JavaVersionUtils;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.management.ManagementFactory;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Pattern;

public class ClassHistogramHelper {

    // The "cmd.exe /c" allows us to execute the "dir" command since it's a CMD command. The "/-n" is to reformat
    // the result of "dir" so the folder name comes first and we can search "starts with" instead of "contains"
    private static final String WINDOWS_DIR_COMMAND = "cmd.exe /c dir /-n ";

    // exclude (arrays of) java agent classes without excluding all new relic classes in general, which other teams created
    private static final Pattern JAVA_AGENT_CLASS = Pattern.compile("^(\\[*L?com.(nr|newrelic).(agent|api.agent|weave|bootstrap|org.apache)).*");

    // split the jmap histogram rows on white space
    private static final Pattern ROW_SPLITTER = Pattern.compile("\\s+");

    private static final Set<String> HOTSPOT = Sets.newHashSet("<constantPoolKlass>", "<constantPoolCacheKlass>",
            "<methodKlass>", "<methodDataKlass>", "<constMethodKlass>", "<instanceKlass>", "<instanceKlassKlass>");

    private static final Set<String> BASIC = Sets.newHashSet("java.lang.Boolean", "java.lang.Byte", "java.lang.Short",
            "java.lang.Integer", "java.lang.Long", "java.lang.Float", "java.lang.Double", "java.lang.Character");

    private static final Set<String> ARRAY_BASIC = Sets.newHashSet("[Ljava.lang.Boolean", "[Ljava.lang.Byte", "[Ljava.lang.Short",
            "[Ljava.lang.Integer", "[Ljava.lang.Long", "[Ljava.lang.Float", "[Ljava.lang.Double", "[Ljava.lang.Character");

    private static final Set<String> ARRAY_PRIMITIVE = Sets.newHashSet("[Z", "[B", "[S", "[I", "[J", "[F", "[D", "[C");

    private enum ClassType {
        HOTSPOT("hotspot"),
        BASIC("basic"),
        ARRAY_BASIC("array_basic"),
        ARRAY_PRIMITIVE("array_primitive"),
        CLASS("class"),
        ARRAY_CLASS("array_class");

        private final String typeName;

        ClassType(String typeName) {
            this.typeName = typeName;
        }
    }

    /**
     * The class types used for faceting in Insights.
     */
    private static ClassType getType(String className) {
        // in order to match all array types, multidimensional arrays are classified as arrays
        if (className.startsWith("[[")) {
            className = className.replaceAll("\\[+", "[");
        }

        if (BASIC.contains(className)) {
            return ClassType.BASIC;
        } else if (ARRAY_BASIC.contains(className)) {
            return ClassType.ARRAY_BASIC;
        } else if (ARRAY_PRIMITIVE.contains(className)) {
            return ClassType.ARRAY_PRIMITIVE;
        } else if (className.startsWith("[L")) {
            return ClassType.ARRAY_CLASS;
        } else if (HOTSPOT.contains(className)) {
            return ClassType.HOTSPOT;
        } else {
            return ClassType.CLASS;
        }
    }

    public static String[] split(String row) {
        return ROW_SPLITTER.split(row);
    }

    public static String getClassLabel(String className) {
        return getType(className).typeName;
    }

    public static boolean isNewRelicJavaAgentClass(String className) {
        return JAVA_AGENT_CLASS.matcher(className).matches();
    }

    /**
     * @return true if OS is Windows, false otherwise
     */
    public static boolean isWindows() {
        String os = ManagementFactory.getOperatingSystemMXBean().getName().toLowerCase();
        return os.startsWith("windows");
    }

    /**
     * @return true if jmap is found, false if not
     */
    public static boolean jmapExists(String jmapPath) {
        if (JavaVersionUtils.isJavaSpecVersionGreaterThanOrEqualTo(JavaVersionUtils.JAVA_7)) {
            List<String> stdout;
            if (isWindows()) {
                // we add quotes here because we remove them in the config and there may be a path with a space in it
                stdout = runSysCommand(WINDOWS_DIR_COMMAND + addQuotes(jmapPath));

                if (stdout != null) {
                    for (String dir : stdout) {
                        // here a sample line would look like:
                        // Java     <DIR>     1/16/2018     9:52:00 AM
                        // the deliminators are spaces which is why we search for "jmap " and not "jmap"
                        if (dir.startsWith("jmap ")) {
                            return true;
                        }
                    }
                }
            } else {
                // If we are not on windows os then we must be on mac/linux/bsd os
                stdout = runSysCommand("ls " + jmapPath);

                if (stdout != null && stdout.contains("jmap")) {
                    return true;
                }
            }

            Agent.LOG.log(Level.INFO, "Class Histogram Service couldn't find jmap");
        } else {
            Agent.LOG.log(Level.INFO, "Class Histogram Service requires Java 7 or greater and the jmap executable. Service will be disabled.");
        }

        return false;
    }

    private static String addQuotes(String... parts) {
        StringBuilder builder = new StringBuilder();
        builder.append("\"");
        for (String part : parts) {
            builder.append(part);
        }
        builder.append("\"");
        return builder.toString();
    }

    /**
     * Runs the jmap utility as a system command.
     */
    public static List<String> runJmapCommand(String jmapPath, int pid) {
        if (isWindows()) {
            // executing a file with a path that has spaces only works if there are quotes around the whole path,
            // not just the part of the path with the space, which is why we do this
            return runSysCommand(addQuotes(jmapPath, "jmap") + " -histo " + pid);
        } else {
            return runSysCommand(jmapPath + "jmap -histo " + pid);
        }
    }

    /**
     * Runs a system command. It consumes both the stderr and stdout buffers to avoid deadlock. Without fully
     * consuming both buffers it is possible for the Process object to hang on the call to waitFor().
     *
     * @param command the system command to run
     * @return list of strings containing the standard output of the system command
     */
    private static List<String> runSysCommand(String command) {
        try {
            Process p = Runtime.getRuntime().exec(command);

            StreamEater stderr = new StreamEater(p.getErrorStream());
            stderr.start();
            StreamEater stdout = new StreamEater(p.getInputStream());
            stdout.start();

            int exitValue = p.waitFor();
            stderr.join();
            stdout.join();

            if (exitValue == 0) {
                return stdout.getOutput();
            }
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINER, t, "Couldn't read value of system command: " + command);
        }
        return null;
    }

    /**
     * A helper class meant to be used when performing system calls to fully consume an input stream.
     *
     * For more information see: http://www.javaworld.com/jw-12-2000/jw-1229-traps.html
     */
    private static class StreamEater extends Thread {
        final InputStream is;
        final List<String> output;

        public StreamEater(InputStream is) {
            this.is = is;
            this.output = new LinkedList<String>();
        }

        public List<String> getOutput() {
            return output;
        }

        @Override
        public void run() {
            try {
                BufferedReader br = new BufferedReader(new InputStreamReader(is));
                String line;
                while ((line = br.readLine()) != null) {
                    line = line.trim();
                    if (!line.isEmpty()) {
                        output.add(line);
                    }
                }
                br.close();
            } catch (Exception e) {
                Agent.LOG.log(Level.FINER, e, "Exception reading output stream");
            }
        }
    }

}
