package com.newrelic.agent.extension.jaxb;

import java.text.MessageFormat;
import java.util.Map;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

/**
 * Unmarshals a node into a JAXB class.
 */
public abstract class Unmarshaller<T> {

    private static final Map<Class<?>, Unmarshaller<?>> DEFAULT_UNMARSHALLERS = createDefaultUnmarshallers();

    private final Class<?> type;

    public Unmarshaller(Class<?> type) {
        this.type = type;
    }

    /**
     * Unmarshals a node into a JAXB class.
     */
    public abstract T unmarshall(Node node) throws UnmarshalException;

    @Override
    public String toString() {
        return "Unmarshaller [type=" + type + "]";
    }

    private static Map<Class<?>, Unmarshaller<?>> createDefaultUnmarshallers() {
        Map<Class<?>, Unmarshaller<?>> unmarshallers = Maps.newHashMap();

        unmarshallers.put(String.class, new Unmarshaller<String>(String.class) {

            @Override
            public String unmarshall(Node node) {
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    Node firstChild = node.getFirstChild();
                    if (firstChild == null) {
                        throw new DOMException(DOMException.NOT_FOUND_ERR, MessageFormat.format(
                                "The element node {0}  is present, but is empty.", node.getNodeName()));
                    } else {
                        return node.getFirstChild().getNodeValue();
                    }

                }
                return node.getNodeValue();
            }
        });

        unmarshallers.put(Boolean.class, new Unmarshaller<Boolean>(Boolean.class) {

            @Override
            public Boolean unmarshall(Node node) {
                return Boolean.valueOf(node.getNodeValue());
            }
        });

        unmarshallers.put(Double.class, new Unmarshaller<Double>(Double.class) {

            @Override
            public Double unmarshall(Node node) {
                return Double.valueOf(node.getNodeValue());
            }
        });

        unmarshallers.put(Long.class, new Unmarshaller<Long>(Long.class) {

            @Override
            public Long unmarshall(Node node) {
                return Long.valueOf(node.getNodeValue());
            }
        });

        unmarshallers.put(Integer.class, new Unmarshaller<Integer>(Integer.class) {

            @Override
            public Integer unmarshall(Node node) {
                return Integer.valueOf(node.getNodeValue());
            }
        });

        return ImmutableMap.copyOf(unmarshallers);
    }

    /**
     * Returns the default unmarshallers for built in types.
     * 
     * @return
     */
    public static Map<? extends Class<?>, ? extends Unmarshaller<?>> getDefaultUnmarshallers() {
        return DEFAULT_UNMARSHALLERS;
    }

}
