package com.newrelic.agent.errors;

import com.google.common.annotations.VisibleForTesting;
import com.newrelic.agent.Agent;
import com.newrelic.agent.Harvestable;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionData;
import com.newrelic.agent.TransactionErrorPriority;
import com.newrelic.agent.TransactionListener;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentConfigImpl;
import com.newrelic.agent.config.AgentConfigListener;
import com.newrelic.agent.config.DistributedTracingConfig;
import com.newrelic.agent.config.ErrorCollectorConfig;
import com.newrelic.agent.config.ExpectedErrorConfig;
import com.newrelic.agent.config.IgnoreErrorConfig;
import com.newrelic.agent.config.StripExceptionConfig;
import com.newrelic.agent.instrumentation.PointCut;
import com.newrelic.agent.instrumentation.methodmatchers.InvalidMethodDescriptor;
import com.newrelic.agent.instrumentation.yaml.PointCutFactory;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.analytics.DistributedSamplingPriorityQueue;
import com.newrelic.agent.service.analytics.ErrorEvent;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsWork;
import com.newrelic.agent.stats.TransactionStats;
import com.newrelic.agent.tracers.ClassMethodSignature;
import com.newrelic.agent.tracing.DistributedTraceService;
import com.newrelic.agent.transaction.TransactionThrowable;
import com.newrelic.agent.transport.HttpError;
import org.apache.http.HttpStatus;

import java.net.HttpURLConnection;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.logging.Level;

public class ErrorServiceImpl implements ErrorService {

    @VisibleForTesting
    final static int ERROR_LIMIT_PER_REPORTING_PERIOD = 20;
    @VisibleForTesting
    final static String STRIPPED_EXCEPTION_REPLACEMENT = "Message removed by New Relic 'strip_exception_messages' setting";
    @VisibleForTesting
    final static Set<String> IGNORE_ERRORS;

    static {
        Set<String> ignoreErrors = new HashSet<>(4);
        ignoreErrors.add("org.eclipse.jetty.continuation.ContinuationThrowable");
        ignoreErrors.add("org.mortbay.jetty.RetryRequest");
        IGNORE_ERRORS = Collections.unmodifiableSet(ignoreErrors);
    }

    @VisibleForTesting
    final AtomicInteger errorCountThisHarvest = new AtomicInteger();
    final AtomicInteger expectedErrorCountThisHarvest = new AtomicInteger();
    private final AtomicInteger errorCount = new AtomicInteger();
    private final AtomicLong totalErrorCount = new AtomicLong();
    private final AtomicReferenceArray<TracedError> tracedErrors;
    private final ConcurrentHashMap<String, DistributedSamplingPriorityQueue<ErrorEvent>> reservoirForApp = new ConcurrentHashMap<>();
    private final int maxUserParameterSize;
    private volatile ErrorCollectorConfig errorCollectorConfig;
    private volatile StripExceptionConfig stripExceptionConfig;
    private final boolean shouldRecordErrorCount;
    protected volatile int maxEventsStored;
    private final String appName;

    protected Harvestable harvestable;

    /**
     * Note that an instance of this class is created for each RPMService, which is a side effect of using
     * auto app naming and the servlet instrumentation.
     */
    public ErrorServiceImpl(String appName) {
        this.appName = appName;
        errorCollectorConfig = ServiceFactory.getConfigService().getErrorCollectorConfig(appName);
        stripExceptionConfig = ServiceFactory.getConfigService().getStripExceptionConfig(appName);
        tracedErrors = new AtomicReferenceArray<>(ERROR_LIMIT_PER_REPORTING_PERIOD);
        ServiceFactory.getTransactionService().addTransactionListener(new MyTransactionListener());
        ServiceFactory.getConfigService().addIAgentConfigListener(new MyConfigListener());
        shouldRecordErrorCount = !Boolean.getBoolean("com.newrelic.agent.errors.no_error_metric");
        maxEventsStored = errorCollectorConfig.getMaxEventsStored();
        maxUserParameterSize = ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameterSize();
    }

    @Override
    public void start() {
    }

    @Override
    public void stop() {
        ServiceFactory.getHarvestService().removeHarvestable(harvestable);
    }

    @Override
    public void addHarvestableToService() {
        Harvestable harvestableToAdd = new ErrorHarvestableImpl(this, appName);
        ServiceFactory.getHarvestService().addHarvestable(harvestableToAdd);
        harvestable = harvestableToAdd;
    }

    public ErrorCollectorConfig getErrorCollectorConfig() {
        return errorCollectorConfig;
    }

    public void setMaxEventsStored(int newMax) {
        maxEventsStored = newMax;
    }

    public void clearReservoir() {
        reservoirForApp.clear();
    }

    public void clearReservoir(String appName) {
        DistributedSamplingPriorityQueue<ErrorEvent> reservoir = reservoirForApp.get(appName);
        if (reservoir != null) {
            reservoir.clear();
        }
    }

    @VisibleForTesting
    public void setHarvestable(Harvestable harvestable) {
        this.harvestable = harvestable;
    }

    public void harvestEvents(final String appName) {
        boolean eventsEnabled = isEventsEnabledForApp(appName);
        if (!eventsEnabled) {
            reservoirForApp.remove(appName);
            return;
        }
        if (maxEventsStored <= 0) {
            clearReservoir(appName);
            return;
        }

        long startTimeInNanos = System.nanoTime();

        final DistributedSamplingPriorityQueue<ErrorEvent> reservoir = reservoirForApp.put(appName,
                new DistributedSamplingPriorityQueue<ErrorEvent>(appName, "Error Service", maxEventsStored));

        if (reservoir != null && reservoir.size() > 0) {
            try {
                ServiceFactory.getRPMService(appName).sendErrorEvents(maxEventsStored, reservoir.getNumberOfTries(),
                        Collections.unmodifiableList(reservoir.asList()));
                final long durationInNanos = System.nanoTime() - startTimeInNanos;
                ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
                    @Override
                    public void doWork(StatsEngine statsEngine) {
                        recordSupportabilityMetrics(statsEngine, durationInNanos, reservoir);
                    }

                    @Override
                    public String getAppName() {
                        return appName;
                    }
                });

                if (reservoir.size() < reservoir.getNumberOfTries()) {
                    int dropped = reservoir.getNumberOfTries() - reservoir.size();
                    Agent.LOG.log(Level.WARNING, "Dropped {0} error events out of {1}.", dropped, reservoir.getNumberOfTries());
                }
            } catch (HttpError e) {
                if (!e.discardHarvestData()) {
                    Agent.LOG.log(Level.FINE, "Unable to send error events. Unsent events will be included in the next harvest.", e);
                    // Save unsent data by merging it with current data using reservoir algorithm
                    DistributedSamplingPriorityQueue<ErrorEvent> currentReservoir = reservoirForApp.get(appName);
                    currentReservoir.retryAll(reservoir);
                } else {
                    // discard harvest data
                    reservoir.clear();
                    Agent.LOG.log(Level.FINE, "Unable to send error events. Unsent events will be dropped.", e);
                }
            } catch (Exception e) {
                // discard harvest data
                reservoir.clear();
                Agent.LOG.log(Level.FINE, "Unable to send error events. Unsent events will be dropped.", e);
            }
        }
    }

    private void recordSupportabilityMetrics(StatsEngine statsEngine, long durationInNanos,
            DistributedSamplingPriorityQueue<ErrorEvent> reservoir) {
        statsEngine.getStats(MetricNames.SUPPORTABILITY_ERROR_SERVICE_TRANSACTION_ERROR_SENT)
                .incrementCallCount(reservoir.size());
        statsEngine.getStats(MetricNames.SUPPORTABILITY_ERROR_SERVICE_TRANSACTION_ERROR_SEEN)
                .incrementCallCount(reservoir.getNumberOfTries());
        statsEngine.getResponseTimeStats(MetricNames.SUPPORTABILITY_ERROR_SERVICE_EVENT_HARVEST_TRANSMIT)
                .recordResponseTime(durationInNanos, TimeUnit.NANOSECONDS);
    }

    // Called from beforeHarvest listener. Send traced errors for this app upstream.
    @VisibleForTesting
    public void harvestTracedErrors(String appName) {
        List<TracedError> tracedErrorList = getAndClearTracedErrors(appName);
        if (tracedErrorList != null && tracedErrorList.size() > 0) {
            try {
                ServiceFactory.getRPMService(appName).sendErrorData(tracedErrorList);
            } catch (Exception ex) {
                // TODO recover from failure to send here - see JAVA-2949.
                Agent.LOG.fine("Unable to send error events.");
            }
        }
    }

    private boolean isEnabledForApp(String appName) {
        return ServiceFactory.getConfigService().getErrorCollectorConfig(appName).isEnabled();
    }

    private boolean isEventsEnabledForApp(String appName) {
        return ServiceFactory.getConfigService().getErrorCollectorConfig(appName).isEventsEnabled();
    }

    @VisibleForTesting
    public void refreshErrorCollectorConfig(AgentConfig agentConfig) {
        ErrorCollectorConfig oldErrorConfig = errorCollectorConfig;
        errorCollectorConfig = agentConfig.getErrorCollectorConfig();
        if (errorCollectorConfig.isEnabled() == oldErrorConfig.isEnabled()) {
            return;
        }
        String msg = MessageFormat.format("Errors will{0} be sent to New Relic for {1}", errorCollectorConfig.isEnabled() ? "" : " not", appName);
        Agent.LOG.info(msg);
    }

    @VisibleForTesting
    void refreshStripExceptionConfig(AgentConfig agentConfig) {
        StripExceptionConfig oldStripExceptionConfig = stripExceptionConfig;
        stripExceptionConfig = agentConfig.getStripExceptionConfig();
        if (stripExceptionConfig.isEnabled() != oldStripExceptionConfig.isEnabled()) {
            Agent.LOG.info(MessageFormat.format(
                    "Exception messages will{0} be stripped before sending to New Relic for {1}",
                    stripExceptionConfig.isEnabled() ? "" : " not", appName));
        }
        if (!stripExceptionConfig.getWhitelist().equals(oldStripExceptionConfig.getWhitelist())) {
            Agent.LOG.info(MessageFormat.format("Exception message whitelist updated to {0} for {1}",
                    stripExceptionConfig.getWhitelist().toString(), appName));
        }
    }

    @Override
    public void reportError(TracedError error) {
        // This is called from a static method in this class but through the ServiceFactory,
        // so it really needs to be public and on the interface even though IJ says it doesn't.
        reportError(error, null, null);
    }

    @Override
    public void reportErrors(TracedError... errors) {
        for (TracedError error : errors) {
            reportError(error);
        }
    }

    @VisibleForTesting // Introspector subclasses this class
    protected void reportError(TracedError error, TransactionData transactionData, TransactionStats transactionStats) {
        if (error == null) {
            return;
        }
        if (error instanceof ThrowableError && isIgnoredError(HttpStatus.SC_OK, ((ThrowableError) error).getThrowable())) {
            if (Agent.LOG.isLoggable(Level.FINER)) {
                Throwable throwable = ((ThrowableError) error).getThrowable();
                String errorString = throwable == null ? "" : throwable.getClass().getName();
                String msg = MessageFormat.format("Ignoring error {0} for {1}", errorString, appName);
                Agent.LOG.finer(msg);
            }
            return;
        }
        if (error.incrementsErrorMetric()) {
            errorCountThisHarvest.incrementAndGet();
        } else if (!(error instanceof DeadlockTraceError)) {
            expectedErrorCountThisHarvest.incrementAndGet();
        }
        if (!errorCollectorConfig.isEnabled() || !isEventsEnabledForApp(appName) || maxEventsStored <= 0) {
            return;
        }

        // Siphon off errors to send up as error events
        DistributedSamplingPriorityQueue<ErrorEvent> eventList = getReservoir(appName);

        ErrorEvent errorEvent = createErrorEvent(appName, error, transactionData, transactionStats);
        eventList.add(errorEvent);

        if (errorCount.get() >= ERROR_LIMIT_PER_REPORTING_PERIOD) {
            Agent.LOG.finer(MessageFormat.format("Error limit exceeded for {0}: {1}", appName, error));
            return;
        }
        int index = (int) totalErrorCount.getAndIncrement() % ERROR_LIMIT_PER_REPORTING_PERIOD;
        if (tracedErrors.compareAndSet(index, null, error)) {
            errorCount.getAndIncrement();
            if (Agent.LOG.isLoggable(Level.FINER)) {
                Agent.LOG.finer(MessageFormat.format("Recording error for {0} : {1}", appName, error));
            }
        }
    }

    @VisibleForTesting // Introspector subclasses this class
    protected static ErrorEvent createErrorEvent(final String theAppName, TracedError error,
            TransactionData transactionData, TransactionStats transactionStats) {
        ErrorEvent errorEvent;
        if (transactionData != null) {
            errorEvent = new ErrorEvent(theAppName, error, transactionData, transactionStats);
        } else {
            errorEvent = new ErrorEvent(theAppName, error);
        }
        return errorEvent;
    }

    /**
     * Get the traced errors for this reporting period and increment the error stats.
     *
     * @param appName the name of the app we're harvesting for.
     * @return the traced errors for this reporting period.
     */
    @VisibleForTesting
    public List<TracedError> getAndClearTracedErrors(String appName) {
        // This is the equivalent of the method called harvest() through v3.37.0 of the Agent.
        // The behaviors are very old and very specific. They are rigidly checked by tests.
        recordMetrics(appName);
        if (ServiceFactory.getRPMService(appName).isConnected()) {
            return getAndClearTracedErrors();
        }

        return Collections.emptyList();
    }

    /**
     * Get the traced errors for this reporting period.
     *
     * @return the traced errors for this reporting period.
     */
    @VisibleForTesting
    public List<TracedError> getAndClearTracedErrors() {
        List<TracedError> errors = new ArrayList<>(ERROR_LIMIT_PER_REPORTING_PERIOD);
        for (int i = 0; i < tracedErrors.length(); i++) {
            TracedError error = tracedErrors.getAndSet(i, null);
            if (error != null) {
                errorCount.getAndDecrement();
                errors.add(error);
            }
        }
        return errors;
    }

    @VisibleForTesting
    public int getTracedErrorsCount() {
        return errorCount.get();
    }

    private void recordMetrics(final String appName) {
        if (shouldRecordErrorCount) {
            ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
                @Override
                public void doWork(StatsEngine statsEngine) {
                    int errorCount = errorCountThisHarvest.getAndSet(0);
                    statsEngine.getStats(MetricNames.ERRORS_ALL).incrementCallCount(errorCount);
                    int expectedErrorCount = expectedErrorCountThisHarvest.getAndSet(0);
                    statsEngine.getStats(MetricNames.ERRORS_EXPECTED_ALL).incrementCallCount(expectedErrorCount);
                }

                @Override
                public String getAppName() {
                    return appName;
                }
            });
        }
    }

    /**
     * Check if the transaction has an error to report.
     */
    private void noticeTransaction(TransactionData td, TransactionStats transactionStats) {
        if (!appName.equals(td.getApplicationName())) {
            return;
        }
        if (!isEnabledForApp(td.getApplicationName())) {
            return;
        }

        String statusMessage = td.getStatusMessage();
        int responseStatus = td.getResponseStatus();
        Throwable throwable = td.getThrowable() == null ? null : td.getThrowable().throwable;
        final boolean isReportable = responseStatus >= HttpURLConnection.HTTP_BAD_REQUEST || throwable != null;
        if (throwable instanceof ReportableError) {
            // Someone manually called NewRelic.noticeError(String message) here
            // so we don't want to use getStrippedExceptionMessage() for replacement
            statusMessage = throwable.getMessage();
            throwable = null;
        }

        if (isReportable) {
            if (!td.hasReportableErrorThatIsNotIgnored()) {
                if (Agent.LOG.isLoggable(Level.FINER)) {
                    String errorString = throwable == null ? "" : throwable.getClass().getName();
                    String msg = MessageFormat.format("Ignoring error {0} for {1} {2} ({3})", errorString,
                            td.getRequestUri(AgentConfigImpl.ERROR_COLLECTOR), appName, responseStatus);
                    Agent.LOG.finer(msg);
                }
                return;
            }

            TracedError error = createTracedError(appName, td, throwable, responseStatus, statusMessage);
            if (shouldRecordErrorCount && error.incrementsErrorMetric()) {
                recordErrorCount(td, transactionStats);
            }

            reportError(error, td, transactionStats);
        }
    }

    private TracedError createTracedError(final String theAppName, TransactionData td, Throwable throwable, int responseStatus, String statusMessage) {
        TracedError error;
        // noticeError(expected = true)?
        boolean markedExpected = td.getThrowable() == null ? false : td.getThrowable().expected;

        Map<String, Object> joinedIntrinsics = new HashMap<>(td.getIntrinsicAttributes());
        DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();
        DistributedTracingConfig distributedTracingConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig();

        if (distributedTracingConfig.isEnabled()) {
            joinedIntrinsics.putAll(distributedTraceService.getIntrinsics(
                    td.getInboundDistributedTracePayload(), td.getGuid(), td.getTripId(), td.getTransportType(),
                    td.getTransportDurationInMillis(), td.getLargestTransportDurationInMillis(), td.getParentId(),
                    td.getParentSpanId(), td.getPriority()));
        }

        if (throwable != null) {
            error = ThrowableError
                    .builder(errorCollectorConfig, theAppName, td.getBlameOrRootMetricName(), throwable, td.getWallClockStartTimeMs())
                    .requestUri(td.getRequestUri(AgentConfigImpl.ERROR_COLLECTOR))
                    .prefixedAttributes(td.getPrefixedAttributes())
                    .userAttributes(td.getUserAttributes())
                    .agentAttributes(td.getAgentAttributes())
                    .errorAttributes(td.getErrorAttributes())
                    .intrinsicAttributes(joinedIntrinsics)
                    .expected(markedExpected)
                    .build();
        } else {
            error = HttpTracedError
                    .builder(errorCollectorConfig, maxUserParameterSize, theAppName, td.getBlameOrRootMetricName(), td.getWallClockStartTimeMs())
                    .statusCodeAndMessage(responseStatus, statusMessage)
                    .transactionData(td)
                    .requestUri(td.getRequestUri(AgentConfigImpl.ERROR_COLLECTOR))
                    .prefixedAttributes(td.getPrefixedAttributes())
                    .userAttributes(td.getUserAttributes())
                    .agentAttributes(td.getAgentAttributes())
                    .errorAttributes(td.getErrorAttributes())
                    .intrinsicAttributes(joinedIntrinsics)
                    .expected(markedExpected)
                    .build();
        }
        return error;
    }

    private void recordErrorCount(TransactionData td, TransactionStats transactionStats) {
        String metricName = getErrorCountMetricName(td);
        if (metricName != null) {
            transactionStats.getUnscopedStats().getStats(metricName).incrementCallCount();
        }
        String metricNameAll = td.isWebTransaction() ? MetricNames.WEB_TRANSACTION_ERRORS_ALL
                : MetricNames.OTHER_TRANSACTION_ERRORS_ALL;
        transactionStats.getUnscopedStats().getStats(metricNameAll).incrementCallCount();
    }

    private String getErrorCountMetricName(TransactionData td) {
        String blameMetricName = td.getBlameMetricName();
        if (blameMetricName != null) {
            StringBuilder output = new StringBuilder(MetricNames.ERRORS_SLASH.length() + blameMetricName.length());
            output.append(MetricNames.ERRORS_SLASH);
            output.append(blameMetricName);
            return output.toString();
        }
        return null;
    }

    @VisibleForTesting
    public DistributedSamplingPriorityQueue<ErrorEvent> getReservoir(String appName) {
        DistributedSamplingPriorityQueue<ErrorEvent> result = reservoirForApp.get(appName);
        while (result == null) {
            // I don't think this loop can actually execute more than once, but it's prudent to assume it can.
            reservoirForApp.putIfAbsent(appName, new DistributedSamplingPriorityQueue<ErrorEvent>(appName, "Error Service", maxEventsStored));
            result = reservoirForApp.get(appName);
        }
        return result;
    }

    @Override
    public boolean isIgnoredError(int responseStatus, Throwable throwable) {
        boolean responseIgnored = false;
        if (responseStatus != 0) {
            responseIgnored = errorCollectorConfig.getIgnoreStatusCodes().contains(responseStatus);
        }

        boolean throwableIgnored = isIgnoredThrowable(throwable);
        return responseIgnored || throwableIgnored;
    }

    @Override
    public boolean isExpectedError(int responseStatus, TransactionThrowable transactionThrowable) {
        if (transactionThrowable != null && transactionThrowable.expected) {
            return true;
        }

        boolean responseExpected = errorCollectorConfig.getExpectedStatusCodes().contains(responseStatus);

        boolean throwableExpected = false;
        Throwable throwable = transactionThrowable == null ? null : transactionThrowable.throwable;
        while (throwable != null) {
            String exceptionClass = throwable.getClass().getName();

            Set<ExpectedErrorConfig> expectedErrors = errorCollectorConfig.getExpectedErrors();
            for (ExpectedErrorConfig expectedError : expectedErrors) {
                String expectedErrorClass = expectedError.getErrorClass();
                if (exceptionClass != null && exceptionClass.equals(expectedErrorClass)) {
                    String expectedErrorMessage = expectedError.getErrorMessage();
                    if (expectedErrorMessage != null) {
                        if (insecureGetMessageNotNull(throwable).contains(expectedErrorMessage)) {
                            throwableExpected = true;
                            break;
                        }
                    } else {
                        throwableExpected = true;
                        break;
                    }
                }
            }

            throwable = throwable.getCause();
        }
        return responseExpected || throwableExpected;
    }

    private boolean isIgnoredThrowable(Throwable throwable) {
        while (throwable != null) {
            String exceptionClass = throwable.getClass().getName();

            if (IGNORE_ERRORS.contains(exceptionClass)) {
                return true;
            }

            Set<IgnoreErrorConfig> ignoreErrors = errorCollectorConfig.getIgnoreErrors();
            for (IgnoreErrorConfig ignoreError : ignoreErrors) {
                String ignoreErrorClass = ignoreError.getErrorClass();
                if (exceptionClass != null && exceptionClass.equals(ignoreErrorClass)) {
                    String ignoreErrorMessage = ignoreError.getErrorMessage();
                    if (ignoreErrorMessage != null) {
                        if (insecureGetMessageNotNull(throwable).contains(ignoreErrorMessage)) {
                            return true;
                        }
                    } else {
                        return true;
                    }
                }
            }

            throwable = throwable.getCause();
        }

        return false;
    }

    // Get the actual exception message, not "stripped" regardless of strip_exception_messages in the config.
    // The return value of this method MUST NOT be used for anything except matching expected or ignored errors.
    private String insecureGetMessageNotNull(Throwable throwable) {
        String message = (throwable == null) ? null : throwable.getMessage();
        return (message == null) ? "" : message;
    }

    /**
     * Report an exception to New Relic. Implements the public API.
     */
    public static void reportException(Throwable throwable, Map<String, String> params, boolean expected) {
        Transaction tx = Transaction.getTransaction(false);
        if (tx != null && tx.isInProgress()) {
            if (params != null) {
                tx.getErrorAttributes().putAll(params);
            }
            synchronized (tx) {
                tx.setThrowable(throwable, TransactionErrorPriority.API, expected);
            }
        } else {
            ErrorCollectorConfig errorCollectorConfig =
                    ServiceFactory.getConfigService().getDefaultAgentConfig().getErrorCollectorConfig();

            // we're not within a transaction. just report the error
            TracedError error = ThrowableError
                    .builder(errorCollectorConfig, null, "Unknown", throwable, System.currentTimeMillis())
                    .errorAttributes(params)
                    .expected(expected)
                    .build();
            ServiceFactory.getRPMService().getErrorService().reportError(error);
        }
    }

    public static void reportError(String message, Map<String, String> params, boolean expected) {
        Transaction tx = Transaction.getTransaction(false);
        if (tx != null && tx.isInProgress()) {
            if (params != null) {
                tx.getErrorAttributes().putAll(params);
            }
            synchronized (tx) {
                tx.setThrowable(new ReportableError(message), TransactionErrorPriority.API, expected);
            }
        } else {
            ErrorCollectorConfig errorCollectorConfig =
                    ServiceFactory.getConfigService().getDefaultAgentConfig().getErrorCollectorConfig();
            int maxUserParameterSize = ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameterSize();

            // we're not within a transaction. just report the error
            TracedError error = HttpTracedError
                    .builder(errorCollectorConfig, maxUserParameterSize, null, "Unknown", System.currentTimeMillis())
                    .message(message)
                    .errorAttributes(params)
                    .expected(expected)
                    .build();
            ServiceFactory.getRPMService().getErrorService().reportError(error);
        }
    }

    public static class ReportableError extends Throwable {
        private static final long serialVersionUID = 3472056044517410355L;

        public ReportableError(String message) {
            super(message);
        }
    }

    private class MyTransactionListener implements TransactionListener {
        @Override
        public void dispatcherTransactionFinished(TransactionData transactionData, TransactionStats transactionStats) {
            noticeTransaction(transactionData, transactionStats);
        }
    }

    private class MyConfigListener implements AgentConfigListener {
        @Override
        public void configChanged(String appName, AgentConfig agentConfig) {
            if (ErrorServiceImpl.this.appName.equals(appName)) {
                Agent.LOG.fine(MessageFormat.format("Error service received configuration change notification for {0}", appName));
                refreshErrorCollectorConfig(agentConfig);
                refreshStripExceptionConfig(agentConfig);
            }
        }
    }

    public static Collection<? extends PointCut> getEnabledErrorHandlerPointCuts() {
        AgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();
        Object exceptionHandlers = config.getErrorCollectorConfig().getProperty("exception_handlers");
        if (exceptionHandlers == null) {
            return Collections.emptyList();
        }

        Collection<PointCut> pointcuts = new ArrayList<>();
        if (exceptionHandlers instanceof Collection<?>) {
            for (Object sigObject : ((Collection<?>) exceptionHandlers)) {
                if (sigObject instanceof ExceptionHandlerSignature) {
                    ExceptionHandlerSignature exHandlerSig = (ExceptionHandlerSignature) sigObject;

                    String msg = MessageFormat.format("Instrumenting exception handler signature {0}",
                            exHandlerSig.toString());
                    Agent.LOG.finer(msg);
                    ExceptionHandlerPointCut pc = new ExceptionHandlerPointCut(exHandlerSig);
                    if (pc.isEnabled()) {
                        pointcuts.add(pc);
                    }
                } else if (sigObject instanceof String) {
                    ClassMethodSignature signature = PointCutFactory.parseClassMethodSignature(sigObject.toString());

                    try {
                        ExceptionHandlerSignature exHandlerSig = new ExceptionHandlerSignature(signature);
                        Agent.LOG.info(MessageFormat.format("Instrumenting exception handler signature {0}",
                                exHandlerSig.toString()));
                        ExceptionHandlerPointCut pc = new ExceptionHandlerPointCut(exHandlerSig);
                        if (pc.isEnabled()) {
                            pointcuts.add(pc);
                        }
                    } catch (InvalidMethodDescriptor e) {
                        Agent.LOG.severe(MessageFormat.format("Unable to instrument exception handler {0} : {1}",
                                sigObject.toString(), e.toString()));
                    }
                } else if (sigObject instanceof Exception) {
                    Agent.LOG.severe(MessageFormat.format("Unable to instrument exception handler : {0}",
                            sigObject.toString()));
                }
            }
        }
        return pointcuts;
    }

    @Override
    public void reportHTTPError(String message, int statusCode, String uri) {
        if (!isIgnoredError(statusCode, null)) {
            TracedError error = HttpTracedError
                    .builder(errorCollectorConfig, maxUserParameterSize, null, "WebTransaction" + uri, System.currentTimeMillis())
                    .statusCodeAndMessage(statusCode, message)
                    .requestUri(uri)
                    .build();
            reportError(error);
            Agent.LOG.finer(MessageFormat.format("Reported HTTP error {0} with status code {1} URI {2}", message, statusCode, uri));
        } else {
            Agent.LOG.finer(MessageFormat.format("Ignoring HTTP error {0} with status code {1} URI {2}", message, statusCode, uri));
        }
    }

    /**
     * Checks if exception should be ignored.
     */
    @Override
    public void reportException(Throwable throwable) {
        if (!isIgnoredThrowable(throwable)) {
            reportException(throwable, Collections.<String, String>emptyMap(), false);
        } else {
            Agent.LOG.finer(MessageFormat.format("Ignoring error with throwable {0} ", throwable));
        }
    }

    @Override
    public boolean useStrippedExceptionReplacement(Throwable throwable) {
        return (stripExceptionConfig.isEnabled() && !stripExceptionConfig.getWhitelist().contains(throwable.getClass().getName()));
    }

    public static String getStrippedExceptionMessage(Throwable throwable) {
        ErrorService errorService = ServiceFactory.getRPMService().getErrorService();
        if (errorService.useStrippedExceptionReplacement(throwable)) {
            return STRIPPED_EXCEPTION_REPLACEMENT;
        }
        return throwable.getMessage();
    }

}
