package com.newrelic.agent.errors;

import com.newrelic.agent.Harvestable;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.analytics.InsightsService;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsWork;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.transport.DataSenderImpl;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class ErrorHarvestableImpl implements Harvestable {

    private ErrorServiceImpl errorService;
    private String appName;
    private long lastHarvest;

    ErrorHarvestableImpl(ErrorServiceImpl errorService, String appName) {
        this.errorService = errorService;
        this.appName = appName;
        this.lastHarvest = System.nanoTime();
    }

    @Override
    public String getEndpointMethodName() {
        return DataSenderImpl.ERROR_EVENT_DATA_METHOD;
    }

    @Override
    public void harvest() {
        recordIntervalMetric();
        errorService.harvestEvents(appName);
        errorService.harvestTracedErrors(appName);
    }

    private void recordIntervalMetric() {
        long startTimeInNanos = System.nanoTime();
        final long harvestIntervalInNanos = startTimeInNanos - lastHarvest;
        lastHarvest = startTimeInNanos;

        ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
            @Override
            public void doWork(StatsEngine statsEngine) {
                if (harvestIntervalInNanos > 0) {
                    statsEngine.getResponseTimeStats(
                            MetricNames.SUPPORTABILITY_ERROR_SERVICE_EVENT_HARVEST_INTERVAL).recordResponseTime(
                            harvestIntervalInNanos, TimeUnit.NANOSECONDS);
                }
            }

            @Override
            public String getAppName() {
                return appName;
            }
        });
    }

    @Override
    public String getAppName() {
        return appName;
    }

    public void setAppName(String appName) {
        this.appName = appName;
    }

    @Override
    public void configure(Map<String, Object> configuration) {
        /*
         * There are two ways to configure the max number of error events we store. One is
         * error_collector.max_event_samples_stored, the other one is max_samples_stored that may come down on
         * connect in the data_methods.error_event_data block. We grab the min of these two values.
         *
         * See https://source.datanerd.us/agents/agent-specs/blob/master/Error-Events.md and
         * https://source.datanerd.us/agents/agent-specs/blob/master/Connect-LEGACY.md#data-methods
         */
        Number maxSamplesCollector = (Number) configuration.get(InsightsService.MAX_SAMPLES_STORED_KEY_NAME);
        int maxSamples = maxSamplesCollector != null ?
                Math.min(maxSamplesCollector.intValue(), errorService.getErrorCollectorConfig().getMaxEventsStored()) :
                errorService.getErrorCollectorConfig().getMaxEventsStored();

        Number reportPeriod = (Number) configuration.get(Harvestable.REPORT_PERIOD_IN_SECONDS_KEY_NAME);

        ServiceFactory.getStatsService().doStatsWork(
                StatsWorks.getRecordMetricWork(MetricNames.SUPPORTABILITY_ERROR_SERVICE_REPORT_PERIOD_IN_SECONDS, reportPeriod.longValue()));

        int maxEventsStored = errorService.maxEventsStored;
        if (maxSamples != maxEventsStored) {
            errorService.setMaxEventsStored(maxSamples);
            errorService.harvestEvents(appName);
            errorService.clearReservoir();
        }
    }

}
