package com.newrelic.agent.config;

import com.google.common.collect.Maps;
import com.newrelic.agent.Agent;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;

public class SystemPropertyProvider {

    // environment variables
    private static final String NEW_RELIC_PREFIX_ENV = "NEW_RELIC_";
    private static final String LICENSE_KEY_ENV = NEW_RELIC_PREFIX_ENV + "LICENSE_KEY";
    private static final String APP_NAME_ENV = NEW_RELIC_PREFIX_ENV + "APP_NAME";
    private static final String LOG_ENV = NEW_RELIC_PREFIX_ENV + "LOG";
    private static final String HOST_ENV = NEW_RELIC_PREFIX_ENV + "HOST";
    private static final String HOST_DISPLAY_NAME_ENV = NEW_RELIC_PREFIX_ENV + "PROCESS_HOST_DISPLAY_NAME";
    private static final String SECURITY_POLICIES_TOKEN_ENV = NEW_RELIC_PREFIX_ENV + "SECURITY_POLICIES_TOKEN";
    // system properties
    private static final String LICENSE_KEY = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.LICENSE_KEY;
    private static final String SECURITY_POLICIES_TOKEN = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.LASP_TOKEN;
    private static final String APP_NAME = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.APP_NAME;
    private static final String LOG_FILE_NAME = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.LOG_FILE_NAME;
    private static final String HOST = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + AgentConfigImpl.HOST;
    private static final String HOST_DISPLAY_NAME = AgentConfigImpl.SYSTEM_PROPERTY_ROOT + "process_host.display_name";
    private static final String NEW_RELIC_SYSTEM_PROPERTY_ROOT = "newrelic.";

    private final Map<String, String> envVars;
    private final Map<String, String> envVarsFlattenedMapping;
    private final Map<String, String> newRelicSystemProps;
    private final Map<String, Object> newRelicPropsWithoutPrefix;
    private final SystemProps systemProps;

    public SystemPropertyProvider() {
        this(SystemProps.getSystemProps());
    }

    public SystemPropertyProvider(SystemProps sysProps) {
        systemProps = sysProps;
        envVars = initEnvVariables();
        envVarsFlattenedMapping = initFlattenedEnvVariables();
        newRelicSystemProps = initNewRelicSystemProperties();
        newRelicPropsWithoutPrefix = createNewRelicSystemPropertiesWithoutPrefix();
    }

    private Map<String, String> initEnvVariables() {
        // general environment variables, originally added for Heroku
        Map<String, String> envVars = new HashMap<String, String>(6);
        envVars.put(LICENSE_KEY, getenv(LICENSE_KEY_ENV));
        envVars.put(APP_NAME, getenv(APP_NAME_ENV));
        envVars.put(LOG_FILE_NAME, getenv(LOG_ENV));
        envVars.put(HOST, getenv(HOST_ENV));
        envVars.put(HOST_DISPLAY_NAME, getenv(HOST_DISPLAY_NAME_ENV));
        // LASP environment variable
        envVars.put(SECURITY_POLICIES_TOKEN, getenv(SECURITY_POLICIES_TOKEN_ENV));
        // distributed tracing environment variables
        envVars.put(DistributedTracingConfig.DISTRIBUTED_TRACING_ENABLED, getenv(DistributedTracingConfig.ENABLED_ENV_KEY));
        envVars.put(SpanEventsConfig.SYSTEM_PROPERTY_SPAN_EVENTS_ENABLED, getenv(SpanEventsConfig.ENABLED_ENV_KEY));
        return envVars;
    }

    private Map<String, String> initFlattenedEnvVariables() {
        // general environment variables, originally added for Heroku
        Map<String, String> envVars = new HashMap<String, String>(6);
        envVars.put(LICENSE_KEY_ENV, LICENSE_KEY);
        envVars.put(APP_NAME_ENV, APP_NAME);
        envVars.put(LOG_ENV, LOG_FILE_NAME);
        envVars.put(HOST_ENV, HOST);
        envVars.put(HOST_DISPLAY_NAME_ENV, HOST_DISPLAY_NAME);
        // LASP environment variable
        envVars.put(SECURITY_POLICIES_TOKEN_ENV, SECURITY_POLICIES_TOKEN);
        // distributed tracing environment variables
        envVars.put(DistributedTracingConfig.ENABLED_ENV_KEY, DistributedTracingConfig.DISTRIBUTED_TRACING_ENABLED);
        envVars.put(SpanEventsConfig.ENABLED_ENV_KEY, SpanEventsConfig.SYSTEM_PROPERTY_SPAN_EVENTS_ENABLED);
        return envVars;
    }

    /**
     * Get the New Relic system properties.
     */
    private Map<String, String> initNewRelicSystemProperties() {
        Map<String, String> nrProps = Maps.newHashMap();
        try {
            for (Entry<Object, Object> entry : systemProps.getAllSystemProperties().entrySet()) {
                String key = entry.getKey().toString();
                if (key.startsWith(NEW_RELIC_SYSTEM_PROPERTY_ROOT)) {
                    String val = entry.getValue().toString();
                    nrProps.put(key, val);
                }
            }
        } catch (SecurityException t) {
            Agent.LOG.log(Level.FINE, "Unable to get system properties");
        }
        return Collections.unmodifiableMap(nrProps);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private Map<String, Object> createNewRelicSystemPropertiesWithoutPrefix() {
        Map<String, Object> nrProps = Maps.newHashMap();
        addNewRelicSystemProperties(nrProps, ((Map) systemProps.getAllSystemProperties()).entrySet());
        addNewRelicEnvProperties(nrProps, ((Map) systemProps.getAllEnvProperties()).entrySet());

        return Collections.unmodifiableMap(nrProps);
    }

    private void addNewRelicSystemProperties(Map<String, Object> nrProps,
            @SuppressWarnings("rawtypes") Set<Entry> entrySet) {
        for (Entry<?, ?> entry : entrySet) {
            String key = entry.getKey().toString();
            if (key.startsWith(AgentConfigImpl.SYSTEM_PROPERTY_ROOT)) {
                addPropertyWithoutSystemPropRoot(nrProps, key, entry.getValue());
            }
        }
    }

    private void addNewRelicEnvProperties(Map<String, Object> nrProps, @SuppressWarnings("rawtypes") Set<Entry> entrySet) {
        for (Entry<?, ?> entry : entrySet) {
            String key = entry.getKey().toString();
            if (key.startsWith(AgentConfigImpl.SYSTEM_PROPERTY_ROOT)) {
                addPropertyWithoutSystemPropRoot(nrProps, key, entry.getValue());
            } else {
                String keyToUse = envVarsFlattenedMapping.get(key);
                if (keyToUse != null) {
                    addPropertyWithoutSystemPropRoot(nrProps, keyToUse, entry.getValue());
                }
            }
        }
    }

    private void addPropertyWithoutSystemPropRoot(Map<String, Object> nrProps, String key, Object value) {
        String val = value.toString();
        key = key.substring(AgentConfigImpl.SYSTEM_PROPERTY_ROOT.length());
        nrProps.put(key, val);
    }

    public String getEnvironmentVariable(String prop) {
        String val = envVars.get(prop);
        if (val != null) {
            return val;
        }
        return getenv(prop);
    }

    public String getSystemProperty(String prop) {
        return systemProps.getSystemProperty(prop);
    }

    private String getenv(String key) {
        return systemProps.getenv(key);
    }

    /**
     * Get a map of the New Relic system properties (any property starting with newrelic.)
     */
    public Map<String, String> getNewRelicSystemProperties() {
        return newRelicSystemProps;
    }

    /**
     * Returns the New Relic system and environment properties with the 'newrelic.config.' prefixes removed.
     *
     * @return
     */
    public Map<String, Object> getNewRelicPropertiesWithoutPrefix() {
        return newRelicPropsWithoutPrefix;
    }

    protected abstract static class SystemProps {

        static SystemProps getSystemProps() {
            try {
                System.getProperties().get("test");
                System.getenv("test");

                return new SystemProps() {

                    @Override
                    String getSystemProperty(String prop) {
                        return System.getProperty(prop);
                    }

                    @Override
                    String getenv(String key) {
                        return System.getenv(key);
                    }

                    @Override
                    Properties getAllSystemProperties() {
                        return System.getProperties();
                    }

                    @Override
                    Map<String, String> getAllEnvProperties() {
                        return System.getenv();
                    }
                };

            } catch (SecurityException e) {
                Agent.LOG.error("Unable to access system properties because of a security exception.");
                return new SystemProps() {

                    @Override
                    String getSystemProperty(String prop) {
                        return null;
                    }

                    @Override
                    String getenv(String key) {
                        return null;
                    }

                    @Override
                    Properties getAllSystemProperties() {
                        return new Properties();
                    }

                    @Override
                    Map<String, String> getAllEnvProperties() {
                        return Collections.emptyMap();
                    }
                };
            }
        }

        abstract String getSystemProperty(String prop);

        abstract String getenv(String key);

        abstract Properties getAllSystemProperties();

        abstract Map<String, String> getAllEnvProperties();
    }
}
