package com.newrelic.agent.config;

import com.google.common.annotations.VisibleForTesting;
import com.newrelic.agent.Agent;
import com.newrelic.agent.ForceDisconnectException;

import java.io.File;
import java.text.MessageFormat;
import java.util.Map;

public class ConfigServiceFactory {

    /**
     * For testing.
     */
    public static ConfigService createConfigService(AgentConfig config, Map<String, Object> localSettings) {
        return new ConfigServiceImpl(config, null, localSettings);
    }

    /**
     * For testing.
     */
    public static ConfigService createConfigServiceUsingSettings(Map<String, Object> settings) {
        return new ConfigServiceImpl(AgentConfigImpl.createAgentConfig(settings), null, settings);
    }

    public static ConfigService createConfigService() throws ConfigurationException, ForceDisconnectException {
        File configFile = getConfigFile();
        Map<String, Object> configSettings = getConfigurationFileSettings(configFile);
        AgentConfig config = AgentConfigImpl.createAgentConfig(configSettings);
        validateConfig(config);
        return new ConfigServiceImpl(config, configFile, configSettings);
    }

    public static Map<String, Object> getConfigurationFileSettings(File configFile) throws ConfigurationException {
        String msg = MessageFormat.format("New Relic Agent: Loading configuration file \"{0}\"", configFile.getPath());
        Agent.LOG.info(msg);
        try {
            return AgentConfigHelper.getConfigurationFileSettings(configFile);
        } catch (Exception e) {
            msg = MessageFormat.format(
                    "An error occurred reading the configuration file {0}. Check the permissions and format of the file. - {1}",
                    configFile.getAbsolutePath(), e.toString());
            throw new ConfigurationException(msg, e);
        }
    }

    /**
     * Get the Agent's configuration file.
     *
     * @return the configuration file
     * @throws ConfigurationException if the configuration file can not be found
     */
    private static File getConfigFile() throws ConfigurationException {
        File configFile = ConfigFileHelper.findConfigFile();
        if (configFile == null) {
            throw new ConfigurationException("Failed to find the configuration file");
        }
        return configFile;
    }

    @VisibleForTesting
    public static void validateConfig(AgentConfig config) throws ConfigurationException, ForceDisconnectException {
        if (config.getApplicationName() == null) {
            throw new ConfigurationException(
                    "The agent requires an application name. Check the app_name setting in newrelic.yml");
        }
        if (config.getApplicationNames().size() > 3) {
            throw new ConfigurationException(
                    "The agent does not support more than three application names. Check the app_name setting in newrelic.yml");
        }
        if (config.isHighSecurity() && config.laspEnabled()) {
            throw new ForceDisconnectException(
                    "Security Policies and High Security Mode cannot both be present in the agent configuration. " +
                            "If Security Policies have been set for your account, please ensure the security_policies_token " +
                            "is set but high_security is disabled (default).");
        }
    }

}
