package com.newrelic.agent.config;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.newrelic.agent.browser.BrowserConfig;
import com.newrelic.agent.database.SqlObfuscator;
import com.newrelic.agent.normalization.NormalizationRuleFactory;
import com.newrelic.agent.reinstrument.RemoteInstrumentationServiceImpl;
import com.newrelic.agent.transport.DataSenderWriter;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class AgentConfigFactory {

    public static final String AGENT_CONFIG = "agent_config";
    public static final String PERIOD_REGEX = "\\.";
    public static final String DOT_SEPARATOR = ".";
    public static final String SLOW_SQL_PREFIX = AgentConfigImpl.SLOW_SQL + DOT_SEPARATOR;
    public static final String TRANSACTION_TRACER_PREFIX = AgentConfigImpl.TRANSACTION_TRACER + DOT_SEPARATOR;
    public static final String TRANSACTION_TRACER_CATEGORY_BACKGROUND_PREFIX = AgentConfigImpl.TRANSACTION_TRACER
            + ".category.background.";
    public static final String TRANSACTION_TRACER_CATEGORY_REQUEST_PREFIX = AgentConfigImpl.TRANSACTION_TRACER
            + ".category.request.";
    public static final String ERROR_COLLECTOR_PREFIX = AgentConfigImpl.ERROR_COLLECTOR + DOT_SEPARATOR;
    public static final String THREAD_PROFILER_PREFIX = AgentConfigImpl.THREAD_PROFILER + DOT_SEPARATOR;
    public static final String TRANSACTION_EVENTS_PREFIX = AgentConfigImpl.TRANSACTION_EVENTS + DOT_SEPARATOR;
    public static final String CUSTOM_INSIGHT_EVENTS_PREFIX = AgentConfigImpl.CUSTOM_INSIGHT_EVENTS + DOT_SEPARATOR;
    public static final String SPAN_EVENTS_PREFIX = AgentConfigImpl.SPAN_EVENTS + DOT_SEPARATOR;
    public static final String BROWSER_MONITORING_PREFIX = AgentConfigImpl.BROWSER_MONITORING + DOT_SEPARATOR;
    public static final String HIGH_SECURITY = "high_security";
    public static final String SECURITY_POLICIES_TOKEN = "security_policies_token";
    public static final String COLLECT_ERRORS = ERROR_COLLECTOR_PREFIX + ErrorCollectorConfigImpl.COLLECT_ERRORS;
    public static final String EXPECTED_CLASSES = ERROR_COLLECTOR_PREFIX + ErrorCollectorConfigImpl.EXPECTED_CLASSES;
    public static final String EXPECTED_STATUS_CODES =
            ERROR_COLLECTOR_PREFIX + ErrorCollectorConfigImpl.EXPECTED_STATUS_CODES;
    public static final String COLLECT_ERROR_EVENTS = ERROR_COLLECTOR_PREFIX + ErrorCollectorConfigImpl.COLLECT_EVENTS;
    public static final String CAPTURE_ERROR_EVENTS = ERROR_COLLECTOR_PREFIX + ErrorCollectorConfigImpl.CAPTURE_EVENTS;
    public static final String CUSTOM_INSIGHTS_ENABLED = AgentConfigImpl.INSIGHTS + DOT_SEPARATOR + InsightsConfigImpl.ENABLED_PROP;
    public static final String MAX_ERROR_EVENT_SAMPLES_STORED = ERROR_COLLECTOR_PREFIX
            + ErrorCollectorConfigImpl.MAX_EVENT_SAMPLES_STORED;
    public static final String COLLECT_TRACES = TRANSACTION_TRACER_PREFIX + TransactionTracerConfigImpl.COLLECT_TRACES;
    public static final String COLLECT_TRANSACTION_EVENTS = TRANSACTION_EVENTS_PREFIX + "collect_analytics_events";
    public static final String COLLECT_SPAN_EVENTS = SPAN_EVENTS_PREFIX + SpanEventsConfig.COLLECT_SPAN_EVENTS;
    public static final String COLLECT_CUSTOM_INSIGHTS_EVENTS = CUSTOM_INSIGHT_EVENTS_PREFIX + InsightsConfigImpl.COLLECT_CUSTOM_EVENTS;
    public static final String RECORD_SQL = TRANSACTION_TRACER_PREFIX + TransactionTracerConfigImpl.RECORD_SQL;
    public static final String SLOW_QUERY_WHITELIST = TRANSACTION_TRACER_PREFIX
            + TransactionTracerConfigImpl.SLOW_QUERY_WHITELIST;
    public static final String CROSS_APPLICATION_TRACER_PREFIX = AgentConfigImpl.CROSS_APPLICATION_TRACER
            + DOT_SEPARATOR;
    public static final String DISTRIBUTED_TRACING_PREFIX = AgentConfigImpl.DISTRIBUTED_TRACING + DOT_SEPARATOR;
    public static final String ENCODING_KEY = CROSS_APPLICATION_TRACER_PREFIX + CrossProcessConfigImpl.ENCODING_KEY;
    public static final String CROSS_PROCESS_ID = CROSS_APPLICATION_TRACER_PREFIX
            + CrossProcessConfigImpl.CROSS_PROCESS_ID;
    public static final String TRUSTED_ACCOUNT_IDS = CROSS_APPLICATION_TRACER_PREFIX
            + CrossProcessConfigImpl.TRUSTED_ACCOUNT_IDS;
    public static final String TRUSTED_ACCOUNT_KEY = DISTRIBUTED_TRACING_PREFIX + DistributedTracingConfig.TRUSTED_ACCOUNT_KEY;
    public static final String ACCOUNT_ID = DISTRIBUTED_TRACING_PREFIX + DistributedTracingConfig.ACCOUNT_ID;
    public static final String PRIMARY_APPLICATION_ID = DISTRIBUTED_TRACING_PREFIX + DistributedTracingConfig.PRIMARY_APPLICATION_ID;
    public static final String DISTRIBUTED_TRACING_ENABLED = DISTRIBUTED_TRACING_PREFIX + DistributedTracingConfig.ENABLED;
    public static final String STRIP_EXCEPTION = AgentConfigImpl.STRIP_EXCEPTION_MESSAGES;
    public static final String STRIP_EXCEPTION_ENABLED =
            STRIP_EXCEPTION + DOT_SEPARATOR + StripExceptionConfigImpl.ENABLED;
    public static final String STRIP_EXCEPTION_WHITELIST =
            STRIP_EXCEPTION + DOT_SEPARATOR + StripExceptionConfigImpl.WHITELIST;
    public static final String DATA_METHODS = "data_methods";

    public static AgentConfig createAgentConfig(Map<String, Object> localSettings, Map<String, Object> serverData,
                                                Map<String, Boolean> laspData) {
        Map<String, Object> mergedSettings = createMap(localSettings);
        mergeServerData(mergedSettings, serverData, laspData);
        return AgentConfigImpl.createAgentConfig(mergedSettings);
    }

    @SuppressWarnings("unchecked")
    public static Map<String, Object> getAgentData(Map<String, Object> serverData) {
        if (serverData == null) {
            return createMap();
        }
        Object agentData = createMap((Map<String, Object>) serverData.get(AGENT_CONFIG));
        if (agentData == null || DataSenderWriter.nullValue().equals(agentData)) {
            return createMap();
        }
        return (Map<String, Object>) agentData;
    }

    // be careful with high security here - ssl must stay at true, record_sql must be off or obfuscated
    public static void mergeServerData(Map<String, Object> settings, Map<String, Object> serverData, Map<String, Boolean> laspData) {
        if (serverData == null && laspData == null) {
            return;
        }

        if (serverData == null) {
            serverData = Maps.newHashMap();
        }
        if (laspData == null) {
            laspData = Maps.newHashMap();
        }

        AgentConfig settingsConfig = AgentConfigImpl.createAgentConfig(settings);
        Map<String, Object> agentData = getAgentData(serverData);

        String recordSqlSecure = getMostSecureSql(agentData, settingsConfig, laspData);

        Boolean laspAttributesInclude = laspData.get(AttributesConfigImpl.ATTS_INCLUDE);
        List<String> attributesInclude = (laspAttributesInclude == null || laspAttributesInclude)
                ? settingsConfig.getAttributesConfig().attributesRootInclude() : Collections.<String>emptyList();
        String attributesIncludeSecure = Joiner.on(",").join(attributesInclude);

        Boolean captureMessageParameters = getLaspValue(laspData, LaspPolicies.LASP_MESSAGE_PARAMETERS, true);
        if (!captureMessageParameters) {
            List<String> filteredAttributes = Lists.newArrayList();
            String[] attributes = attributesIncludeSecure.split(",");
            for (String attribute : attributes) {
                // If we don't want to capture message parameters we need to remove them from the includes list
                if (attribute.startsWith("message.parameters.")) {
                    continue;
                }
                filteredAttributes.add(attribute);
            }
            attributesIncludeSecure = Joiner.on(",").join(filteredAttributes);
        }

        // we OR this comparison instead of ANDing it because the most secure state is if strip_exception_messages is true
        Boolean stripExceptionMessagesSecure = settingsConfig.getStripExceptionConfig().isEnabled()
                || getLaspValue(laspData, STRIP_EXCEPTION_ENABLED, false);

        Boolean customEventsSecure;
        Boolean serverSideCustomEvents = (Boolean) serverData.get(InsightsConfigImpl.COLLECT_CUSTOM_EVENTS);
        if (serverSideCustomEvents == null) {
            customEventsSecure = settingsConfig.getInsightsConfig().isEnabled()
                    && getLaspValue(laspData, CUSTOM_INSIGHTS_ENABLED, true);
        } else {
            customEventsSecure = settingsConfig.getInsightsConfig().isEnabled()
                    && getLaspValue(laspData, CUSTOM_INSIGHTS_ENABLED, true) && serverSideCustomEvents;
        }

        Boolean customParametersSecure = getLaspValue(laspData, LaspPolicies.LASP_CUSTOM_PARAMETERS, true);

        Boolean customInstrumentationEditor = getLaspValue(laspData, LaspPolicies.LASP_CUSTOM_INSTRUMENTATION_EDITOR, true);

        // calling remove here prevents mergeAgentData from always overriding local config
        // remove deprecated cross_application_tracing property
        agentData.remove(CrossProcessConfigImpl.CROSS_APPLICATION_TRACING);
        agentData.remove(HIGH_SECURITY);
        agentData.remove("reinstrument");
        agentData.remove("reinstrument.attributes_enabled");
        agentData.remove(STRIP_EXCEPTION);
        agentData.remove(STRIP_EXCEPTION_ENABLED);
        agentData.remove(STRIP_EXCEPTION_WHITELIST);
        agentData.remove(SLOW_QUERY_WHITELIST);
        Object serverRecordSql = agentData.remove(RECORD_SQL);

        // handle high security - ssl needs to stay off and record_sql must stay as obfuscated or off
        if (isHighSecurity(settingsConfig.getProperty(AgentConfigImpl.HIGH_SECURITY))) {
            agentData.remove(AgentConfigImpl.IS_SSL);
            // check record sql
            if (isValidRecordSqlValue(serverRecordSql)) {
                addServerProp(RECORD_SQL, serverRecordSql, settings);
            }
        } else {
            // we are not in high security - set whatever property you want
            addServerProp(RECORD_SQL, serverRecordSql, settings);
        }

        addServerProp(AgentConfigImpl.TRANSACTION_NAMING_SCHEME, serverData.get(AgentConfigImpl.TRANSACTION_NAMING_SCHEME), settings);

        mergeAgentData(settings, agentData);
        // server properties
        addServerProp(AgentConfigImpl.APDEX_T, serverData.get(AgentConfigImpl.APDEX_T), settings);

        addServerProp(COLLECT_ERRORS, serverData.get(ErrorCollectorConfigImpl.COLLECT_ERRORS), settings);
        addServerProp(COLLECT_ERROR_EVENTS, serverData.get(ErrorCollectorConfigImpl.COLLECT_EVENTS), settings);
        addServerProp(CAPTURE_ERROR_EVENTS, serverData.get(ErrorCollectorConfigImpl.CAPTURE_EVENTS), settings);
        addServerProp(MAX_ERROR_EVENT_SAMPLES_STORED, serverData.get(ErrorCollectorConfigImpl.MAX_EVENT_SAMPLES_STORED), settings);
        addServerProp(COLLECT_TRACES, serverData.get(TransactionTracerConfigImpl.COLLECT_TRACES), settings);
        addServerProp(COLLECT_TRANSACTION_EVENTS, serverData.get("collect_analytics_events"), settings);
        addServerProp(COLLECT_CUSTOM_INSIGHTS_EVENTS, serverData.get(InsightsConfigImpl.COLLECT_CUSTOM_EVENTS), settings);
        addServerProp(COLLECT_SPAN_EVENTS, serverData.get(SpanEventsConfig.COLLECT_SPAN_EVENTS), settings);
        // key transaction server properties
        addServerProp(AgentConfigImpl.KEY_TRANSACTIONS, serverData.get(AgentConfigImpl.KEY_TRANSACTIONS), settings);
        // cross application tracing server properties
        addServerProp(CROSS_PROCESS_ID, serverData.get(CrossProcessConfigImpl.CROSS_PROCESS_ID), settings);
        addServerProp(ENCODING_KEY, serverData.get(CrossProcessConfigImpl.ENCODING_KEY), settings);
        addServerProp(TRUSTED_ACCOUNT_IDS, serverData.get(CrossProcessConfigImpl.TRUSTED_ACCOUNT_IDS), settings);
        // distributed tracing server properties received on connect
        addServerProp(TRUSTED_ACCOUNT_KEY, serverData.get(DistributedTracingConfig.TRUSTED_ACCOUNT_KEY), settings);
        addServerProp(ACCOUNT_ID, serverData.get(DistributedTracingConfig.ACCOUNT_ID), settings);
        addServerProp(PRIMARY_APPLICATION_ID, serverData.get(DistributedTracingConfig.PRIMARY_APPLICATION_ID), settings);
        // Expected errors server properties
        addServerProp(EXPECTED_CLASSES, serverData.get(ErrorCollectorConfigImpl.EXPECTED_CLASSES), settings);
        addServerProp(EXPECTED_STATUS_CODES, serverData.get(ErrorCollectorConfigImpl.EXPECTED_STATUS_CODES), settings);

        if (settingsConfig.getProperty(SECURITY_POLICIES_TOKEN) != null) {
            addServerProp(RECORD_SQL, recordSqlSecure, settings);
            addServerProp(AttributesConfigImpl.ATTS_INCLUDE, attributesIncludeSecure, settings);
            addServerProp(STRIP_EXCEPTION_ENABLED, stripExceptionMessagesSecure, settings);
            addServerProp(CUSTOM_INSIGHTS_ENABLED, customEventsSecure, settings);
            addServerProp(LaspPolicies.LASP_CUSTOM_PARAMETERS, customParametersSecure, settings);
            addServerProp(LaspPolicies.LASP_CUSTOM_INSTRUMENTATION_EDITOR, customInstrumentationEditor, settings);
        }

        // Copy "data_methods" over from the serverData since it doesn't live in the "agent_config" subsection (it's a top level property from the collector)
        addServerProp(AgentConfigFactory.DATA_METHODS, serverData.get(DATA_METHODS), settings);

        // Browser settings
        addServerProp(BrowserConfig.BROWSER_KEY, serverData.get(BrowserConfig.BROWSER_KEY), settings);
        addServerProp(BrowserConfig.BROWSER_LOADER_VERSION, serverData.get(BrowserConfig.BROWSER_LOADER_VERSION), settings);
        addServerProp(BrowserConfig.JS_AGENT_LOADER, serverData.get(BrowserConfig.JS_AGENT_LOADER), settings);
        addServerProp(BrowserConfig.JS_AGENT_FILE, serverData.get(BrowserConfig.JS_AGENT_FILE), settings);
        addServerProp(BrowserConfig.BEACON, serverData.get(BrowserConfig.BEACON), settings);
        addServerProp(BrowserConfig.ERROR_BEACON, serverData.get(BrowserConfig.ERROR_BEACON), settings);
        addServerProp(BrowserConfig.APPLICATION_ID, serverData.get(BrowserConfig.APPLICATION_ID), settings);

        // Normalization settings
        addServerProp(NormalizationRuleFactory.URL_RULES_KEY, serverData.get(NormalizationRuleFactory.URL_RULES_KEY), settings);
        addServerProp(NormalizationRuleFactory.METRIC_NAME_RULES_KEY, serverData.get(NormalizationRuleFactory.METRIC_NAME_RULES_KEY), settings);
        addServerProp(NormalizationRuleFactory.TRANSACTION_NAME_RULES_KEY, serverData.get(NormalizationRuleFactory.TRANSACTION_NAME_RULES_KEY), settings);
        addServerProp(NormalizationRuleFactory.TRANSACTION_SEGMENT_TERMS_KEY, serverData.get(NormalizationRuleFactory.TRANSACTION_SEGMENT_TERMS_KEY), settings);

        // Remote instrumentation
        addServerProp(RemoteInstrumentationServiceImpl.INSTRUMENTATION_CONFIG, serverData.get(RemoteInstrumentationServiceImpl.INSTRUMENTATION_CONFIG), settings);
    }

    private static String getMostSecureSql(Map<String, Object> agentData, AgentConfig settings,
                                           Map<String, Boolean> laspData) {

        String server = (String) agentData.get(RECORD_SQL);
        String local = settings.getTransactionTracerConfig().getRecordSql();
        Boolean lasp = getLaspValue(laspData, RECORD_SQL, null);

        if (SqlObfuscator.OFF_SETTING.equals(server) || SqlObfuscator.OFF_SETTING.equals(local)
                || (lasp != null && !lasp)) {
            return SqlObfuscator.OFF_SETTING;
        }
        if (SqlObfuscator.OBFUSCATED_SETTING.equals(server) || SqlObfuscator.OBFUSCATED_SETTING.equals(local)
                || (lasp != null && lasp)) {
            return SqlObfuscator.OBFUSCATED_SETTING;
        }
        return SqlObfuscator.RAW_SETTING;
    }

    private static Boolean getLaspValue(Map<String, Boolean> policies, String key, Boolean defaultValue) {
        if (policies != null && policies.containsKey(key)) {
            return policies.get(key);
        }
        return defaultValue;
    }

    private static boolean isValidRecordSqlValue(Object recordSqlValue) {
        if ((recordSqlValue == null) || !(recordSqlValue instanceof String)) {
            return false;
        }
        String rSql = ((String) recordSqlValue).toLowerCase();
        if (!(rSql.equals(SqlObfuscator.OFF_SETTING) || rSql.equals(SqlObfuscator.OBFUSCATED_SETTING))) {
            return false;
        }
        return true;
    }

    private static boolean isHighSecurity(Object value) {
        return (value != null) && (value instanceof Boolean) && (((Boolean) value).booleanValue());
    }

    private static void mergeAgentData(Map<String, Object> settings, Map<String, Object> agentData) {
        Map.Entry<String, Object> entry;
        for (Iterator<Map.Entry<String, Object>> it = agentData.entrySet().iterator(); it.hasNext(); addServerProp(
                entry.getKey(), entry.getValue(), settings)) {
            entry = it.next();
        }
    }

    @SuppressWarnings("unchecked")
    private static void addServerProp(String prop, Object val, Map<String, Object> settings) {
        if (val == null) {
            return;
        }
        Map<String, Object> currentMap = settings;
        int count = 0;
        String[] propArray = prop.split(PERIOD_REGEX);
        for (String propPart : propArray) {
            count++;
            if (count < propArray.length) {
                Map<String, Object> propMap = null;

                Object propValue = currentMap.get(propPart);
                if (propValue instanceof Map) {
                    propMap = (Map<String, Object>) currentMap.get(propPart);
                } else if (propValue instanceof ServerProp) {
                    propMap = (Map<String, Object>) ((ServerProp) propValue).getValue();
                }

                if (propMap == null) {
                    propMap = createMap();
                    currentMap.put(propPart, propMap);
                }
                currentMap = propMap;
            } else {
                currentMap.put(propPart, ServerProp.createPropObject(val));
            }
        }
    }

    private static Map<String, Object> createMap() {
        return new HashMap<String, Object>();
    }

    public static Map<String, Object> createMap(Map<String, Object> settings) {
        Map<String, Object> result = createMap();
        if (settings == null) {
            return result;
        }
        Map.Entry<String, Object> entry;
        for (Iterator<Map.Entry<String, Object>> it = settings.entrySet().iterator(); it.hasNext(); putOrCreate(
                entry.getKey(), entry.getValue(), result)) {
            entry = it.next();
        }
        return result;
    }

    @SuppressWarnings("unchecked")
    private static void putOrCreate(String key, Object val, Map<String, Object> settings) {
        if (val instanceof Map<?, ?>) {
            settings.put(key, createMap((Map<String, Object>) val));
        } else {
            settings.put(key, val);
        }
    }

}
