package com.newrelic.agent.attributes;

import com.newrelic.agent.Agent;
import com.newrelic.agent.SpanImpl;
import com.newrelic.agent.service.ServiceFactory;

import java.util.Collections;
import java.util.Map;
import java.util.logging.Level;

public class CustomSpanAttributeSender extends AttributeSender {

    private static final String ATTRIBUTE_TYPE = "custom";

    private static final int maxUserParameters = ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameters();

    private final SpanImpl span;

    public CustomSpanAttributeSender(SpanImpl span) {
        this.span = span;
    }

    @Override
    protected String getAttributeType() {
        return ATTRIBUTE_TYPE;
    }

    @Override
    protected Map<String, Object> getAttributeMap() throws Throwable {
        return span.getUserAttributes();
    }

    public Object verifyParameterAndReturnValue(String key, Object value, String methodCalled) {
        try {
            if (ServiceFactory.getConfigService().getDefaultAgentConfig().isHighSecurity()) {
                Agent.LOG.log(Level.FINER,
                        "Unable to add {0} attribute when {1} was invoked with key \"{2}\" while in high security mode.",
                        getAttributeType(), methodCalled, key);
                return null;
            }
            if (!ServiceFactory.getConfigService().getDefaultAgentConfig().isCustomParametersAllowed()) {
                Agent.LOG.log(Level.FINER,
                        "Unable to add {0} attribute when {1} was invoked with key \"{2}\" while lasp custom_parameters disabled.",
                        getAttributeType(), methodCalled, key);
                return null;
            }
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINEST,
                    "Unable to verify attribute. Exception thrown while verifying security policies", t);
            return null;
        }
        return super.verifyParameterAndReturnValue(key, value, methodCalled);
    }

    public Map<String, Object> verifyParametersAndReturnValues(Map<String, Object> params, String methodCalled) {
        try {
            if (ServiceFactory.getConfigService().getDefaultAgentConfig().isHighSecurity()) {
                Agent.LOG.log(Level.FINER,
                        "Unable to add {0} attributes when {1} was invoked with keys \"{2}\" while in high security mode.",
                        getAttributeType(), methodCalled, params.keySet());
                return Collections.emptyMap();
            }
            if (!ServiceFactory.getConfigService().getDefaultAgentConfig().isCustomParametersAllowed()) {
                Agent.LOG.log(Level.FINER,
                        "Unable to add {0} attributes when {1} was invoked with keys \"{2}\" while lasp custom_parameters disabled.",
                        getAttributeType(), methodCalled, params.keySet());
                return Collections.emptyMap();
            }
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINEST,
                    "Unable to verify attributes. Exception thrown while verifying security policies", t);
            return Collections.emptyMap();
        }
        return super.verifyParametersAndReturnValues(params, methodCalled);
    }

    protected void addCustomAttributeImpl(String key, Object value, String methodName) {
        try {
            // check the max size for user attributes
            Map<String, Object> attributeMap = getAttributeMap();
            if (attributeMap != null && attributeMap.size() >= maxUserParameters) {
                Agent.LOG.log(Level.FINER, "Unable to add {0} attribute for key \"{1}\" because the limit is {2}.", getAttributeType(), key, maxUserParameters);
                return;
            }
            super.addCustomAttributeImpl(key, value, methodName);
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINER, "Exception adding {0} parameter for key: \"{1}\": {2}", getAttributeType(), key,
                    t);
        }
    }

    protected void addCustomAttributesImpl(Map<String, Object> params, String methodName) {
        try {
            // check the max size for user attributes
            Map<String, Object> attributeMap = getAttributeMap();
            if (attributeMap != null && attributeMap.size() >= maxUserParameters) {
                Agent.LOG.log(Level.FINER, "Unable to add {0} attributes for keys \"{1}\" because the limit is {2}.", getAttributeType(), params.keySet(), maxUserParameters);
                return;
            }
            super.addCustomAttributesImpl(params, methodName);
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINER, "Exception adding {0} parameters for keys: \"{1}\": {2}", getAttributeType(), params.keySet(), t);
        }
    }
}
