package com.newrelic.agent.attributes;

import com.google.common.collect.ImmutableMap;
import com.newrelic.agent.bridge.AgentBridge;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentConfigImpl;
import com.newrelic.agent.config.AttributesConfigImpl;

import java.util.Map;
import java.util.logging.Level;

/**
 * Filters attributes. While this filter does provide boolean on whether request parameter and message parameters should
 * be filtered, there is no concept of a user vs agent attribute here. This means for high security, user attributes
 * need to be turned off elsewhere.
 */
public class AttributesFilter {

    // booleans to optimize the collection of request/message parameters
    private final boolean captureRequestParameters;
    private final boolean captureMessageParameters;

    private final DestinationFilter rootFilter;
    private final DestinationFilter errorFilter;
    private final DestinationFilter eventsFilter;
    private final DestinationFilter traceFilter;
    private final DestinationFilter browserFilter;
    private final Map<String, DestinationFilter> destinationFilterMap;

    public AttributesFilter(AgentConfig config) {
        this(config, AttributesConfigImpl.DEFAULT_BROWSER_EXCLUDES, AttributesConfigImpl.DEFAULT_ERRORS_EXCLUDES,
                AttributesConfigImpl.DEFAULT_EVENTS_EXCLUDES, AttributesConfigImpl.DEFAULT_TRACES_EXCLUDES);
    }

    // have this constructor for testing
    public AttributesFilter(AgentConfig config, String[] defaultExcludeBrowser, String[] defaultExcludeErrors,
            String[] defaultExcludeEvents, String[] defaultExcludeTraces) {

        rootFilter = new DestinationFilter(AgentConfigImpl.ATTRIBUTES, true, config, defaultExcludeErrors,
                AgentConfigImpl.ATTRIBUTES);

        errorFilter = new DestinationFilter(AgentConfigImpl.ERROR_COLLECTOR, true, config, defaultExcludeErrors,
                AgentConfigImpl.ERROR_COLLECTOR);

        eventsFilter = new DestinationFilter(AgentConfigImpl.TRANSACTION_EVENTS, true, config, defaultExcludeEvents,
                AgentConfigImpl.TRANSACTION_EVENTS);

        traceFilter = new DestinationFilter(AgentConfigImpl.TRANSACTION_TRACER, true, config, defaultExcludeTraces,
                AgentConfigImpl.TRANSACTION_TRACER);

        // browser - default is false
        browserFilter = new DestinationFilter(AgentConfigImpl.BROWSER_MONITORING, false, config, defaultExcludeBrowser,
                AgentConfigImpl.BROWSER_MONITORING);

        destinationFilterMap = ImmutableMap.<String, DestinationFilter>builder()
                .put(AgentConfigImpl.ATTRIBUTES, rootFilter)
                .put(AgentConfigImpl.ERROR_COLLECTOR, errorFilter)
                .put(AgentConfigImpl.TRANSACTION_EVENTS, eventsFilter)
                .put(AgentConfigImpl.TRANSACTION_TRACER, traceFilter)
                .put(AgentConfigImpl.BROWSER_MONITORING, browserFilter)
                .build();

        // browser is not included in this list because we will never send request params to browser monitoring
        // since the request parameters are pulled in the do finish
        boolean enabled = errorFilter.isEnabled() || eventsFilter.isEnabled() || traceFilter.isEnabled();
        captureRequestParameters = captureAllParams(enabled, config.isHighSecurity(), "request.parameters.");
        captureMessageParameters = captureAllParams(enabled, config.isHighSecurity(), "message.parameters.");
    }

    /*
     * Returns true if we definitely need to capture parameters (request or message depending on the paramStart). In
     * order to not make the logic too complex, there are cases when this will return true but not actually capture any
     * parameters.
     */
    private boolean captureAllParams(boolean enabled, boolean highSecurity, String paramStart) {
        if (!enabled || highSecurity) {
            return false;
        } else {
            return errorFilter.isPotentialConfigMatch(paramStart)
                    || eventsFilter.isPotentialConfigMatch(paramStart)
                    || traceFilter.isPotentialConfigMatch(paramStart)
                    || browserFilter.isPotentialConfigMatch(paramStart);
        }
    }

    public boolean captureRequestParams() {
        return captureRequestParameters;
    }

    public boolean captureMessageParams() {
        return captureMessageParameters;
    }

    public boolean isAttributesEnabledForErrors() {
        return errorFilter.isEnabled();
    }

    public boolean isAttributesEnabledForEvents() {
        return eventsFilter.isEnabled();
    }

    public boolean isAttributesEnabledForTraces() {
        return traceFilter.isEnabled();
    }

    public boolean isAttributesEnabledForBrowser() {
        return browserFilter.isEnabled();
    }

    public Map<String, ? extends Object> filterErrorAttributes(Map<String, ? extends Object> values) {
        return errorFilter.filterAttributes(values);
    }

    public Map<String, ? extends Object> filterEventAttributes(Map<String, ? extends Object> values) {
        return eventsFilter.filterAttributes(values);
    }

    public Map<String, ? extends Object> filterTraceAttributes(Map<String, ? extends Object> values) {
        return traceFilter.filterAttributes(values);
    }

    public Map<String, ? extends Object> filterBrowserAttributes(Map<String, ? extends Object> values) {
        return browserFilter.filterAttributes(values);
    }

    public Map<String, ? extends Object> filterAttributes(Map<String, ? extends Object> values) {
        return rootFilter.filterAttributes(values);
    }

    public Map<String, ? extends Object> filterAttributesForDestination(Map<String, ? extends Object> values,
            String destination) {
        DestinationFilter destinationFilter = destinationFilterMap.get(destination);
        if (destinationFilter == null) {
            AgentBridge.getAgent().getLogger().log(Level.SEVERE,
                    "Invalid destination for attribute filter {0}. Attributes are not filtered.", destination);
            return values;
        }

        return destinationFilter.filterAttributes(values);
    }

}
