package com.newrelic.agent.attributes;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import com.newrelic.agent.Agent;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.service.ServiceFactory;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

public abstract class AttributeSender {
    protected static String ATTRIBUTE_TYPE;

    // Most attribute senders are transaction (demand that a transaction be in progress).
    // Subclasses may choose to disable this check.
    private boolean transactional = true;

    // Set of APIs that can report attributes outside of a transaction
    private static final Set<String> sendParametersOutsideOfTxn = ImmutableSet.of("noticeError", "Span.addCustomParameter", "Span.addCustomParameters");

    private static final int maxUserParameters = ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameters();

    /**
     * The name used in logging.
     */
    protected abstract String getAttributeType();

    /**
     * The map attributes will be added to.
     */
    protected abstract Map<String, Object> getAttributeMap() throws Throwable;

    /**
     * Allow attribute sending when there is no current transaction.
     *
     * @param newSetting new value of transactional setting.
     */
    protected void setTransactional(boolean newSetting) {
        this.transactional = newSetting;
    }

    protected void addCustomAttributeImpl(String key, Object value, String methodName) {
        // perform general checks
        Object filteredValue = verifyParameterAndReturnValue(key, value, methodName);
        // null will be returned if the key/value failed validation
        if (filteredValue == null) {
            return;
        }
        try {
            Map<String, Object> attributeMap = getAttributeMap();
            if (attributeMap != null) {
                attributeMap.put(key, filteredValue);
                Agent.LOG.log(Level.FINER, "Added {0} attribute \"{1}\": {2}", getAttributeType(), key, filteredValue);
                MetricNames.recordApiSupportabilityMetric(MetricNames.SUPPORTABILITY_API_ADD_CUSTOM_PARAMETER);

            }
        } catch (Throwable t) {
            if (Agent.LOG.isLoggable(Level.FINEST)) {
                Agent.LOG.log(Level.FINEST, "Exception adding attribute for key: \"{0}\": {1}", key, t);
            } else if (Agent.LOG.isLoggable(Level.FINER)) {
                Agent.LOG.log(Level.FINER, "Exception adding attribute for key: \"{0}\": {1}", key);
            }
        }
    }

    protected void addCustomAttributesImpl(Map<String, Object> params, String methodName) {
        // an empty map will be returned if all keys/values fail verification
        Map<String, Object> filteredValues = verifyParametersAndReturnValues(params, methodName);
        if (filteredValues == null || filteredValues.isEmpty()) {
            return;
        }
        try {
            Map<String, Object> attributeMap = getAttributeMap();
            if (attributeMap != null) {
                attributeMap.putAll(filteredValues);
                Agent.LOG.log(Level.FINER, "Added {0} attributes \"{1}\"", getAttributeType(), filteredValues);
                MetricNames.recordApiSupportabilityMetric(MetricNames.SUPPORTABILITY_API_ADD_CUSTOM_PARAMETER);
            }
        } catch (Throwable t) {
            if (Agent.LOG.isLoggable(Level.FINEST)) {
                Agent.LOG.log(Level.FINEST, "Exception adding attributes for keys: \"{0}\": {1}", filteredValues.keySet(), t);
            } else if (Agent.LOG.isLoggable(Level.FINER)) {
                Agent.LOG.log(Level.FINER, "Exception adding attributes for keys: \"{0}\": {1}", filteredValues.keySet());
            }
        }
    }

    /**
     * Verifies the input key and value. Null is returned if the key/value is invalid, else the value is returned.
     * Parameters added by the noticeError or transaction-less Span APIs will be reported regardless if they're in a transaction
     * or not. In cases where parameters are longer than the maxUserParameterSize the returned value will be truncated.
     */
    public Object verifyParameterAndReturnValue(String key, Object value, String methodCalled) {
        if (key == null) {
            Agent.LOG.log(Level.FINER, "Unable to add {0} attribute because {1} was invoked with a null key",
                    getAttributeType(), methodCalled);
            return null;
        }
        if (value == null) {
            Agent.LOG.log(Level.FINER,
                    "Unable to add {0} attribute because {1} was invoked with a null value for key \"{2}\"",
                    getAttributeType(), methodCalled, key);
            return null;
        }
        // key and value are limited to 255 characters
        Transaction tx = Transaction.getTransaction(false);
        int maxUserParameterSize = tx == null
                ? ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameterSize()
                : tx.getAgentConfig().getMaxUserParameterSize();

        if (!validateAndLogKeyLength(key, maxUserParameterSize, methodCalled)) {
            return null;
        }

        if (value instanceof String) {
            value = truncateValue(key, (String) value, maxUserParameterSize, methodCalled);
        }

        if (sendParametersOutsideOfTxn.contains(methodCalled)) {
            return value;
        }

        if (transactional && (tx == null || !tx.isInProgress())) {
            Agent.LOG.log(Level.FINER,
                    "Unable to add {0} attribute with key \"{1}\" because {2} was invoked outside a New Relic transaction.",
                    getAttributeType(), key, methodCalled);
            return null;
        }

        return value;
    }

    /**
     * Verifies a map of  key/value pairs and returns a new map containing the verified parameters. An empty
     * map is returned if there are no valid keys/values pairs or if addCustomParameters(Map<String, Object>) is invoked
     * outside of a New Relic transaction, with the exception being that parameters added by the noticeError or
     * transaction-less Span APIs will be reported regardless if they're in a transaction or not. In cases where
     * parameters are longer than the maxUserParameterSize the returned value will be truncated.
     */
    public Map<String, Object> verifyParametersAndReturnValues(Map<String, Object> params, String methodCalled) {
        Map<String, Object> verifiedParams = new LinkedHashMap<String, Object>();
        if (params == null || params.isEmpty()) {
            return Collections.emptyMap();
        } else {
            Transaction tx = Transaction.getTransaction(false);
            // key and value are limited to 255 characters
            int maxUserParameterSize = tx == null
                    ? ServiceFactory.getConfigService().getDefaultAgentConfig().getMaxUserParameterSize()
                    : tx.getAgentConfig().getMaxUserParameterSize();

            // iterate over params in map, creating new map containing only valid entries
            for (Map.Entry<String, Object> current : params.entrySet()) {
                String currentKey = current.getKey();
                Object currentValue = current.getValue();

                if (currentKey == null) {
                    Agent.LOG.log(Level.FINER, "Unable to add {0} attribute because {1} was invoked with a null key",
                            getAttributeType(), methodCalled);
                    continue; // ignore and continue to next entry
                }
                if (currentValue == null) {
                    Agent.LOG.log(Level.FINER,
                            "Unable to add {0} attribute because {1} was invoked with a null value for key \"{2}\"",
                            getAttributeType(), methodCalled, currentKey);
                    continue; // ignore and continue to next entry
                }
                if (!validateAndLogKeyLength(currentKey, maxUserParameterSize, methodCalled)) {
                    continue; // ignore and continue to next entry
                }
                if (currentValue instanceof String) {
                    currentValue = truncateValue(currentKey, (String) currentValue, maxUserParameterSize, methodCalled);
                }
                int remainingParamCapacity = maxUserParameters - verifiedParams.size();

                // verified custom parameters added via noticeError or transaction-less Span APIs should be kept
                // regardless if they occur in a transaction or not
                if (sendParametersOutsideOfTxn.contains(methodCalled)) {
                    // check that the max number of params hasn't been reached before adding more
                    if (remainingParamCapacity > 0) {
                        verifiedParams.put(currentKey, currentValue);
                    } else {
                        logParametersToDrop(verifiedParams, params);
                        return verifiedParams;
                    }
                } else {
                    // other APIs adding custom params must occur in a txn, return empty map early if not in txn
                    if (transactional && (tx == null || !tx.isInProgress())) {
                        Agent.LOG.log(Level.FINER,
                                "Unable to add {0} attributes with keys \"{1}\" because {2} was invoked outside a New Relic transaction.",
                                getAttributeType(), params.keySet(), methodCalled);
                        return Collections.emptyMap();
                    } else {
                        // check that the max number of params hasn't been reached before adding more
                        if (remainingParamCapacity > 0) {
                            verifiedParams.put(currentKey, currentValue);
                        } else {
                            logParametersToDrop(verifiedParams, params);
                            return verifiedParams;
                        }
                    }
                }
            }
        }
        return verifiedParams;
    }

    private void logParametersToDrop(Map<String, Object> verified, Map<String, Object> allParams) {
        MapDifference<String, Object> diff = Maps.difference(verified, allParams);
        // attribute map is at the max, return the map of params that can be added
        Agent.LOG.log(Level.FINER,
                "Unable to add attributes for keys \"{0}\" because the limit on {1} attributes has been reached.",
                diff.entriesOnlyOnRight().keySet(), getAttributeType());
    }

    private String truncateValue(String key, String value, int maxUserParameterSize, String methodCalled) {
        String truncatedVal = truncateString(value, maxUserParameterSize);
        if (!value.equals(truncatedVal)) {
            Agent.LOG.log(Level.FINER,
                    "{0} was invoked with a value longer than {2} bytes for key \"{3}\". The value will be shortened to the first {4} characters.",
                    methodCalled, value, maxUserParameterSize, key, truncatedVal.length());
        }
        return truncatedVal;
    }

    private boolean validateAndLogKeyLength(String key, int maxUserParameterSize, String methodCalled) {
        try {
            if (key.getBytes("UTF-8").length > maxUserParameterSize) {
                Agent.LOG.log(Level.FINER,
                        "Unable to add {0} attribute because {1} was invoked with a key longer than {2} bytes. Key is \"{3}\".",
                        getAttributeType(), methodCalled, maxUserParameterSize, key);
                return false;
            }
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINEST, "Exception while verifying attribute", t);
            return false;
        }
        return true;
    }

    /**
     * This function truncates a Unicode String so that it can be encoded in maxBytes. It uses the UTF-8 encoding to
     * determine where the String has to be truncated.
     *
     * @param s String to be truncated
     * @param maxBytes Maximum number of bytes in UTF-8 charset encoding
     * @return
     */
    public static String truncateString(String s, int maxBytes) {
        int truncatedSize = 0;

        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            // ranges from https://tools.ietf.org/html/rfc3629

            int characterSize; // Character size in bytes
            // first range: 0000 0000-0000 007F
            if (c <= 0x007f) {
                characterSize = 1;
            } else if (c <= 0x07FF) {
                // second range: 0000 0080-0000 07FF
                characterSize = 2;
            } else if (c <= 0xd7ff) {
                // third range: 0000 0800-0000 FFFF
                characterSize = 3;
            } else if (c <= 0xDFFF) {
                // fourth range: 0x10000 to 0x10FFFF
                // this is the surrogate area (D800 <= c <= DFFF) and is used
                // to encode the characters in the fourth range, which require
                // an additional character in the string buffer
                characterSize = 4;
                // skip the additional character used for encoding
                i++;
            } else {
                // remaining third range: DFFF < c <= FFFF
                characterSize = 3;
            }

            if (truncatedSize + characterSize > maxBytes) {
                return s.substring(0, i);
            }
            truncatedSize += characterSize;
        }
        return s;
    }

    /**
     * Add a key/value pair to the current transaction. These are reported in errors and transaction traces.
     */
    public void addAttribute(String key, String value, String methodName) {
        addCustomAttributeImpl(key, value, methodName);
    }

    /**
     * Add a key/value pair to the current transaction. These are reported in errors and transaction traces.
     */
    public void addAttribute(String key, Number value, String methodName) {
        addCustomAttributeImpl(key, value, methodName);
    }

    /**
     * Add a key/value pair to the current transaction. These are reported in errors and transaction traces.
     */
    public void addAttribute(String key, Boolean value, String methodName) {
        addCustomAttributeImpl(key, value, methodName);
    }

    /**
     * Add a key with a map of values to the current transaction. These are reported in errors and transaction traces.
     */
    public void addAttribute(String key, Map<String, String> values, String methodName) {
        addCustomAttributeImpl(key, values, methodName);
    }

    /**
     * Add a map of attributes to the current transaction. These are reported in Spans.
     */
    public void addAttributes(Map<String, Object> params, String methodName) {
        addCustomAttributesImpl(params, methodName);
    }
}