package com.newrelic.agent;

import java.util.concurrent.atomic.AtomicReference;

import com.newrelic.agent.bridge.AsyncApi;
import com.newrelic.api.agent.NewRelic;

public enum TransactionErrorPriority {
    /**
     * Set by the user via a call to {@link NewRelic#noticeError()} or {@link AsyncApi#errorAsync(Object, Throwable)}.
     * Only take the first.
     */
    API {
        @Override
        protected boolean updatePriority(AtomicReference<TransactionErrorPriority> current) {
            // Only take the first api level error.
            if (this == current.get()) {
                return false;
            }
            return current.compareAndSet(TRACER, this) || current.compareAndSet(ASYNC_POINTCUT, this);
        }
    },

    /**
     * Set by finishing a root tracer with a throwable. Last set wins.
     */
    TRACER {
        @Override
        protected boolean updatePriority(AtomicReference<TransactionErrorPriority> current) {
            // if TRACER, allow setting with the same priority, last set wins.
            if (this == current.get()) {
                return true;
            }
            return current.compareAndSet(ASYNC_POINTCUT, this);
        }
    },

    /**
     * Should only succeed on root transaction. Only take the first.
     */
    ASYNC_POINTCUT {
        @Override
        protected boolean updatePriority(AtomicReference<TransactionErrorPriority> current) {
            // Only take the first pointcut level error.
            return false;
        }
    };

    protected abstract boolean updatePriority(AtomicReference<TransactionErrorPriority> current);

    /**
     * ASYNC_POINTCUT < TRACER < API
     * 
     * "this" references the new priority. "other" is the existing priority.
     * 
     * @return value is whether to update the throwable field.
     */
    public boolean updateCurrentPriority(AtomicReference<TransactionErrorPriority> current) {
        // always set if null
        if (current.compareAndSet(null, this)) {
            return true;
        }
        return updatePriority(current);
    }
}
