package com.newrelic.agent;

import com.newrelic.agent.bridge.NoOpDistributedTracePayload;
import com.newrelic.agent.bridge.NoOpTracedMethod;
import com.newrelic.agent.bridge.TracedMethod;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.tracers.Tracer;
import com.newrelic.api.agent.DistributedTracePayload;
import com.newrelic.api.agent.ExternalParameters;
import com.newrelic.api.agent.OutboundHeaders;
import com.newrelic.api.agent.Transaction;

import java.util.concurrent.atomic.AtomicBoolean;

public class Segment implements com.newrelic.agent.bridge.TracedActivity {
    private volatile Tracer underlyingTracer;
    private volatile Tracer parent;
    private volatile WeakRefTransaction weakRefTransaction;

    private final long parentInitialExclusiveDuration;
    private final AtomicBoolean isFinished = new AtomicBoolean(false);
    private final String initiatingThread;

    public static final String UNNAMED_SEGMENT = "Unnamed Segment";
    public static final String START_THREAD = "start_thread";
    public static final String END_THREAD = "end_thread";


    /**
     * Construct a new {@link Segment}
     *
     * @param parent Parent Tracer of this {@link Segment}. Must not be null.
     * @param tracer Underlying Tracer supporting this {@link Segment}. Must not be null.
     */
    public Segment(Tracer parent, Tracer tracer) {
        this.initiatingThread = Thread.currentThread().getName();
        this.parent = parent;
        this.underlyingTracer = tracer;
        this.weakRefTransaction = new WeakRefTransaction(parent.getTransactionActivity().getTransaction());
        this.parentInitialExclusiveDuration = parent.getExclusiveDuration();
    }

    public Transaction getTransaction() {
        return weakRefTransaction;
    }

    @Override
    public void ignore() {
        ignoreIfUnfinished();
    }

    @Override
    public void reportAsExternal(ExternalParameters externalParameters) {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.reportAsExternal(externalParameters);
        }
    }

    @Override
    public void setMetricName(String... metricNameParts) {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.setMetricName(metricNameParts);
        }
    }

    @Override
    public void addOutboundRequestHeaders(OutboundHeaders outboundHeaders) {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.addOutboundRequestHeaders(outboundHeaders);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public TracedMethod getTracedMethod() {
        if (underlyingTracer == null) {
            return NoOpTracedMethod.INSTANCE;
        }
        return underlyingTracer;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setAsyncThreadName(String threadName) {
    }

    public Tracer getParent() {
        return parent;
    }

    public Tracer getTracer() {
        return underlyingTracer;
    }

    public long getParentInitialExclusiveDuration() {
        return this.parentInitialExclusiveDuration;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void ignoreIfUnfinished() {
        if (!isFinished.getAndSet(true)) {
            Tracer tracer = parent;
            if (tracer != null) {
                tracer.getTransactionActivity().getTransaction().ignoreSegmentIfUnfinished(this);
            }

            // Remove references to underlying and parent tracer to prevent GC issues
            underlyingTracer = null;
            parent = null;
            weakRefTransaction = null;
        }
    }

    @Override
    public void finish() {
        finish(null);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void end() {
        finish(null);
    }

    public DistributedTracePayload createDistributedTracePayload() {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            return ServiceFactory.getDistributedTraceService().createDistributedTracePayload(tracer);
        }
        return NoOpDistributedTracePayload.INSTANCE;
    }

    public void acceptDistributedTracePayload(String payload) {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.getTransactionActivity().getTransaction().acceptDistributedTracePayload(payload);
        }
    }

    public void acceptDistributedTracePayload(DistributedTracePayload payload) {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.getTransactionActivity().getTransaction().acceptDistributedTracePayload(payload);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void finish(final Throwable t) {
        if (!isFinished.getAndSet(true)) {
            markFinishTime();

            final Tracer tracer = parent;
            final Segment segment = this;
            final String endThreadName = Thread.currentThread().getName();

            if (tracer != null) {
                ServiceFactory.getAsyncExpirationService().expireSegment(new Runnable() {
                    @Override
                    public void run() {
                        tracer.getTransactionActivity().getTransaction().finishSegment(segment, t, parent, endThreadName);

                        // Remove references to underlying and parent tracer to prevent GC issues
                        underlyingTracer = null;
                        parent = null;
                        weakRefTransaction = null;
                    }
                });
            }

        }
    }

    public String getInitiatingThread() {
        return initiatingThread;
    }

    public void setTruncated() {
        Tracer tracer = underlyingTracer;
        if (tracer != null) {
            tracer.setMetricNameFormatInfo(tracer.getMetricName(),
                "Truncated/" + tracer.getMetricName(),
                tracer.getTransactionSegmentUri());
        }
    }

    private void markFinishTime() {
        Tracer underlyingTracer = this.underlyingTracer;
        if (underlyingTracer != null) {
            underlyingTracer.markFinishTime();
        }
    }
}
