package com.newrelic.agent;

import java.util.Set;

import com.google.common.collect.ImmutableSet;
import com.newrelic.api.agent.HeaderType;
import com.newrelic.api.agent.InboundHeaders;
import com.newrelic.api.agent.OutboundHeaders;

/**
 * Utility methods for consistent processing message headers regardless whether they arrive by HTTP or JMS. The
 * underlying issue is that HTTP header strings (e.g. "X-Some-Header") are syntactically illegal in JMS because dashes
 * are not allowed there. So when given a headers collection, we need to ask its type so we can deduce the key used to
 * retrieve the header value. Presumably we may support additional protocols in the future.
 */
public class HeadersUtil {
    private HeadersUtil() {
        // do not instance, even by reflection ;-)
        throw new UnsupportedOperationException();
    }

    /**
     * The request header for storing our cross-process id in external calls.
     */
    public static final String NEWRELIC_ID_HEADER = "X-NewRelic-ID";
    public static final String NEWRELIC_ID_MESSAGE_HEADER = "NewRelicID";

    /**
     * The request header for storing our transaction id in external calls.
     */
    public static final String NEWRELIC_TRANSACTION_HEADER = "X-NewRelic-Transaction";
    public static final String NEWRELIC_TRANSACTION_MESSAGE_HEADER = "NewRelicTransaction";

    /**
     * The response header for returning transaction data to clients.
     */
    public static final String NEWRELIC_APP_DATA_HEADER = "X-NewRelic-App-Data";
    public static final String NEWRELIC_APP_DATA_MESSAGE_HEADER = "NewRelicAppData";

    /**
     * The request header for tracing a transaction created by New Relic Synthetics
     */
    public static final String NEWRELIC_SYNTHETICS_HEADER = "X-NewRelic-Synthetics";
    public static final String NEWRELIC_SYNTHETICS_MESSAGE_HEADER = "NewRelicSynthetics";

    /**
     * The request header for storing our distributed trace payloads.
     */
    public static final String NEWRELIC_TRACE_HEADER = "newrelic";
    private static final String NEWRELIC_TRACE_HEADER_UPPER = "NEWRELIC";
    private static final String NEWRELIC_TRACE_HEADER_FIRST_CAPITALIZED = "Newrelic";

    public static final String NEWRELIC_TRACE_MESSAGE_HEADER = "newrelic";

    /**
     * The request header for storing distributed trace baggage.
     */
    public static final String NEWRELIC_BAGGAGE_HEADER = "X-NewRelic-Baggage";

    /**
     * Minimum supported version of New Relic Synthetics protocol.
     */
    public static final int SYNTHETICS_MIN_VERSION = 1;

    /**
     * Maximum supported version of New Relic Synthetics protocol.
     */
    public static final int SYNTHETICS_MAX_VERSION = 1;

    /**
     * Value that can never appear as a Synthetics protocol version.
     */
    public static final int SYNTHETICS_VERSION_NONE = -1;

    // Note - For now we do *not* want NEWRELIC_TRACE_* here because it is not an obfuscated header
    public static final Set<String> NEWRELIC_HEADERS = ImmutableSet.of(NEWRELIC_ID_HEADER, NEWRELIC_ID_MESSAGE_HEADER,
            NEWRELIC_TRANSACTION_HEADER, NEWRELIC_TRANSACTION_MESSAGE_HEADER, NEWRELIC_APP_DATA_HEADER,
            NEWRELIC_APP_DATA_MESSAGE_HEADER, NEWRELIC_SYNTHETICS_HEADER, NEWRELIC_SYNTHETICS_MESSAGE_HEADER);

    public static String getIdHeader(InboundHeaders headers) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_ID_HEADER, NEWRELIC_ID_MESSAGE_HEADER);
        return (key == null) ? null : headers.getHeader(key);
    }

    public static String getTransactionHeader(InboundHeaders headers) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRANSACTION_HEADER,
                NEWRELIC_TRANSACTION_MESSAGE_HEADER);
        return (key == null) ? null : headers.getHeader(key);
    }

    public static String getAppDataHeader(InboundHeaders headers) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_APP_DATA_HEADER,
                NEWRELIC_APP_DATA_MESSAGE_HEADER);
        return (key == null) ? null : headers.getHeader(key);
    }

    public static String getSyntheticsHeader(InboundHeaders headers) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_SYNTHETICS_HEADER,
                NEWRELIC_SYNTHETICS_MESSAGE_HEADER);
        return (key == null) ? null : headers.getHeader(key);
    }

    public static String getTraceHeader(InboundHeaders headers) {
        // Receiving app should check three different kinds of headers keys cases: lowercase, uppercase, first letter capitalized
        String lowercase = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRACE_HEADER, NEWRELIC_TRACE_MESSAGE_HEADER);
        if (lowercase != null) {
            String value = headers.getHeader(lowercase);
            if (value != null) {
                return value;
            }
        }

        String uppercase = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRACE_HEADER_UPPER, NEWRELIC_TRACE_MESSAGE_HEADER);
        if (uppercase != null) {
            String value = headers.getHeader(uppercase);
            if (value != null) {
                return value;
            }
        }

        String firstUppercase = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRACE_HEADER_FIRST_CAPITALIZED, NEWRELIC_TRACE_MESSAGE_HEADER);
        if (firstUppercase != null) {
            String value = headers.getHeader(firstUppercase);
            if (value != null) {
                return value;
            }
        }

        return null;
    }

    /**
     * HeaderWrapper utility methods.
     */

    public static void setIdHeader(OutboundHeaders headers, String crossProcessId) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_ID_HEADER, NEWRELIC_ID_MESSAGE_HEADER);
        headers.setHeader(key, crossProcessId);
    }

    public static void setTransactionHeader(OutboundHeaders headers, String value) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRANSACTION_HEADER,
                NEWRELIC_TRANSACTION_MESSAGE_HEADER);
        headers.setHeader(key, value);
    }

    public static void setAppDataHeader(OutboundHeaders headers, String value) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_APP_DATA_HEADER,
                NEWRELIC_APP_DATA_MESSAGE_HEADER);
        headers.setHeader(key, value);
    }

    public static void setSyntheticsHeader(OutboundHeaders headers, String value) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_SYNTHETICS_HEADER,
                NEWRELIC_SYNTHETICS_MESSAGE_HEADER);
        headers.setHeader(key, value);
    }

    public static void setTraceHeader(OutboundHeaders headers, String value) {
        String key = getTypedHeaderKey(headers.getHeaderType(), NEWRELIC_TRACE_HEADER,
                NEWRELIC_TRACE_MESSAGE_HEADER);
        headers.setHeader(key, value);
    }

    // Get the header key appropriate to the protocol (HTTP or MQ)
    private static String getTypedHeaderKey(HeaderType type, String httpHeader, String messageHeader) {
        if (type == null) {
            return null;
        }

        switch (type) {
        case MESSAGE:
            return messageHeader;

        case HTTP:
        default:
            return httpHeader;
        }
    }
}
