package com.newrelic.agent;

import java.io.UnsupportedEncodingException;

import com.newrelic.agent.util.Obfuscator;
import com.newrelic.api.agent.HeaderType;
import com.newrelic.api.agent.InboundHeaders;

/**
 * Wrapper that deobfuscates New Relic transaction headers.
 *
 */
public class DeobfuscatedInboundHeaders implements InboundHeaders {
    InboundHeaders delegate;
    String encodingKey;

    /**
     * 
     * @param headers inbound headers delegate to wrap. Must not be null.
     * @param encodingKey
     */
    public DeobfuscatedInboundHeaders(InboundHeaders headers, String encodingKey) {
        this.delegate = headers;
        this.encodingKey = encodingKey;
    }

    @Override
    public HeaderType getHeaderType() {
        return delegate.getHeaderType();
    }

    @Override
    public String getHeader(String name) {
        if (encodingKey == null && !isDistributedTraceHeader(name)) {
            return null;
        }

        if (HeadersUtil.NEWRELIC_HEADERS.contains(name)) {
            String obfuscatedValue = delegate.getHeader(name);
            if (obfuscatedValue == null) {
                obfuscatedValue = delegate.getHeader(name.toUpperCase());
            }
            if (obfuscatedValue == null) {
                obfuscatedValue = delegate.getHeader(name.toLowerCase());
            }
            if (obfuscatedValue == null) {
                return null;
            }
            try {
                return Obfuscator.deobfuscateNameUsingKey(obfuscatedValue, encodingKey);
            } catch (UnsupportedEncodingException e) {
                return null;
            }
        }

        String value = delegate.getHeader(name);
        if (value == null) {
            value = delegate.getHeader(name.toUpperCase());
        }
        if (value == null) {
            value = delegate.getHeader(name.toLowerCase());
        }

        return value;
    }

    private boolean isDistributedTraceHeader(String name) {
        return name.equalsIgnoreCase(HeadersUtil.NEWRELIC_TRACE_HEADER) ||
                name.equalsIgnoreCase(HeadersUtil.NEWRELIC_TRACE_MESSAGE_HEADER);
    }
}
