package com.newrelic.agent;

import com.google.common.collect.ImmutableMap;
import com.newrelic.agent.bridge.AgentBridge;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentJarHelper;
import com.newrelic.agent.config.ConfigService;
import com.newrelic.agent.config.JarResource;
import com.newrelic.agent.config.JavaVersionUtils;
import com.newrelic.agent.install.ConfigInstaller;
import com.newrelic.agent.logging.AgentLogManager;
import com.newrelic.agent.logging.IAgentLogger;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.ServiceManager;
import com.newrelic.agent.service.ServiceManagerImpl;
import com.newrelic.agent.stats.StatsService;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.util.asm.ClassStructure;
import com.newrelic.bootstrap.BootstrapLoader;
import com.newrelic.weave.utils.Streams;
import org.objectweb.asm.ClassReader;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.instrument.Instrumentation;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.regex.Pattern;

/**
 * New Relic Agent core class. The premain you see here is but a fleeting shadow of the true premain. The real premain,
 * called directly by the JVM in response to the -javaagent flag, can be found in BootstrapAgent.java.
 *
 * @author cirne
 * @author saxon
 * @author roger
 */
public final class Agent extends AbstractService implements IAgent {

    /**
     * Access to logger. Implementation note: logging is directed to the console until the AgentService is initialized
     * by the ServiceManager. This occurs during ServiceManager.start(). Prior to initialization, the log level is INFO.
     */
    public final static IAgentLogger LOG = AgentLogManager.getLogger();

    private final static String NEWRELIC_BOOTSTRAP = "newrelic-bootstrap";
    private final static String AGENT_ENABLED_PROPERTY = "newrelic.config.agent_enabled";

    private final static boolean DEBUG = Boolean.getBoolean("newrelic.debug");
    private final static String VERSION = Agent.initVersion();

    private static long agentPremainTime;

    private volatile boolean enabled = true;
    private final Instrumentation instrumentation;
    private volatile InstrumentationProxy instrumentationProxy;

    private Agent(Instrumentation instrumentation) {
        super(IAgent.class.getSimpleName());
        this.instrumentation = instrumentation;
    }

    @Override
    protected void doStart() {
        ConfigService configService = ServiceFactory.getConfigService();
        AgentConfig config = configService.getDefaultAgentConfig();
        AgentLogManager.configureLogger(config);

        logHostIp();
        LOG.info(MessageFormat.format("New Relic Agent v{0} is initializing...", Agent.getVersion()));

        enabled = config.isAgentEnabled();
        if (!enabled) {
            Agent.LOG.info("New Relic agent is disabled.");
        }

        if (config.liteMode()) {
            Agent.LOG.info("New Relic agent is running in lite mode. All instrumentation modules are disabled");
            StatsService statsService = ServiceFactory.getServiceManager().getStatsService();
            statsService.getMetricAggregator().incrementCounter(MetricNames.SUPPORTABILITY_LITE_MODE);
        }

        instrumentationProxy = InstrumentationProxy.getInstrumentationProxy(instrumentation);

        initializeBridgeApis();

        final long startTime = System.currentTimeMillis();
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                jvmShutdown(startTime);
            }
        };
        Thread shutdownThread = new Thread(runnable, "New Relic JVM Shutdown");
        Runtime.getRuntime().addShutdownHook(shutdownThread);
    }

    private void initializeBridgeApis() {
        com.newrelic.api.agent.NewRelicApiImplementation.initialize();
        com.newrelic.agent.PrivateApiImpl.initialize(Agent.LOG);
    }

    /**
     * Logs the host and the IP address of the server currently running this agent.
     */
    private void logHostIp() {
        try {
            InetAddress address = InetAddress.getLocalHost();
            Agent.LOG.info("Agent Host: " + address.getHostName() + " IP: " + address.getHostAddress());
        } catch (UnknownHostException e) {
            Agent.LOG.info("New Relic could not identify host/ip.");
        }
    }

    @Override
    protected void doStop() {
    }

    @Override
    public void shutdownAsync() {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                shutdown();
            }
        };
        Thread shutdownThread = new Thread(runnable, "New Relic Shutdown");
        shutdownThread.start();
    }

    private void jvmShutdown(long startTime) {
        getLogger().fine("Agent JVM shutdown hook: enter.");

        AgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();

        // Only add the "transaction wait" shutdown hook if one has been configured
        if (config.waitForTransactionsInMillis() > 0) {
            getLogger().fine("Agent JVM shutdown hook: waiting for transactions to finish");

            // While there are still transactions in progress and we haven't hit the configured timeout keep
            // waiting and checking for the transactions to finish before allowing the shutdown to continue.
            long finishTime = System.currentTimeMillis() + config.waitForTransactionsInMillis();
            TransactionService txService = ServiceFactory.getTransactionService();
            while (txService.getTransactionsInProgress() > 0 && System.currentTimeMillis() < finishTime) {
                try {
                    Thread.sleep(10);
                } catch (InterruptedException e) {
                }
            }

            getLogger().fine("Agent JVM shutdown hook: transactions finished");
        }

        if (config.isSendDataOnExit()
                && ((System.currentTimeMillis() - startTime) >= config.getSendDataOnExitThresholdInMillis())) {
            // Grab all RPMService instances (may be multiple with auto_app_naming enabled) and harvest them
            List<IRPMService> rpmServices = ServiceFactory.getRPMServiceManager().getRPMServices();
            for (IRPMService rpmService : rpmServices) {
                rpmService.harvestNow();
            }
        }

        getLogger().info("JVM is shutting down");
        shutdown();
        getLogger().fine("Agent JVM shutdown hook: done.");
    }

    private synchronized void shutdown() {
        try {
            ServiceFactory.getServiceManager().stop();
            getLogger().info("New Relic Agent has shutdown");
        } catch (Throwable t) {
            Agent.LOG.log(Level.SEVERE, t, "Error shutting down New Relic Agent");
        }
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    @Override
    public InstrumentationProxy getInstrumentation() {
        return instrumentationProxy;
    }

    public static String getVersion() {
        return VERSION;
    }

    /**
     * Get the agent version from Agent.properties.
     */
    private static String initVersion() {
        try {
            ResourceBundle bundle = ResourceBundle.getBundle(Agent.class.getName());
            return bundle.getString("version");
        } catch (Throwable t) {
            // ignore
        }
        return "0.0";
    }

    public static boolean isDebugEnabled() {
        return DEBUG;
    }

    private static volatile boolean canFastPath = true;

    /**
     * Return true if this Agent can use "fast path" optimizations.
     *
     * @return true if this Agent can use "fast path" optimizations. Fast path optimizations cannot be enabled if
     * certain legacy instrumentation is in use (and possibly for other reasons as well).
     */
    public static boolean canFastPath() {
        return canFastPath;
    }

    /**
     * Disable "fast path" optimizations for the lifetime of this Agent.
     */
    public static void disableFastPath() {
        // The pre-check on the volatile variable is *believed* to be important for performance, but this has
        // yet to be confirmed by experiment using JMH. This particular variable implements a kind of latch:
        // that changes state at most once during the lifetime of the JVM. Therefore, it only needs to be modified
        // if it's in the default state, and then never again. In this special case, the hypothesis is that we
        // retain the virtues of read-sharing for the cache line containing the volatile if we always check
        // before writing to it.
        if (canFastPath) {
            canFastPath = false;
        }
    }

    /**
     * Called by the "real" premain() in the BootstrapAgent.
     */
    public static void premain(String agentArgs, Instrumentation inst, long startTime) {
        // This *MUST* be done first thing in the premain
        addMixinInterfacesToBootstrap(inst);

        // Although the logger is statically initialized at the top of this class, it will only write to the standard
        // output until it is configured. This occurs within ServiceManager.start() via a call back to the doStart()
        // method, above in this class. The "log" statements below go only to the console. During this time, the logger
        // runs at INFO level. We could use println here, but it's easier to keep the LOG calls rather than duplicating
        // their textual output format.

        if (ServiceFactory.getServiceManager() != null) {
            LOG.warning("New Relic Agent is already running! Check if more than one -javaagent switch is used on the command line.");
            return;
        }
        String enabled = System.getProperty(AGENT_ENABLED_PROPERTY);
        if (enabled != null && !Boolean.parseBoolean(enabled.toString())) {
            LOG.warning("New Relic Agent is disabled by a system property.");
            return;
        }
        String jvmName = System.getProperty("java.vm.name");
        if (jvmName.contains("Oracle JRockit")) {
            String msg = MessageFormat.format(
                    "New Relic Agent {0} does not support the Oracle JRockit JVM (\"{1}\").", Agent.getVersion(), jvmName);
            LOG.warning(msg);
        }
        try {
            ServiceManager serviceManager;
            try {
                IAgent agent = new Agent(inst);
                serviceManager = new ServiceManagerImpl(agent);
                ServiceFactory.setServiceManager(serviceManager);

                if (ConfigInstaller.isLicenseKeyEmpty(
                        serviceManager.getConfigService().getDefaultAgentConfig().getLicenseKey())) {
                    LOG.error("license_key is empty in the config. Not starting New Relic Agent.");
                    return;
                }

                if (!serviceManager.getConfigService().getDefaultAgentConfig().isAgentEnabled()) {
                    LOG.warning("agent_enabled is false in the config. Not starting New Relic Agent.");
                    return;
                }

                // Now that we know the agent is enabled, add the ApiClassTransformer
                BootstrapLoader.forceCorrectNewRelicApi(inst);

                // init problem classes before class transformer service is active
                InitProblemClasses.loadInitialClasses();
            } catch (ForceDisconnectException e) {
                /* Note: Our use of ForceDisconnectException is a bit misleading here as we haven't even tried to connect
                 * to RPM at this point (that happens a few lines down when we call serviceManager.start()). This exception
                 * comes from ConfigServiceFactory when it attempts to validate the local yml and finds that both HSM and
                 * LASP are enabled. The LASP spec says in this scenario that "Shutdown will follow the behavior of the
                 * ForceDisconnectException response from "New Relic." Not specifically that we should throw ForceDisconnectException.
                 * Perhaps we should throw a different, more accurately named exception, that simply has the same behavior
                 * as ForceDisconnectException as it will be replaced by a 410 response code in Protocol 17.
                 */
                LOG.log(Level.SEVERE, e.getMessage());
                return;
            } catch (Throwable t) {
                // this is the last point where we can stop the agent gracefully if something has gone wrong.
                LOG.log(Level.SEVERE, t,
                        "Unable to start the New Relic Agent. Your application will continue to run but it will not be monitored.");
                return;
            }

            // The following method will immediately configure the log so that the rest of our initialization sequence
            // is written to the newrelic_agent.log rather than to the console. Configuring the log also applies the
            // log_level setting from the newrelic.yml so debugging levels become available here, if so configured.

            serviceManager.start();

            LOG.info(MessageFormat.format("New Relic Agent v{0} has started", Agent.getVersion()));

            if (System.getProperty("newrelic.bootstrap_classpath") != null) {
                // This is an obsolete system property that caused the entire Agent to be loaded on the bootstrap.
                LOG.info("The \"newrelic.bootstrap_classpath\" property is no longer used. Please remove it from your configuration.");
            }
            LOG.info("Agent class loader: " + AgentBridge.getAgent().getClass().getClassLoader());

            String javaEndorsedDirs = System.getProperty("java.endorsed.dirs");
            // The classes in this dir will be loaded directly onto the bootstrap class loader, which may cause
            // NoClassDefFoundError's to occur. See https://newrelic.atlassian.net/browse/JAVA-3958
            if (javaEndorsedDirs != null && !javaEndorsedDirs.isEmpty()) {
                try {
                    // Split out each directory using the path separator
                    String[] endorsedDirs = javaEndorsedDirs.split(String.valueOf(File.pathSeparatorChar));
                    for (String endorsedDir : endorsedDirs) {
                        File endorsedDirFile = new File(endorsedDir);
                        // If the path exists and it's a directory we need to see if there are any files in that directory
                        if (endorsedDirFile.exists() && endorsedDirFile.isDirectory()) {
                            File[] files = endorsedDirFile.listFiles();
                            if (files != null && files.length > 0) {
                                // The directory has at least one file in it, log a warning about it
                                LOG.log(Level.WARNING, "The 'java.endorsed.dirs' system property is set to {0} and not empty for this jvm. " +
                                        "This may cause unexpected behavior.", endorsedDir);
                            }
                        }
                    }
                } catch (Throwable t) {
                    LOG.log(Level.FINE, t, "An unexpected error occurred while checking for java.endorsed.dirs property");
                }
            }

            if (serviceManager.getConfigService().getDefaultAgentConfig().isStartupTimingEnabled()) {
                recordPremainTime(serviceManager.getStatsService(), startTime);
            }
        } catch (Throwable t) {
            // There's no way to gracefully pull the agent out due to our bytecode modification and class structure changes (pointcuts).
            // We're likely to throw an exception into the user's app if we try to continue.
            String msg = "Unable to start New Relic Agent. Please remove -javaagent from your startup arguments and contact New Relic support.";
            try {
                LOG.log(Level.SEVERE, t, msg);
            } catch (Throwable t2) {
                // ignore
            }
            System.err.println(msg);

            if (t instanceof NoClassDefFoundError) {
                String version = System.getProperty("java.version");
                if (version.startsWith("9") || version.startsWith("10")) {
                    String message = "We currently do not support Java 9+ in modular mode. If you are running with " +
                            "it and want to use the agent, use command line flag '--add-modules' to add appropriate modules";
                    System.err.println(message);
                } else if (version.startsWith("11")) {
                    String message = "Applications that previously relied on the command line flag '--add-modules' will no longer work with Java EE " +
                            "dependencies. You must add all Java EE dependencies to your build file manually, and then remove the --add-modules flag for them.";
                    System.err.println(message);
                }
            }
            t.printStackTrace();
            System.exit(1);
        }
    }

    public static void main(String[] args) {
        String javaSpecVersion = JavaVersionUtils.getJavaSpecificationVersion();
        if (!JavaVersionUtils.isAgentSupportedJavaSpecVersion(javaSpecVersion)) {
            System.err.println("----------");
            System.err.println(JavaVersionUtils.getUnsupportedAgentJavaSpecVersionMessage(javaSpecVersion));
            System.err.println("----------");
            return;
        }

        new AgentCommandLineParser().parseCommand(args);
    }

    public static long getAgentPremainTimeInMillis() {
        return agentPremainTime;
    }

    private static void recordPremainTime(StatsService statsService, long startTime) {
        agentPremainTime = System.currentTimeMillis() - startTime;
        LOG.log(Level.INFO, "Premain startup complete in {0}ms", agentPremainTime);
        statsService.doStatsWork(StatsWorks.getRecordResponseTimeWork(MetricNames.SUPPORTABILITY_TIMING_PREMAIN,
                agentPremainTime));

        Map<String, Object> environmentInfo = ImmutableMap.<String, Object>builder()
                .put("Duration", agentPremainTime)
                .put("Version", getVersion())
                .put("JRE Vendor", System.getProperty("java.vendor"))
                .put("JRE Version", System.getProperty("java.version"))
                .put("JVM Vendor", System.getProperty("java.vm.vendor"))
                .put("JVM Version", System.getProperty("java.vm.version"))
                .put("JVM Runtime Version", System.getProperty("java.runtime.version"))
                .put("OS Name", System.getProperty("os.name"))
                .put("OS Version", System.getProperty("os.version"))
                .put("OS Arch", System.getProperty("os.arch"))
                .put("Processors", Runtime.getRuntime().availableProcessors())
                .put("Free Memory", Runtime.getRuntime().freeMemory())
                .put("Total Memory", Runtime.getRuntime().totalMemory())
                .put("Max Memory", Runtime.getRuntime().maxMemory()).build();
        LOG.log(Level.FINE, "Premain environment info: {0}", environmentInfo.toString());
    }

    /**
     * Extract all the mixins from the Agent jar and add them to the bootstrap classloader. JAVA-609.
     *
     * @param inst the JVM instrumentation interface
     */
    private static void addMixinInterfacesToBootstrap(Instrumentation inst) {
        if (isDisableMixinsOnBootstrap()) {
            System.out.println("New Relic Agent: mixin interfaces not moved to bootstrap");
            return;
        }
        JarResource agentJarResource = null;
        URL agentJarUrl;
        try {
            agentJarResource = AgentJarHelper.getAgentJarResource();
            agentJarUrl = AgentJarHelper.getAgentJarUrl();
            addMixinInterfacesToBootstrap(agentJarResource, agentJarUrl, inst);
        } finally {
            try {
                agentJarResource.close();
            } catch (Throwable th) {
                logIfNRDebug("closing Agent jar resource", th);
            }
        }
    }

    /**
     * Extract all the mixins from the Agent jar and add them to the bootstrap classloader. JAVA-609.
     * <p>
     * Implementation note: In addition to the mixin interfaces themselves, a small number of dependent classes are
     * extracted into the generated jar and loaded on the bootstrap. These are marked with the LoadOnBootstrap
     * annotation. One class so marked is the InterfaceMixin annotation class itself.
     * <p>
     * This method duplicates some of the functionality found in the ClassAppender class in the Weaver. This duplication
     * is intentional. This code runs before bootstrap classpath setup is complete. Attempting to reuse the
     * ClassAppender causes one or more classes to be loaded by the "wrong" classloader as described in the top comment
     * to this class. This could be fixed, but would invite later failures during code maintenance if dependencies were
     * re-introduced. It is safer to ensure that code used under these special conditions remains right here.
     *
     * @param agentJarResource the Agent's jar file, or a test jar file for unit testing.
     * @param agentJarUrl the Agent's jar URL, or a test URL for unit testing.
     * @param inst the JVM instrumentation interface, or a mock for unit testing.
     */
    public static void addMixinInterfacesToBootstrap(JarResource agentJarResource, URL agentJarUrl, Instrumentation inst) {
        boolean succeeded = false;
        final Pattern packageSearchPattern = Pattern.compile("com/newrelic/agent/instrumentation/pointcuts/(.*).class");

        // Don't be tempted to try something like this, either:
        //
        // Class<?> interfaceMixinClass = InterfaceMixin.class;
        // String interfaceMixinAnnotation = 'L' + interfaceMixinClass.getName().replace('.', '/') + ';';
        //
        // ... it will defeat our purpose by pulling class InterfaceMixin to "this" class loader. Instead:

        final String interfaceMixinAnnotation = "Lcom/newrelic/agent/instrumentation/pointcuts/InterfaceMixin;";
        final String loadOnBootstrapAnnotation = "Lcom/newrelic/agent/instrumentation/pointcuts/LoadOnBootstrap;";
        final String interfaceMapperAnnotation = "Lcom/newrelic/agent/instrumentation/pointcuts/InterfaceMapper;";
        final String methodMapperAnnotation = "Lcom/newrelic/agent/instrumentation/pointcuts/MethodMapper;";
        final String fieldAccessorAnnotation = "Lcom/newrelic/agent/instrumentation/pointcuts/FieldAccessor;";

        final List<String> bootstrapAnnotations = Arrays.asList(new String[] { interfaceMixinAnnotation,
                interfaceMapperAnnotation, methodMapperAnnotation, fieldAccessorAnnotation, loadOnBootstrapAnnotation });

        File generatedFile = null;
        JarOutputStream outputJarStream = null;
        try {
            generatedFile = File.createTempFile(NEWRELIC_BOOTSTRAP, ".jar", BootstrapLoader.getTempDir());
            Manifest manifest = createManifest();
            outputJarStream = createJarOutputStream(generatedFile, manifest);
            long modTime = System.currentTimeMillis();

            Collection<String> fileNames = AgentJarHelper.findJarFileNames(agentJarUrl, packageSearchPattern);
            for (String fileName : fileNames) {
                int size = (int) agentJarResource.getSize(fileName);
                ByteArrayOutputStream out = new ByteArrayOutputStream(size);
                Streams.copy(agentJarResource.getInputStream(fileName), out, size, true);
                byte[] classBytes = out.toByteArray();

                ClassReader cr = new ClassReader(classBytes);
                ClassStructure structure = ClassStructure.getClassStructure(cr, ClassStructure.CLASS_ANNOTATIONS);
                Collection<String> annotations = structure.getClassAnnotations().keySet();
                if (containsAnyOf(bootstrapAnnotations, annotations)) {
                    JarEntry entry = new JarEntry(fileName);
                    entry.setTime(modTime);
                    outputJarStream.putNextEntry(entry);
                    outputJarStream.write(classBytes);
                }
            }

            outputJarStream.closeEntry();
            succeeded = true;
        } catch (IOException iox) {
            logIfNRDebug("generating mixin jar file", iox);
        } finally {
            try {
                outputJarStream.close();
            } catch (Throwable th) {
                logIfNRDebug("closing outputJarStream", th);
            }
        }

        if (succeeded) {
            // And finally, tada ...
            JarFile jarFile = null;
            try {
                jarFile = new JarFile(generatedFile);
                inst.appendToBootstrapClassLoaderSearch(jarFile);
                generatedFile.deleteOnExit();
            } catch (IOException iox) {
                logIfNRDebug("adding dynamic mixin jar to bootstrap", iox);
            } finally {
                try {
                    jarFile.close();
                } catch (Throwable th) {
                    logIfNRDebug("closing generated jar file", th);
                }
            }
        }
    }

    // Return true if the second collection contains any member of the first collection.
    private static final boolean containsAnyOf(Collection<?> searchFor, Collection<?> searchIn) {
        for (Object key : searchFor) {
            if (searchIn.contains(key)) {
                return true;
            }
        }
        return false;
    }

    // The "newrelic.disable.mixins.on.bootstrap" flag prevents us from dynamically generating the
    // mixin interface jar file and placing it on the bootstrap classloader path. This restores
    // the "old" 3.x Agent behavior that was in effect from 3.0.x through 3.12.x. Again, since we
    // haven't initialized the Agent we cannot use the standard AgentConfig for this.
    private static final boolean isDisableMixinsOnBootstrap() {
        String newrelicDisableMixinsOnBootstrap = "newrelic.disable.mixins.on.bootstrap";
        return System.getProperty(newrelicDisableMixinsOnBootstrap) != null
                && Boolean.getBoolean(newrelicDisableMixinsOnBootstrap);
    }

    // Use of this method should be limited to serious error cases that would cause the Agent to
    // shut down if not caught.
    private static final void logIfNRDebug(String msg, Throwable th) {
        if (isDebugEnabled()) {
            System.out.println("While bootstrapping the Agent: " + msg + ": " + th.getStackTrace());
        }
    }

    private static final JarOutputStream createJarOutputStream(File jarFile, Manifest manifest) throws IOException {
        FileOutputStream outStream = new FileOutputStream(jarFile);
        return new java.util.jar.JarOutputStream(outStream, manifest);
    }

    private static final Manifest createManifest() {
        Manifest manifest = new Manifest();
        Attributes a = manifest.getMainAttributes();
        a.put(Attributes.Name.MANIFEST_VERSION, "1.0");
        a.put(Attributes.Name.IMPLEMENTATION_TITLE, "Interface Mixins");
        a.put(Attributes.Name.IMPLEMENTATION_VERSION, "1.0");
        a.put(Attributes.Name.IMPLEMENTATION_VENDOR, "New Relic");
        return manifest;
    }

}
