/*
 * Copyright 2016 - 2019 Neurotech MRC. http://neuromd.com/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.neuromd.neurosdk;

import android.content.Context;
import android.util.Log;
import java.util.List;

/**
 * Provides common methods for device scanning and
 */
public class DeviceEnumerator {
    private boolean mIsClosed = false;
	private final long mEnumeratorPtr;
    private final long mDeviceListChangedListenerPtr;

	public final SubscribersNotifier deviceListChanged = new SubscribersNotifier<>();

	static {
        System.loadLibrary("android-neurosdk");
    }

    public DeviceEnumerator(Context context, DeviceType deviceType) {
        mEnumeratorPtr = createDeviceEnumerator(context, deviceType);
        mDeviceListChangedListenerPtr = enumeratorSetDeviceListChangedCallback(mEnumeratorPtr);
    }

    public void finalize() throws Throwable {
        if (!mIsClosed) close();
        super.finalize();
    }
	
    public void close(){
        if (mIsClosed) return;
       mIsClosed = true;
       enumeratorUnsubscribeDeviceListChanged(mDeviceListChangedListenerPtr);
       enumeratorDelete(mEnumeratorPtr);
    }

    public List<DeviceInfo> devices() {
        throwIfClosed();
		return enumeratorGetDeviceList(mEnumeratorPtr);
    }

    public Device createDevice(DeviceInfo info) {
        throwIfClosed();
        return new Device(info, mEnumeratorPtr);
    }

    private void onDeviceListChanged() {
        if (mIsClosed) return;
        deviceListChanged.sendNotification(this, null);
    }

    private void throwIfClosed(){
        if (mIsClosed)
            throw new UnsupportedOperationException("DeviceEnumerator is closed");
    }

    private static native long createDeviceEnumerator(Context context, DeviceType type);
    private static native void enumeratorDelete(long enumeratorPtr);
    private native long enumeratorSetDeviceListChangedCallback(long enumeratorPtr);
    private static native List<DeviceInfo> enumeratorGetDeviceList(long enumeratorPtr);
    private static native void enumeratorUnsubscribeDeviceListChanged(long listenerHandle);
}