/*
 * Copyright 2016 - 2017 Neurotech MRC. http://neuromd.com/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.neuromd.neurosdk;

import java.util.Hashtable;

public class Device {
    static {
        System.loadLibrary("android-neurosdk");
    }

    public class DoubleChannelData {
        private final ChannelInfo mChannelInfo;
        private final double[] mDataArray;

        public ChannelInfo channelInfo() {
            return mChannelInfo;
        }

        public double[] dataArray() {
            return mDataArray;
        }

        public DoubleChannelData(double[] array, ChannelInfo info) {
            mDataArray = array;
            mChannelInfo = info;
        }
    }

    public class IntChannelData {
        private final ChannelInfo mChannelInfo;
        private final int[] mDataArray;

        public ChannelInfo channelInfo() {
            return mChannelInfo;
        }

        public int[] dataArray() {
            return mDataArray;
        }

        public IntChannelData(int[] array, ChannelInfo info) {
            mDataArray = array;
            mChannelInfo = info;
        }
    }

    private boolean mIsClosed = false;
    private final long mParamChangedListenerPtr;
    private final Hashtable<INotificationCallback<DoubleChannelData>, Long> mDoubleChannelListenerHandles = new Hashtable<>();
    private final Hashtable<INotificationCallback<IntChannelData>, Long> mIntChannelListenerHandles = new Hashtable<>();

    private final SubscribersNotifier<DoubleChannelData> mDoubleChannelDataReceived = new SubscribersNotifier<>();
    private final SubscribersNotifier<IntChannelData> mIntChannelDataReceived = new SubscribersNotifier<>();

    private final long mDevicePtr;

    Device(DeviceInfo info, long enumeratorPtr){
		mDevicePtr = createDevice(info, enumeratorPtr);
        Assert.expects(mDevicePtr != 0, "Device pointer is null");
        mParamChangedListenerPtr = deviceSubscribeParamChanged(mDevicePtr);
    }

    public void finalize() throws Throwable {
        if (!mIsClosed) close();
        super.finalize();
    }

    /**
     * Subscribe this event to receive notifications about changes of device parameters
     */
    public final SubscribersNotifier<ParameterName> parameterChanged = new SubscribersNotifier<>();

    public long devicePtr(){
        throwIfClosed();
        return mDevicePtr;
    }

    /**
     * Tries to establish connection with device
     * Check DeviceState parameter or subscribe parameterChanged event for operation result
     */
    public void connect(){
        throwIfClosed();
        deviceConnect(mDevicePtr);
    }

    /**
     * Disconnects from device
     * Check DeviceState parameter or subscribe parameterChanged event for operation result
     */
    public void disconnect(){
        throwIfClosed();
        deviceDisconnect(mDevicePtr);
    }

    /**
     * Returns information about supported channels
     * Check this information before creation of channel. If device does not support channel,
     * channel object won't be initialized with it
     * @return Array of channel info objects
     */
    public ChannelInfo[] channels(){
        throwIfClosed();
        return deviceAvailableChannels(mDevicePtr);
    }

    /**
     * Returns supported commands of device
     * @return Array of supported commands
     */
    public Command[] commands(){
        throwIfClosed();
        return deviceAvailableCommands(mDevicePtr);
    }

    /**
     * Returns all available parameters of device, their types and access rights
     * @return Array of available parameters
     */
    public Parameter[] parameters(){
        throwIfClosed();
        return deviceAvailableParameters(mDevicePtr);
    }

    /**
     * Tries to execute command and returns value indicating operations success. Will throw if
     * device does not support specified command. To get supported commands call commands() method
     * @param cmd Command to execute
     * @return Operation success indicator
     * @throws UnsupportedOperationException
     */
    public void execute(Command cmd){
        throwIfClosed();
        deviceExecute(mDevicePtr, cmd);
    }

    /**
     * Return value of specified parameter of device. Will throw if parameter does not present in
     * device. To get supported parameters and type information for parameter call parameters()
     * method. It returns Parameter object which consists of parameter name, type and access mode
     * @param param ParameterName to read
     * @return Parameter value
     * @throws UnsupportedOperationException
     */
    public <ParamType> ParamType readParam(ParameterName param){
        throwIfClosed();
        ParameterTypeInfo paramTypeInfo = new ParameterTypeInfo(param);
        Object paramValue = paramTypeInfo.paramReader().readParam(this);
        return (ParamType)paramValue;
    }

    /**
     * Sets value for specified parameter and returns value indicating success of operation. Will
     * throw if parameter does not present in device or has only Read access mode. To get supported
     * parameters and type information for parameter call parameters() method. It returns Parameter
     * object which consists of parameter name, type and access mode
     * @param param Name of parameter to set
     * @param value Parameter value
     * @return Operation success
     * @throws UnsupportedOperationException
     */
    public boolean setParam(ParameterName param, Object value){
        throwIfClosed();
        ParameterTypeInfo paramTypeInfo = new ParameterTypeInfo(param);
        paramTypeInfo.paramSetter().setParam(this, value);
        return true;
    }
    public void addDoubleChannelDataListener(INotificationCallback<DoubleChannelData> callback, ChannelInfo channelInfo){
        throwIfClosed();
        if (callback == null) return;
        long listenerHandle = deviceSubscribeDoubleChannelDataReceived(mDevicePtr, channelInfo);
        long previous = mDoubleChannelListenerHandles.put(callback, listenerHandle);
        if (previous != 0){
            freeListenerHandle(previous);
        }
        mDoubleChannelDataReceived.subscribe(callback);
    }

    public void addIntChannelDataListener(INotificationCallback<IntChannelData> callback, ChannelInfo channelInfo){
        throwIfClosed();
        if (callback == null) return;
        long listenerHandle = deviceSubscribeIntChannelDataReceived(mDevicePtr, channelInfo);
        long previous = mIntChannelListenerHandles.put(callback, listenerHandle);
        if (previous != 0){
            freeListenerHandle(previous);
        }
        mIntChannelDataReceived.subscribe(callback);
    }

    public void removeIntChannelDataListener(INotificationCallback<IntChannelData> callback){
        throwIfClosed();
        if (mIntChannelListenerHandles.containsKey(callback))
        {
            freeListenerHandle(mIntChannelListenerHandles.get(callback));
            mIntChannelListenerHandles.remove(callback);
        }
    }

    public void removeDoubleChannelDataListener(INotificationCallback<DoubleChannelData> callback){
        throwIfClosed();
        if (mDoubleChannelListenerHandles.containsKey(callback)){
            freeListenerHandle(mDoubleChannelListenerHandles.get(callback));
            mDoubleChannelListenerHandles.remove(callback);
        }
    }

    public void close(){
       if (mIsClosed) return;
       mIsClosed = true;
       removeAllCahnnelListeners();
       freeListenerHandle(mParamChangedListenerPtr);
       deviceDelete(mDevicePtr);
    }

    private void removeAllCahnnelListeners(){
        for (long listener : mIntChannelListenerHandles.values())
        {
            freeListenerHandle(listener);
        }
        mIntChannelListenerHandles.clear();
        for (long listener : mDoubleChannelListenerHandles.values())
        {
            freeListenerHandle(listener);
        }
        mDoubleChannelListenerHandles.clear();
    }

    private void onParameterChanged(long devicePtr, ParameterName parameterName){
        if (mIsClosed) return;
        if (mDevicePtr != devicePtr) return;
        parameterChanged.sendNotification(this, parameterName);
    }

    private void onDoubleDataReceived(long devicePtr, ChannelInfo info, double[] dataArray){
        if (mIsClosed) return;
        if (mDevicePtr != devicePtr) return;
        mDoubleChannelDataReceived.sendNotification(this, new DoubleChannelData(dataArray, info));
    }

    private void onIntDataReceived(long devicePtr, ChannelInfo info, int[] dataArray){
        if (mIsClosed) return;
        if (mDevicePtr != devicePtr) return;
        mIntChannelDataReceived.sendNotification(this, new IntChannelData(dataArray, info));
    }

    private void throwIfClosed(){
        if (mIsClosed)
            throw new UnsupportedOperationException("Device is closed");
    }

    private static native void freeListenerHandle(long listenerHandle);
    private static native long createDevice(DeviceInfo deviceInfo, long enumeratorPtr);
    private static native void deviceConnect(long devicePtr);
    private static native void deviceDisconnect(long devicePtr);
    private static native void deviceDelete(long devicePtr);
    private static native ChannelInfo[] deviceAvailableChannels(long devicePtr);
    private static native Command[] deviceAvailableCommands(long devicePtr);
    private static native Parameter[] deviceAvailableParameters(long devicePtr);
    private static native void deviceExecute(long devicePtr, Command command);
    private native long deviceSubscribeParamChanged(long devicePtr);
    private native long deviceSubscribeDoubleChannelDataReceived(long devicePtr, ChannelInfo info);
    private native long deviceSubscribeIntChannelDataReceived(long devicePtr, ChannelInfo info);
}
