/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.oauthbearer.secured;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.kafka.common.security.oauthbearer.secured.Initable;
import org.apache.kafka.common.security.oauthbearer.secured.Retry;
import org.apache.kafka.common.utils.Time;
import org.jose4j.jwk.HttpsJwks;
import org.jose4j.jwk.JsonWebKey;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RefreshingHttpsJwks
implements Initable,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(RefreshingHttpsJwks.class);
    private static final int MISSING_KEY_ID_CACHE_MAX_ENTRIES = 16;
    static final long MISSING_KEY_ID_CACHE_IN_FLIGHT_MS = 60000L;
    static final int MISSING_KEY_ID_MAX_KEY_LENGTH = 1000;
    private static final int SHUTDOWN_TIMEOUT = 10;
    private static final TimeUnit SHUTDOWN_TIME_UNIT = TimeUnit.SECONDS;
    private final HttpsJwks httpsJwks;
    private final ScheduledExecutorService executorService;
    private final Time time;
    private final long refreshMs;
    private final long refreshRetryBackoffMs;
    private final long refreshRetryBackoffMaxMs;
    private final ReadWriteLock refreshLock = new ReentrantReadWriteLock();
    private final Map<String, Long> missingKeyIds;
    private final AtomicBoolean refreshInProgressFlag = new AtomicBoolean(false);
    private List<JsonWebKey> jsonWebKeys;
    private boolean isInitialized;

    public RefreshingHttpsJwks(Time time, HttpsJwks httpsJwks, long refreshMs, long refreshRetryBackoffMs, long refreshRetryBackoffMaxMs) {
        if (refreshMs <= 0L) {
            throw new IllegalArgumentException("JWKS validation key refresh configuration value retryWaitMs value must be positive");
        }
        this.httpsJwks = httpsJwks;
        this.time = time;
        this.refreshMs = refreshMs;
        this.refreshRetryBackoffMs = refreshRetryBackoffMs;
        this.refreshRetryBackoffMaxMs = refreshRetryBackoffMaxMs;
        this.executorService = Executors.newSingleThreadScheduledExecutor();
        this.missingKeyIds = new LinkedHashMap<String, Long>(16, 0.75f, true){

            @Override
            protected boolean removeEldestEntry(Map.Entry<String, Long> eldest) {
                return this.size() > 16;
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void init() throws IOException {
        try {
            List<JsonWebKey> localJWKs;
            log.debug("init started");
            try {
                localJWKs = this.httpsJwks.getJsonWebKeys();
            }
            catch (JoseException e) {
                throw new IOException("Could not refresh JWKS", e);
            }
            try {
                this.refreshLock.writeLock().lock();
                this.jsonWebKeys = Collections.unmodifiableList(localJWKs);
            }
            finally {
                this.refreshLock.writeLock().unlock();
            }
            this.executorService.scheduleAtFixedRate(this::refresh, this.refreshMs, this.refreshMs, TimeUnit.MILLISECONDS);
            log.info("JWKS validation key refresh thread started with a refresh interval of {} ms", (Object)this.refreshMs);
        }
        finally {
            this.isInitialized = true;
            log.debug("init completed");
        }
    }

    @Override
    public void close() {
        try {
            log.debug("close started");
            try {
                log.debug("JWKS validation key refresh thread shutting down");
                this.executorService.shutdown();
                if (!this.executorService.awaitTermination(10L, SHUTDOWN_TIME_UNIT)) {
                    log.warn("JWKS validation key refresh thread termination did not end after {} {}", (Object)10, (Object)SHUTDOWN_TIME_UNIT);
                }
            }
            catch (InterruptedException e) {
                log.warn("JWKS validation key refresh thread error during close", e);
            }
        }
        finally {
            log.debug("close completed");
        }
    }

    public List<JsonWebKey> getJsonWebKeys() throws JoseException, IOException {
        if (!this.isInitialized) {
            throw new IllegalStateException("Please call init() first");
        }
        try {
            this.refreshLock.readLock().lock();
            List<JsonWebKey> list = this.jsonWebKeys;
            return list;
        }
        finally {
            this.refreshLock.readLock().unlock();
        }
    }

    public String getLocation() {
        return this.httpsJwks.getLocation();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refresh() {
        if (!this.refreshInProgressFlag.compareAndSet(false, true)) {
            log.debug("OAuth JWKS refresh is already in progress; ignoring concurrent refresh");
            return;
        }
        try {
            log.info("OAuth JWKS refresh of {} starting", (Object)this.httpsJwks.getLocation());
            Retry<List> retry = new Retry<List>(this.refreshRetryBackoffMs, this.refreshRetryBackoffMaxMs);
            List localJWKs = retry.execute(() -> {
                try {
                    log.debug("JWKS validation key calling refresh of {} starting", (Object)this.httpsJwks.getLocation());
                    this.httpsJwks.refresh();
                    List<JsonWebKey> jwks = this.httpsJwks.getJsonWebKeys();
                    log.debug("JWKS validation key refresh of {} complete", (Object)this.httpsJwks.getLocation());
                    return jwks;
                }
                catch (Exception e) {
                    throw new ExecutionException(e);
                }
            });
            try {
                this.refreshLock.writeLock().lock();
                for (JsonWebKey jwk : localJWKs) {
                    this.missingKeyIds.remove(jwk.getKeyId());
                }
                this.jsonWebKeys = Collections.unmodifiableList(localJWKs);
            }
            finally {
                this.refreshLock.writeLock().unlock();
            }
            log.info("OAuth JWKS refresh of {} complete", (Object)this.httpsJwks.getLocation());
        }
        catch (ExecutionException e) {
            log.warn("OAuth JWKS refresh of {} encountered an error; not updating local JWKS cache", (Object)this.httpsJwks.getLocation(), (Object)e);
        }
        finally {
            this.refreshInProgressFlag.set(false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean maybeExpediteRefresh(String keyId) {
        if (keyId.length() > 1000) {
            int actualLength = keyId.length();
            String s2 = keyId.substring(0, 1000);
            String snippet = String.format("%s (trimmed to first %s characters out of %s total)", s2, 1000, actualLength);
            log.warn("Key ID {} was too long to cache", (Object)snippet);
            return false;
        }
        try {
            this.refreshLock.writeLock().lock();
            Long nextCheckTime = this.missingKeyIds.get(keyId);
            long currTime = this.time.milliseconds();
            log.debug("For key ID {}, nextCheckTime: {}, currTime: {}", keyId, nextCheckTime, currTime);
            if (nextCheckTime == null || nextCheckTime <= currTime) {
                nextCheckTime = currTime + 60000L;
                this.missingKeyIds.put(keyId, nextCheckTime);
                this.executorService.schedule(this::refresh, 0L, TimeUnit.MILLISECONDS);
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        finally {
            this.refreshLock.writeLock().unlock();
        }
    }
}

