/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.discovery;

import com.google.common.base.Strings;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import com.netflix.appinfo.AmazonInfo;
import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.appinfo.DataCenterInfo;
import com.netflix.appinfo.HealthCheckCallback;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.config.DynamicPropertyFactory;
import com.netflix.discovery.AbstractAzToRegionMapper;
import com.netflix.discovery.BackupRegistry;
import com.netflix.discovery.DNSBasedAzToRegionMapper;
import com.netflix.discovery.DiscoveryManager;
import com.netflix.discovery.EurekaClientConfig;
import com.netflix.discovery.InstanceRegionChecker;
import com.netflix.discovery.PropertyBasedAzToRegionMapper;
import com.netflix.discovery.StatusChangeEvent;
import com.netflix.discovery.shared.Application;
import com.netflix.discovery.shared.Applications;
import com.netflix.discovery.shared.EurekaJerseyClient;
import com.netflix.discovery.shared.LookupService;
import com.netflix.eventbus.spi.EventBus;
import com.netflix.servo.monitor.Counter;
import com.netflix.servo.monitor.Monitors;
import com.netflix.servo.monitor.Stopwatch;
import com.netflix.servo.monitor.Timer;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.filter.ClientFilter;
import com.sun.jersey.api.client.filter.GZIPContentEncodingFilter;
import com.sun.jersey.client.apache4.ApacheHttpClient4;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimerTask;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import javax.annotation.PreDestroy;
import javax.inject.Singleton;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class DiscoveryClient
implements LookupService {
    private static final Logger logger = LoggerFactory.getLogger(DiscoveryClient.class);
    private static final DynamicPropertyFactory configInstance = DynamicPropertyFactory.getInstance();
    private static final String DNS_PROVIDER_URL = "dns:";
    private static final String DNS_NAMING_FACTORY = "com.sun.jndi.dns.DnsContextFactory";
    private static final String JAVA_NAMING_FACTORY_INITIAL = "java.naming.factory.initial";
    private static final String JAVA_NAMING_PROVIDER_URL = "java.naming.provider.url";
    private static final String DNS_RECORD_TYPE = "TXT";
    private static final String VALUE_DELIMITER = ",";
    private static final String COMMA_STRING = ",";
    private static final String DISCOVERY_APPID = "DISCOVERY";
    private static final String UNKNOWN = "UNKNOWN";
    private static final DirContext dirContext = DiscoveryClient.getDirContext();
    private static final String PREFIX = "DiscoveryClient_";
    private final Timer GET_SERVICE_URLS_DNS_TIMER = Monitors.newTimer((String)"DiscoveryClient_GetServiceUrlsFromDNS");
    private final Timer REGISTER_TIMER = Monitors.newTimer((String)"DiscoveryClient_Register");
    private final Timer REFRESH_TIMER = Monitors.newTimer((String)"DiscoveryClient_Refresh");
    private final Timer REFRESH_DELTA_TIMER = Monitors.newTimer((String)"DiscoveryClient_RefreshDelta");
    private final Timer RENEW_TIMER = Monitors.newTimer((String)"DiscoveryClient_Renew");
    private final Timer CANCEL_TIMER = Monitors.newTimer((String)"DiscoveryClient_Cancel");
    private final Timer FETCH_REGISTRY_TIMER = Monitors.newTimer((String)"DiscoveryClient_FetchRegistry");
    private final Counter SERVER_RETRY_COUNTER = Monitors.newCounter((String)"DiscoveryClient_Retry");
    private final Counter ALL_SERVER_FAILURE_COUNT = Monitors.newCounter((String)"DiscoveryClient_Failed");
    private final Counter REREGISTER_COUNTER = Monitors.newCounter((String)"DiscoveryClient_Reregister");
    private volatile HealthCheckCallback healthCheckCallback;
    private volatile AtomicReference<List<String>> eurekaServiceUrls = new AtomicReference();
    private volatile AtomicReference<Applications> localRegionApps = new AtomicReference();
    private volatile Map<String, Applications> remoteRegionVsApps = new ConcurrentHashMap<String, Applications>();
    private InstanceInfo instanceInfo;
    private String appPathIdentifier;
    private boolean isRegisteredWithDiscovery = false;
    private String discoveryServerAMIId;
    private EurekaJerseyClient.JerseyClient discoveryJerseyClient;
    private ApacheHttpClient4 discoveryApacheClient;
    protected static EurekaClientConfig clientConfig;
    private final AtomicReference<String> remoteRegionsToFetch;
    private final InstanceRegionChecker instanceRegionChecker;
    private volatile InstanceInfo.InstanceStatus lastRemoteInstanceStatus = InstanceInfo.InstanceStatus.UNKNOWN;
    private final ScheduledExecutorService scheduler;
    private final EventBus eventBus;

    public DiscoveryClient(InstanceInfo myInfo, EurekaClientConfig config) {
        this(myInfo, config, null);
    }

    @Inject
    public DiscoveryClient(InstanceInfo myInfo, EurekaClientConfig config, DiscoveryClientOptionalArgs args) {
        try {
            this.eventBus = args != null ? args.eventBus : null;
            this.scheduler = Executors.newScheduledThreadPool(4, new ThreadFactoryBuilder().setNameFormat("DiscoveryClient-%d").setDaemon(true).build());
            clientConfig = config;
            String zone = DiscoveryClient.getZone(myInfo);
            this.eurekaServiceUrls.set(this.getDiscoveryServiceUrls(zone));
            this.scheduler.scheduleWithFixedDelay(this.getServiceUrlUpdateTask(zone), clientConfig.getEurekaServiceUrlPollIntervalSeconds(), clientConfig.getEurekaServiceUrlPollIntervalSeconds(), TimeUnit.SECONDS);
            this.localRegionApps.set(new Applications());
            if (myInfo != null) {
                this.instanceInfo = myInfo;
                this.appPathIdentifier = this.instanceInfo.getAppName() + "/" + this.instanceInfo.getId();
            }
            String proxyHost = clientConfig.getProxyHost();
            String proxyPort = clientConfig.getProxyPort();
            this.discoveryJerseyClient = EurekaJerseyClient.createJerseyClient("DiscoveryClient-HTTPClient", clientConfig.getEurekaServerConnectTimeoutSeconds() * 1000, clientConfig.getEurekaServerReadTimeoutSeconds() * 1000, clientConfig.getEurekaServerTotalConnectionsPerHost(), clientConfig.getEurekaServerTotalConnections(), clientConfig.getEurekaConnectionIdleTimeoutSeconds());
            this.discoveryApacheClient = this.discoveryJerseyClient.getClient();
            ClientConfig cc = this.discoveryJerseyClient.getClientconfig();
            this.remoteRegionsToFetch = new AtomicReference<String>(clientConfig.fetchRegistryForRemoteRegions());
            AbstractAzToRegionMapper azToRegionMapper = clientConfig.shouldUseDnsForFetchingServiceUrls() ? new DNSBasedAzToRegionMapper() : new PropertyBasedAzToRegionMapper(clientConfig);
            if (null != this.remoteRegionsToFetch.get()) {
                azToRegionMapper.setRegionsToFetch(this.remoteRegionsToFetch.get().split(","));
            }
            this.instanceRegionChecker = new InstanceRegionChecker(azToRegionMapper, clientConfig.getRegion());
            boolean enableGZIPContentEncodingFilter = config.shouldGZipContent();
            if (enableGZIPContentEncodingFilter) {
                this.discoveryApacheClient.addFilter((ClientFilter)new GZIPContentEncodingFilter(false));
            }
            if (proxyHost != null && proxyPort != null) {
                cc.getProperties().put("com.sun.jersey.impl.client.httpclient.proxyURI", "http://" + proxyHost + ":" + proxyPort);
            }
        }
        catch (Throwable e) {
            throw new RuntimeException("Failed to initialize DiscoveryClient!", e);
        }
        if (clientConfig.shouldFetchRegistry() && !this.fetchRegistry(false)) {
            this.fetchRegistryFromBackup();
        }
        this.initScheduledTasks();
        try {
            Monitors.registerObject((Object)this);
        }
        catch (Throwable e) {
            logger.warn("Cannot register timers", e);
        }
        DiscoveryManager.getInstance().setDiscoveryClient(this);
        DiscoveryManager.getInstance().setEurekaClientConfig(config);
    }

    @Override
    public Application getApplication(String appName) {
        return this.getApplications().getRegisteredApplications(appName);
    }

    @Override
    public Applications getApplications() {
        return this.localRegionApps.get();
    }

    public Applications getApplicationsForARegion(@Nullable String region) {
        if (this.instanceRegionChecker.isLocalRegion(region)) {
            return this.localRegionApps.get();
        }
        return this.remoteRegionVsApps.get(region);
    }

    public Set<String> getAllKnownRegions() {
        String localRegion = this.instanceRegionChecker.getLocalRegion();
        if (!this.remoteRegionVsApps.isEmpty()) {
            Set<String> regions = this.remoteRegionVsApps.keySet();
            HashSet<String> toReturn = new HashSet<String>(regions);
            toReturn.add(localRegion);
            return toReturn;
        }
        return Collections.singleton(localRegion);
    }

    @Override
    public List<InstanceInfo> getInstancesById(String id) {
        ArrayList<InstanceInfo> instancesList = new ArrayList<InstanceInfo>();
        for (Application app : this.getApplications().getRegisteredApplications()) {
            InstanceInfo instanceInfo = app.getByInstanceId(id);
            if (instanceInfo == null) continue;
            instancesList.add(instanceInfo);
        }
        return instancesList;
    }

    public void registerHealthCheckCallback(HealthCheckCallback callback) {
        if (this.instanceInfo == null) {
            logger.error("Cannot register a listener for instance info since it is null!");
        }
        if (callback != null) {
            this.healthCheckCallback = callback;
        }
    }

    public List<InstanceInfo> getInstancesByVipAddress(String vipAddress, boolean secure) {
        return this.getInstancesByVipAddress(vipAddress, secure, this.instanceRegionChecker.getLocalRegion());
    }

    public List<InstanceInfo> getInstancesByVipAddress(String vipAddress, boolean secure, @Nullable String region) {
        Applications applications;
        if (vipAddress == null) {
            throw new IllegalArgumentException("Supplied VIP Address cannot be null");
        }
        if (this.instanceRegionChecker.isLocalRegion(region)) {
            applications = this.localRegionApps.get();
        } else {
            applications = this.remoteRegionVsApps.get(region);
            if (null == applications) {
                logger.debug("No applications are defined for region {}, so returning an empty instance list for vip address {}.", (Object)region, (Object)vipAddress);
                return Collections.emptyList();
            }
        }
        if (!secure) {
            return applications.getInstancesByVirtualHostName(vipAddress);
        }
        return applications.getInstancesBySecureVirtualHostName(vipAddress);
    }

    public List<InstanceInfo> getInstancesByVipAddressAndAppName(String vipAddress, String appName, boolean secure) {
        List<InstanceInfo> result = new ArrayList<InstanceInfo>();
        if (vipAddress == null && appName == null) {
            throw new IllegalArgumentException("Supplied VIP Address and application name cannot both be null");
        }
        if (vipAddress != null && appName == null) {
            return this.getInstancesByVipAddress(vipAddress, secure);
        }
        if (vipAddress == null && appName != null) {
            Application application = this.getApplication(appName);
            if (application != null) {
                result = application.getInstances();
            }
            return result;
        }
        for (Application app : this.getApplications().getRegisteredApplications()) {
            block1: for (InstanceInfo instance : app.getInstances()) {
                String[] instanceVipAddresses;
                String instanceVipAddress = secure ? instance.getSecureVipAddress() : instance.getVIPAddress();
                if (instanceVipAddress == null) continue;
                for (String vipAddressFromList : instanceVipAddresses = instanceVipAddress.split(",")) {
                    if (!vipAddress.equalsIgnoreCase(vipAddressFromList.trim()) || !appName.equalsIgnoreCase(instance.getAppName())) continue;
                    result.add(instance);
                    continue block1;
                }
            }
        }
        return result;
    }

    @Override
    public InstanceInfo getNextServerFromEureka(String virtualHostname, boolean secure) {
        List<InstanceInfo> instanceInfoList = this.getInstancesByVipAddress(virtualHostname, secure);
        if (instanceInfoList == null || instanceInfoList.isEmpty()) {
            throw new RuntimeException("No matches for the virtual host name :" + virtualHostname);
        }
        Applications apps = this.localRegionApps.get();
        int index = (int)(apps.getNextIndex(virtualHostname.toUpperCase(), secure).incrementAndGet() % (long)instanceInfoList.size());
        return instanceInfoList.get(index);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Applications getApplications(String serviceUrl) {
        ClientResponse response = null;
        Applications apps = null;
        try {
            response = this.getUrl(serviceUrl + "apps/");
            apps = (Applications)response.getEntity(Applications.class);
            logger.debug(PREFIX + this.appPathIdentifier + " -  refresh status: " + response.getStatus());
            Applications applications = apps;
            return applications;
        }
        catch (Exception e) {
            logger.error(PREFIX + this.appPathIdentifier + " - was unable to refresh its cache! status = " + e.getMessage(), (Throwable)e);
        }
        finally {
            if (response != null) {
                response.close();
            }
        }
        return apps;
    }

    private boolean shouldRegister(InstanceInfo myInfo) {
        if (!clientConfig.shouldRegisterWithEureka()) {
            return false;
        }
        if (myInfo != null && myInfo.getDataCenterInfo().equals((Object)DataCenterInfo.Name.Amazon)) {
            return true;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void register() {
        logger.info(PREFIX + this.appPathIdentifier + ": registering service...");
        ClientResponse response = null;
        try {
            response = this.makeRemoteCall(Action.Register);
            this.isRegisteredWithDiscovery = true;
            logger.info(PREFIX + this.appPathIdentifier + " - registration status: " + (response != null ? Integer.valueOf(response.getStatus()) : "not sent"));
        }
        catch (Throwable e) {
            logger.error(PREFIX + this.appPathIdentifier + " - registration failed" + e.getMessage(), e);
        }
        finally {
            if (response != null) {
                response.close();
            }
        }
    }

    public static List<String> getEurekaServiceUrlsFromConfig(String instanceZone, boolean preferSameZone) {
        int currentOffset;
        ArrayList<String> orderedUrls = new ArrayList<String>();
        String region = DiscoveryClient.getRegion();
        Object[] availZones = clientConfig.getAvailabilityZones(clientConfig.getRegion());
        if (availZones == null || availZones.length == 0) {
            availZones = new String[]{"default"};
        }
        logger.debug("The availability zone for the given region {} are %s", (Object)region, (Object)Arrays.toString(availZones));
        int myZoneOffset = DiscoveryClient.getZoneOffset(instanceZone, preferSameZone, (String[])availZones);
        List<String> serviceUrls = clientConfig.getEurekaServerServiceUrls((String)availZones[myZoneOffset]);
        if (serviceUrls != null) {
            orderedUrls.addAll(serviceUrls);
        }
        int n = currentOffset = myZoneOffset == availZones.length - 1 ? 0 : myZoneOffset + 1;
        while (currentOffset != myZoneOffset) {
            serviceUrls = clientConfig.getEurekaServerServiceUrls((String)availZones[currentOffset]);
            if (serviceUrls != null) {
                orderedUrls.addAll(serviceUrls);
            }
            if (currentOffset == availZones.length - 1) {
                currentOffset = 0;
                continue;
            }
            ++currentOffset;
        }
        if (orderedUrls.size() < 1) {
            throw new IllegalArgumentException("DiscoveryClient: invalid serviceUrl specified!");
        }
        return orderedUrls;
    }

    @PreDestroy
    public void shutdown() {
        this.cancelScheduledTasks();
        if (this.instanceInfo != null && this.shouldRegister(this.instanceInfo)) {
            this.instanceInfo.setStatus(InstanceInfo.InstanceStatus.DOWN);
            this.unregister();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void unregister() {
        ClientResponse response = null;
        try {
            response = this.makeRemoteCall(Action.Cancel);
            logger.info(PREFIX + this.appPathIdentifier + " - deregister  status: " + (response != null ? Integer.valueOf(response.getStatus()) : "not registered"));
        }
        catch (Throwable e) {
            logger.error(PREFIX + this.appPathIdentifier + " - de-registration failed" + e.getMessage(), e);
        }
        finally {
            if (response != null) {
                response.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean fetchRegistry(boolean forceFullRegistryFetch) {
        ClientResponse response;
        block12: {
            response = null;
            Stopwatch tracer = this.FETCH_REGISTRY_TIMER.start();
            try {
                Applications applications = this.getApplications();
                if (clientConfig.shouldDisableDelta() || !Strings.isNullOrEmpty((String)clientConfig.getRegistryRefreshSingleVipAddress()) || forceFullRegistryFetch || applications == null || applications.getRegisteredApplications().size() == 0 || applications.getVersion() == -1L) {
                    logger.info("Disable delta property : {}", (Object)clientConfig.shouldDisableDelta());
                    logger.info("Single vip registry refresh property : {}", (Object)clientConfig.getRegistryRefreshSingleVipAddress());
                    logger.info("Force full registry fetch : {}", (Object)forceFullRegistryFetch);
                    logger.info("Application is null : {}", (Object)(applications == null ? 1 : 0));
                    logger.info("Registered Applications size is zero : {}", (Object)(applications.getRegisteredApplications().size() == 0 ? 1 : 0));
                    logger.info("Application version is -1: {}", (Object)(applications.getVersion() == -1L ? 1 : 0));
                    response = this.getAndStoreFullRegistry();
                } else {
                    Applications delta = null;
                    response = this.makeRemoteCall(Action.Refresh_Delta);
                    if (response.getStatus() == Response.Status.OK.getStatusCode()) {
                        delta = (Applications)response.getEntity(Applications.class);
                    }
                    if (delta == null) {
                        logger.warn("The server does not allow the delta revision to be applied because it is not safe. Hence got the full registry.");
                        this.closeResponse(response);
                        response = this.getAndStoreFullRegistry();
                    } else {
                        this.updateDelta(delta);
                        String reconcileHashCode = this.getReconcileHashCode(applications);
                        if (!reconcileHashCode.equals(delta.getAppsHashCode()) || clientConfig.shouldLogDeltaDiff()) {
                            response = this.reconcileAndLogDifference(response, delta, reconcileHashCode);
                        }
                    }
                }
                applications.setAppsHashCode(applications.getReconcileHashCode());
                this.logTotalInstances();
                logger.debug(PREFIX + this.appPathIdentifier + " -  refresh status: " + response.getStatus());
                this.updateInstanceRemoteStatus();
                if (tracer == null) break block12;
            }
            catch (Throwable e) {
                try {
                    logger.error(PREFIX + this.appPathIdentifier + " - was unable to refresh its cache! status = " + e.getMessage(), e);
                    boolean bl = false;
                    return bl;
                }
                catch (Throwable throwable) {
                    throw throwable;
                }
                finally {
                    if (tracer != null) {
                        tracer.stop();
                    }
                    this.closeResponse(response);
                }
            }
            tracer.stop();
        }
        this.closeResponse(response);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void updateInstanceRemoteStatus() {
        InstanceInfo remoteInstanceInfo;
        Application app;
        InstanceInfo.InstanceStatus currentRemoteInstanceStatus = null;
        if (this.instanceInfo.getAppName() != null && (app = this.getApplication(this.instanceInfo.getAppName())) != null && (remoteInstanceInfo = app.getByInstanceId(this.instanceInfo.getId())) != null) {
            currentRemoteInstanceStatus = remoteInstanceInfo.getStatus();
        }
        if (currentRemoteInstanceStatus == null) {
            currentRemoteInstanceStatus = InstanceInfo.InstanceStatus.UNKNOWN;
        }
        if (this.lastRemoteInstanceStatus != currentRemoteInstanceStatus) {
            try {
                if (this.eventBus != null) {
                    StatusChangeEvent event = new StatusChangeEvent(this.lastRemoteInstanceStatus, currentRemoteInstanceStatus);
                    this.eventBus.publish((Object)event);
                }
            }
            finally {
                this.lastRemoteInstanceStatus = currentRemoteInstanceStatus;
            }
        }
    }

    public InstanceInfo.InstanceStatus getInstanceRemoteStatus() {
        return this.lastRemoteInstanceStatus;
    }

    private String getReconcileHashCode(Applications applications) {
        TreeMap<String, AtomicInteger> instanceCountMap = new TreeMap<String, AtomicInteger>();
        if (this.isFetchingRemoteRegionRegistries()) {
            for (Applications remoteApp : this.remoteRegionVsApps.values()) {
                remoteApp.populateInstanceCountMap(instanceCountMap);
            }
        }
        applications.populateInstanceCountMap(instanceCountMap);
        return Applications.getReconcileHashCode(instanceCountMap);
    }

    private ClientResponse getAndStoreFullRegistry() throws Throwable {
        ClientResponse response = this.makeRemoteCall(Action.Refresh);
        logger.info("Getting all instance registry info from the eureka server");
        Applications apps = (Applications)response.getEntity(Applications.class);
        if (apps == null) {
            logger.error("The application is null for some reason. Not storing this information");
        } else {
            this.localRegionApps.set(this.filterAndShuffle(apps));
        }
        logger.info("The response status is {}", (Object)response.getStatus());
        return response;
    }

    private void logTotalInstances() {
        int totInstances = 0;
        for (Application application : this.getApplications().getRegisteredApplications()) {
            totInstances += application.getInstancesAsIsFromEureka().size();
        }
        logger.debug("The total number of all instances in the client now is {}", (Object)totInstances);
    }

    private ClientResponse reconcileAndLogDifference(ClientResponse response, Applications delta, String reconcileHashCode) throws Throwable {
        logger.warn("The Reconcile hashcodes do not match, client : {}, server : {}. Getting the full registry", (Object)reconcileHashCode, (Object)delta.getAppsHashCode());
        this.closeResponse(response);
        response = this.makeRemoteCall(Action.Refresh);
        Applications serverApps = (Applications)response.getEntity(Applications.class);
        try {
            Map<String, List<String>> reconcileDiffMap = this.getApplications().getReconcileMapDiff(serverApps);
            String reconcileString = "";
            for (Map.Entry<String, List<String>> mapEntry : reconcileDiffMap.entrySet()) {
                reconcileString = reconcileString + mapEntry.getKey() + ": ";
                for (String displayString : mapEntry.getValue()) {
                    reconcileString = reconcileString + displayString;
                }
                reconcileString = reconcileString + "\n";
            }
            logger.warn("The reconcile string is {}", (Object)reconcileString);
        }
        catch (Throwable e) {
            logger.error("Could not calculate reconcile string ", e);
        }
        this.localRegionApps.set(this.filterAndShuffle(serverApps));
        this.getApplications().setVersion(delta.getVersion());
        logger.warn("The Reconcile hashcodes after complete sync up, client : {}, server : {}.", (Object)this.getApplications().getReconcileHashCode(), (Object)delta.getAppsHashCode());
        return response;
    }

    private void updateDelta(Applications delta) {
        int deltaCount = 0;
        for (Application app : delta.getRegisteredApplications()) {
            for (InstanceInfo instance : app.getInstances()) {
                Application existingApp;
                Applications applications = this.getApplications();
                String instanceRegion = this.instanceRegionChecker.getInstanceRegion(instance);
                if (!this.instanceRegionChecker.isLocalRegion(instanceRegion)) {
                    Applications remoteApps = this.remoteRegionVsApps.get(instanceRegion);
                    if (null == remoteApps) {
                        remoteApps = new Applications();
                        this.remoteRegionVsApps.put(instanceRegion, remoteApps);
                    }
                    applications = remoteApps;
                }
                ++deltaCount;
                if (InstanceInfo.ActionType.ADDED.equals((Object)instance.getActionType())) {
                    existingApp = applications.getRegisteredApplications(instance.getAppName());
                    if (existingApp == null) {
                        applications.addApplication(app);
                    }
                    logger.debug("Added instance {} to the existing apps in region {}", (Object)instance.getId(), (Object)instanceRegion);
                    applications.getRegisteredApplications(instance.getAppName()).addInstance(instance);
                    continue;
                }
                if (InstanceInfo.ActionType.MODIFIED.equals((Object)instance.getActionType())) {
                    existingApp = applications.getRegisteredApplications(instance.getAppName());
                    if (existingApp == null) {
                        applications.addApplication(app);
                    }
                    logger.debug("Modified instance {} to the existing apps ", (Object)instance.getId());
                    applications.getRegisteredApplications(instance.getAppName()).addInstance(instance);
                    continue;
                }
                if (!InstanceInfo.ActionType.DELETED.equals((Object)instance.getActionType())) continue;
                existingApp = applications.getRegisteredApplications(instance.getAppName());
                if (existingApp == null) {
                    applications.addApplication(app);
                }
                logger.debug("Deleted instance {} to the existing apps ", (Object)instance.getId());
                applications.getRegisteredApplications(instance.getAppName()).removeInstance(instance);
            }
        }
        logger.debug("The total number of instances fetched by the delta processor : {}", (Object)deltaCount);
        this.getApplications().setVersion(delta.getVersion());
        this.getApplications().shuffleInstances(clientConfig.shouldFilterOnlyUpInstances());
        for (Applications applications : this.remoteRegionVsApps.values()) {
            applications.setVersion(delta.getVersion());
            applications.shuffleInstances(clientConfig.shouldFilterOnlyUpInstances());
        }
    }

    private ClientResponse makeRemoteCall(Action action) throws Throwable {
        return this.makeRemoteCall(action, 0);
    }

    private ClientResponse makeRemoteCall(Action action, int serviceUrlIndex) throws Throwable {
        String urlPath = null;
        Stopwatch tracer = null;
        String serviceUrl = this.eurekaServiceUrls.get().get(serviceUrlIndex);
        ClientResponse response = null;
        logger.debug("Discovery Client talking to the server {}", (Object)serviceUrl);
        try {
            String vipAddress;
            if (UNKNOWN.equals(this.instanceInfo.getAppName()) && !Action.Refresh.equals((Object)action) && !Action.Refresh_Delta.equals((Object)action)) {
                ClientResponse clientResponse = null;
                return clientResponse;
            }
            WebResource r = this.discoveryApacheClient.resource(serviceUrl);
            switch (action) {
                case Renew: {
                    tracer = this.RENEW_TIMER.start();
                    urlPath = "apps/" + this.appPathIdentifier;
                    response = (ClientResponse)r.path(urlPath).queryParam("status", this.instanceInfo.getStatus().toString()).queryParam("lastDirtyTimestamp", this.instanceInfo.getLastDirtyTimestamp().toString()).put(ClientResponse.class);
                    break;
                }
                case Refresh: {
                    tracer = this.REFRESH_TIMER.start();
                    vipAddress = clientConfig.getRegistryRefreshSingleVipAddress();
                    String string = urlPath = vipAddress == null ? "apps/" : "vips/" + vipAddress;
                    if (this.isFetchingRemoteRegionRegistries()) {
                        urlPath = urlPath + "?regions=" + this.remoteRegionsToFetch;
                    }
                    response = this.getUrl(serviceUrl + urlPath);
                    break;
                }
                case Refresh_Delta: {
                    tracer = this.REFRESH_DELTA_TIMER.start();
                    urlPath = "apps/delta";
                    if (this.isFetchingRemoteRegionRegistries()) {
                        urlPath = urlPath + "?regions=" + this.remoteRegionsToFetch;
                    }
                    response = this.getUrl(serviceUrl + urlPath);
                    break;
                }
                case Register: {
                    tracer = this.REGISTER_TIMER.start();
                    urlPath = "apps/" + this.instanceInfo.getAppName();
                    response = (ClientResponse)r.path(urlPath).type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, (Object)this.instanceInfo);
                    break;
                }
                case Cancel: {
                    tracer = this.CANCEL_TIMER.start();
                    urlPath = "apps/" + this.appPathIdentifier;
                    response = (ClientResponse)r.path(urlPath).delete(ClientResponse.class);
                    if (this.isRegisteredWithDiscovery || response.getStatus() != Response.Status.NOT_FOUND.getStatusCode()) break;
                    ClientResponse clientResponse = response;
                    return clientResponse;
                }
            }
            if (logger.isInfoEnabled()) {
                logger.info("Finished a call to service url {} and url path {} with status code {}.", (Object[])new String[]{serviceUrl, urlPath, String.valueOf(response.getStatus())});
            }
            if (this.isOk(action, response.getStatus())) {
                vipAddress = response;
                return vipAddress;
            }
            try {
                logger.warn("Action: " + (Object)((Object)action) + "  => returned status of " + response.getStatus() + " from " + serviceUrl + urlPath);
                throw new RuntimeException("Bad status: " + response.getStatus());
            }
            catch (Throwable t) {
                this.closeResponse(response);
                String msg = "Can't get a response from " + serviceUrl + urlPath;
                if (this.eurekaServiceUrls.get().size() > ++serviceUrlIndex) {
                    logger.warn(msg, t);
                    logger.warn("Trying backup: " + this.eurekaServiceUrls.get().get(serviceUrlIndex));
                    this.SERVER_RETRY_COUNTER.increment();
                    ClientResponse clientResponse = this.makeRemoteCall(action, serviceUrlIndex);
                    return clientResponse;
                }
                this.ALL_SERVER_FAILURE_COUNT.increment();
                logger.error(msg + "\nCan't contact any eureka nodes - possibly a security group issue?", t);
                throw t;
            }
        }
        finally {
            if (tracer != null) {
                tracer.stop();
            }
        }
    }

    private void closeResponse(ClientResponse response) {
        if (response != null) {
            try {
                response.close();
            }
            catch (Throwable th) {
                logger.error("Cannot release response resource :", th);
            }
        }
    }

    private void initScheduledTasks() {
        if (clientConfig.shouldFetchRegistry()) {
            this.scheduler.scheduleWithFixedDelay(new CacheRefreshThread(), clientConfig.getRegistryFetchIntervalSeconds(), clientConfig.getRegistryFetchIntervalSeconds(), TimeUnit.SECONDS);
        }
        if (this.shouldRegister(this.instanceInfo)) {
            logger.info("Starting heartbeat executor: renew interval is: " + this.instanceInfo.getLeaseInfo().getRenewalIntervalInSecs());
            this.scheduler.scheduleWithFixedDelay(new HeartbeatThread(), this.instanceInfo.getLeaseInfo().getRenewalIntervalInSecs(), this.instanceInfo.getLeaseInfo().getRenewalIntervalInSecs(), TimeUnit.SECONDS);
            this.scheduler.scheduleWithFixedDelay(new InstanceInfoReplicator(), 10 + clientConfig.getInstanceInfoReplicationIntervalSeconds(), clientConfig.getInstanceInfoReplicationIntervalSeconds(), TimeUnit.SECONDS);
        }
    }

    private void cancelScheduledTasks() {
        this.scheduler.shutdownNow();
    }

    public List<String> getServiceUrlsFromDNS(String instanceZone, boolean preferSameZone) {
        Stopwatch t = this.GET_SERVICE_URLS_DNS_TIMER.start();
        String region = DiscoveryClient.getRegion();
        Map<String, List<String>> zoneDnsNamesMap = DiscoveryClient.getZoneBasedDiscoveryUrlsFromRegion(region);
        Set<String> availableZones = zoneDnsNamesMap.keySet();
        ArrayList<String> zones = new ArrayList<String>(availableZones);
        if (zones.isEmpty()) {
            throw new RuntimeException("No available zones configured for the instanceZone " + instanceZone);
        }
        int zoneIndex = 0;
        boolean zoneFound = false;
        for (String zone : zones) {
            logger.debug("Checking if the instance zone {} is the same as the zone from DNS {}", (Object)instanceZone, (Object)zone);
            if (preferSameZone) {
                if (instanceZone.equalsIgnoreCase(zone)) {
                    zoneFound = true;
                }
            } else if (!instanceZone.equalsIgnoreCase(zone)) {
                zoneFound = true;
            }
            if (zoneFound) {
                Object[] args = new Object[]{zones, instanceZone, zoneIndex};
                logger.debug("The zone index from the list {} that matches the instance zone {} is {}", args);
                break;
            }
            ++zoneIndex;
        }
        if (zoneIndex >= zones.size()) {
            logger.warn("No match for the zone {} in the list of available zones {}", (Object)instanceZone, (Object)Arrays.toString(zones.toArray()));
        } else {
            for (int i = 0; i < zoneIndex; ++i) {
                String zone;
                zone = (String)zones.remove(0);
                zones.add(zone);
            }
        }
        ArrayList<String> serviceUrls = new ArrayList<String>();
        for (String zone : zones) {
            for (String zoneCname : zoneDnsNamesMap.get(zone)) {
                ArrayList<String> ec2Urls = new ArrayList<String>(DiscoveryClient.getEC2DiscoveryUrlsFromZone(zoneCname, DiscoveryUrlType.CNAME));
                if (ec2Urls.size() > 1) {
                    this.arrangeListBasedonHostname(ec2Urls);
                }
                for (String ec2Url : ec2Urls) {
                    String serviceUrl = "http://" + ec2Url + ":" + clientConfig.getEurekaServerPort() + "/" + clientConfig.getEurekaServerURLContext() + "/";
                    logger.debug("The EC2 url is {}", (Object)serviceUrl);
                    serviceUrls.add(serviceUrl);
                }
            }
        }
        String primaryServiceUrl = (String)serviceUrls.remove(0);
        this.arrangeListBasedonHostname(serviceUrls);
        serviceUrls.add(0, primaryServiceUrl);
        logger.debug("This client will talk to the following serviceUrls in order : {} ", (Object)Arrays.toString(serviceUrls.toArray()));
        t.stop();
        return serviceUrls;
    }

    public List<String> getDiscoveryServiceUrls(String zone) {
        boolean shouldUseDns = clientConfig.shouldUseDnsForFetchingServiceUrls();
        if (shouldUseDns) {
            return this.getServiceUrlsFromDNS(zone, clientConfig.shouldPreferSameZoneEureka());
        }
        return DiscoveryClient.getEurekaServiceUrlsFromConfig(zone, clientConfig.shouldPreferSameZoneEureka());
    }

    public static String getZone(InstanceInfo myInfo) {
        String awsInstanceZone;
        String instanceZone;
        String[] availZones = clientConfig.getAvailabilityZones(clientConfig.getRegion());
        String string = instanceZone = availZones == null || availZones.length == 0 ? "default" : availZones[0];
        if (myInfo != null && myInfo.getDataCenterInfo().getName() == DataCenterInfo.Name.Amazon && (awsInstanceZone = ((AmazonInfo)myInfo.getDataCenterInfo()).get(AmazonInfo.MetaDataKey.availabilityZone)) != null) {
            instanceZone = awsInstanceZone;
        }
        return instanceZone;
    }

    public static String getRegion() {
        String region = clientConfig.getRegion();
        if (region == null) {
            region = "default";
        }
        region = region.trim().toLowerCase();
        return region;
    }

    static Map<String, List<String>> getZoneBasedDiscoveryUrlsFromRegion(String region) {
        String discoveryDnsName = null;
        try {
            discoveryDnsName = "txt." + region + "." + clientConfig.getEurekaServerDNSName();
            logger.debug("The region url to be looked up is {} :", (Object)discoveryDnsName);
            TreeSet<String> zoneCnamesForRegion = new TreeSet<String>(DiscoveryClient.getCnamesFromDirContext(dirContext, discoveryDnsName));
            TreeMap<String, List<String>> zoneCnameMapForRegion = new TreeMap<String, List<String>>();
            for (String zoneCname : zoneCnamesForRegion) {
                String zone = null;
                if (DiscoveryClient.isEC2Url(zoneCname)) {
                    throw new RuntimeException("Cannot find the right DNS entry for " + discoveryDnsName + ". " + "Expected mapping of the format <aws_zone>.<domain_name>");
                }
                String[] cnameTokens = zoneCname.split("\\.");
                zone = cnameTokens[0];
                logger.debug("The zoneName mapped to region {} is {}", (Object)region, (Object)zone);
                ArrayList<String> zoneCnamesSet = (ArrayList<String>)zoneCnameMapForRegion.get(zone);
                if (zoneCnamesSet == null) {
                    zoneCnamesSet = new ArrayList<String>();
                    zoneCnameMapForRegion.put(zone, zoneCnamesSet);
                }
                zoneCnamesSet.add(zoneCname);
            }
            return zoneCnameMapForRegion;
        }
        catch (Throwable e) {
            throw new RuntimeException("Cannot get cnames bound to the region:" + discoveryDnsName, e);
        }
    }

    private static boolean isEC2Url(String zoneCname) {
        return zoneCname.startsWith("ec2");
    }

    public static Set<String> getEC2DiscoveryUrlsFromZone(String dnsName, DiscoveryUrlType type) {
        TreeSet<String> eipsForZone = null;
        try {
            dnsName = "txt." + dnsName;
            logger.debug("The zone url to be looked up is {} :", (Object)dnsName);
            Set<String> ec2UrlsForZone = DiscoveryClient.getCnamesFromDirContext(dirContext, dnsName);
            for (String ec2Url : ec2UrlsForZone) {
                logger.debug("The eureka url for the dns name {} is {}", (Object)dnsName, (Object)ec2Url);
                ec2UrlsForZone.add(ec2Url);
            }
            if (DiscoveryUrlType.CNAME.equals((Object)type)) {
                return ec2UrlsForZone;
            }
            eipsForZone = new TreeSet<String>();
            for (String cname : ec2UrlsForZone) {
                String[] tokens = cname.split("\\.");
                String ec2HostName = tokens[0];
                String[] ips = ec2HostName.split("-");
                StringBuffer eipBuffer = new StringBuffer();
                for (int ipCtr = 1; ipCtr < 5; ++ipCtr) {
                    eipBuffer.append(ips[ipCtr]);
                    if (ipCtr >= 4) continue;
                    eipBuffer.append(".");
                }
                eipsForZone.add(eipBuffer.toString());
            }
            logger.debug("The EIPS for {} is {} :", (Object)dnsName, eipsForZone);
        }
        catch (Throwable e) {
            throw new RuntimeException("Cannot get cnames bound to the region:" + dnsName, e);
        }
        return eipsForZone;
    }

    private static int getZoneOffset(String myZone, boolean preferSameZone, String[] availZones) {
        for (int i = 0; i < availZones.length; ++i) {
            if (myZone == null || availZones[i].equalsIgnoreCase(myZone.trim()) != preferSameZone) continue;
            return i;
        }
        logger.warn("DISCOVERY: Could not pick a zone based on preferred zone settings. My zone - {}, preferSameZone- {}. Defaulting to " + availZones[0], (Object)myZone, (Object)preferSameZone);
        return 0;
    }

    private boolean isOk(Action action, int httpStatus) {
        if (httpStatus >= 200 && httpStatus < 300) {
            return true;
        }
        if (Action.Renew == action && httpStatus == 404) {
            return true;
        }
        return Action.Refresh_Delta == action && (httpStatus == 403 || httpStatus == 404);
    }

    private InstanceInfo getCoordinatingServer() {
        Application app = this.getApplication(DISCOVERY_APPID);
        List<InstanceInfo> discoveryInstances = null;
        InstanceInfo instanceToReturn = null;
        if (app != null) {
            discoveryInstances = app.getInstances();
        }
        if (discoveryInstances != null) {
            for (InstanceInfo instance : discoveryInstances) {
                if (instance == null || !instance.isCoordinatingDiscoveryServer().booleanValue()) continue;
                instanceToReturn = instance;
                break;
            }
        }
        return instanceToReturn;
    }

    private ClientResponse getUrl(String fullServiceUrl) {
        ClientResponse cr = (ClientResponse)this.discoveryApacheClient.resource(fullServiceUrl).accept(new MediaType[]{MediaType.APPLICATION_JSON_TYPE}).get(ClientResponse.class);
        return cr;
    }

    private boolean isHealthCheckEnabled() {
        return this.healthCheckCallback != null && InstanceInfo.InstanceStatus.STARTING != this.instanceInfo.getStatus() && InstanceInfo.InstanceStatus.OUT_OF_SERVICE != this.instanceInfo.getStatus();
    }

    private static DirContext getDirContext() {
        Hashtable<String, String> env = new Hashtable<String, String>();
        env.put(JAVA_NAMING_FACTORY_INITIAL, DNS_NAMING_FACTORY);
        env.put(JAVA_NAMING_PROVIDER_URL, DNS_PROVIDER_URL);
        InitialDirContext dirContext = null;
        try {
            dirContext = new InitialDirContext(env);
        }
        catch (Throwable e) {
            throw new RuntimeException("Cannot get dir context for some reason", e);
        }
        return dirContext;
    }

    private static Set<String> getCnamesFromDirContext(DirContext dirContext, String discoveryDnsName) throws Throwable {
        String[] cnames;
        Attributes attrs = dirContext.getAttributes(discoveryDnsName, new String[]{DNS_RECORD_TYPE});
        Attribute attr = attrs.get(DNS_RECORD_TYPE);
        String txtRecord = null;
        if (attr != null) {
            txtRecord = attr.get().toString();
        }
        TreeSet<String> cnamesSet = new TreeSet<String>();
        if (txtRecord == null || "".equals(txtRecord.trim())) {
            return cnamesSet;
        }
        for (String cname : cnames = txtRecord.split(" ")) {
            cnamesSet.add(cname);
        }
        return cnamesSet;
    }

    private void fetchRegistryFromBackup() {
        try {
            BackupRegistry backupRegistryInstance = this.newBackupRegistryInstance();
            if (null != backupRegistryInstance) {
                Applications apps = null;
                if (this.isFetchingRemoteRegionRegistries()) {
                    String remoteRegionsStr = this.remoteRegionsToFetch.get();
                    if (null != remoteRegionsStr) {
                        apps = backupRegistryInstance.fetchRegistry(remoteRegionsStr.split(","));
                    }
                } else {
                    apps = backupRegistryInstance.fetchRegistry();
                }
                if (apps != null) {
                    Applications applications = this.filterAndShuffle(apps);
                    applications.setAppsHashCode(applications.getReconcileHashCode());
                    this.localRegionApps.set(applications);
                    this.logTotalInstances();
                    logger.info("Fetched registry successfully from the backup");
                }
            } else {
                logger.warn("No backup registry instance defined & unable to find any discovery servers.");
            }
        }
        catch (Throwable e) {
            logger.warn("Cannot fetch applications from apps although backup registry was specified", e);
        }
    }

    @Nullable
    protected BackupRegistry newBackupRegistryInstance() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        String backupRegistryClassName = clientConfig.getBackupRegistryImpl();
        if (null != backupRegistryClassName) {
            return (BackupRegistry)Class.forName(backupRegistryClassName).newInstance();
        }
        return null;
    }

    private TimerTask getServiceUrlUpdateTask(final String zone) {
        return new TimerTask(){

            @Override
            public void run() {
                try {
                    List<String> serviceUrlList = DiscoveryClient.this.getDiscoveryServiceUrls(zone);
                    if (serviceUrlList.isEmpty()) {
                        logger.warn("The service url list is empty");
                        return;
                    }
                    if (!serviceUrlList.equals(DiscoveryClient.this.eurekaServiceUrls.get())) {
                        logger.info("Updating the serviceUrls as they seem to have changed from {} to {} ", (Object)Arrays.toString(((List)DiscoveryClient.this.eurekaServiceUrls.get()).toArray()), (Object)Arrays.toString(serviceUrlList.toArray()));
                        DiscoveryClient.this.eurekaServiceUrls.set(serviceUrlList);
                    }
                }
                catch (Throwable e) {
                    logger.error("Cannot get the eureka service urls :", e);
                }
            }
        };
    }

    private Applications filterAndShuffle(Applications apps) {
        if (apps != null) {
            if (this.isFetchingRemoteRegionRegistries()) {
                ConcurrentHashMap<String, Applications> remoteRegionVsApps = new ConcurrentHashMap<String, Applications>();
                apps.shuffleAndIndexInstances(remoteRegionVsApps, clientConfig, this.instanceRegionChecker);
                for (Applications applications : remoteRegionVsApps.values()) {
                    applications.shuffleInstances(clientConfig.shouldFilterOnlyUpInstances());
                }
                this.remoteRegionVsApps = remoteRegionVsApps;
            } else {
                apps.shuffleInstances(clientConfig.shouldFilterOnlyUpInstances());
            }
        }
        return apps;
    }

    private boolean isFetchingRemoteRegionRegistries() {
        return null != this.remoteRegionsToFetch.get();
    }

    private void arrangeListBasedonHostname(List<String> list) {
        int listSize = 0;
        if (list != null) {
            listSize = list.size();
        }
        if (this.instanceInfo == null || listSize == 0) {
            return;
        }
        int instanceHashcode = this.instanceInfo.getHostName().hashCode();
        if (instanceHashcode < 0) {
            instanceHashcode *= -1;
        }
        int backupInstance = instanceHashcode % listSize;
        for (int i = 0; i < backupInstance; ++i) {
            String zone = list.remove(0);
            list.add(zone);
        }
    }

    class CacheRefreshThread
    extends TimerTask {
        CacheRefreshThread() {
        }

        @Override
        public void run() {
            try {
                boolean remoteRegionsModified = false;
                String latestRemoteRegions = clientConfig.fetchRegistryForRemoteRegions();
                if (null != latestRemoteRegions) {
                    String currentRemoteRegions = (String)DiscoveryClient.this.remoteRegionsToFetch.get();
                    if (!latestRemoteRegions.equals(currentRemoteRegions)) {
                        if (DiscoveryClient.this.remoteRegionsToFetch.compareAndSet(currentRemoteRegions, latestRemoteRegions)) {
                            String[] remoteRegions = latestRemoteRegions.split(",");
                            DiscoveryClient.this.instanceRegionChecker.getAzToRegionMapper().setRegionsToFetch(remoteRegions);
                            remoteRegionsModified = true;
                        } else {
                            logger.info("Remote regions to fetch modified concurrently, ignoring change from {} to {}", (Object)currentRemoteRegions, (Object)latestRemoteRegions);
                        }
                    } else {
                        DiscoveryClient.this.instanceRegionChecker.getAzToRegionMapper().refreshMapping();
                    }
                }
                DiscoveryClient.this.fetchRegistry(remoteRegionsModified);
                if (logger.isInfoEnabled()) {
                    StringBuilder allAppsHashCodes = new StringBuilder();
                    allAppsHashCodes.append("Local region apps hashcode: ");
                    allAppsHashCodes.append(((Applications)DiscoveryClient.this.localRegionApps.get()).getAppsHashCode());
                    allAppsHashCodes.append(", is fetching remote regions? ");
                    allAppsHashCodes.append(DiscoveryClient.this.isFetchingRemoteRegionRegistries());
                    for (Map.Entry entry : DiscoveryClient.this.remoteRegionVsApps.entrySet()) {
                        allAppsHashCodes.append(", Remote region: ");
                        allAppsHashCodes.append((String)entry.getKey());
                        allAppsHashCodes.append(" , apps hashcode: ");
                        allAppsHashCodes.append(((Applications)entry.getValue()).getAppsHashCode());
                    }
                    logger.info("Completed cache refresh task for discovery. All Apps hash code is {} ", (Object)allAppsHashCodes.toString());
                }
            }
            catch (Throwable th) {
                logger.error("Cannot fetch registry from server", th);
            }
        }
    }

    private class InstanceInfoReplicator
    extends TimerTask {
        private InstanceInfoReplicator() {
        }

        @Override
        public void run() {
            try {
                ApplicationInfoManager.getInstance().refreshDataCenterInfoIfRequired();
                InstanceInfo discoveryServer = DiscoveryClient.this.getCoordinatingServer();
                if (discoveryServer != null && DataCenterInfo.Name.Amazon.equals(discoveryServer.getDataCenterInfo())) {
                    String amiId = ((AmazonInfo)discoveryServer.getDataCenterInfo()).get(AmazonInfo.MetaDataKey.amiId);
                    if (DiscoveryClient.this.discoveryServerAMIId == null) {
                        DiscoveryClient.this.discoveryServerAMIId = amiId;
                    } else if (!DiscoveryClient.this.discoveryServerAMIId.equals(amiId)) {
                        logger.info("The eureka AMI ID changed from " + DiscoveryClient.this.discoveryServerAMIId + " to " + amiId + ". Pushing the appinfo to eureka");
                        DiscoveryClient.this.instanceInfo.setIsDirty(true);
                        DiscoveryClient.this.discoveryServerAMIId = amiId;
                    }
                }
                if (DiscoveryClient.this.isHealthCheckEnabled()) {
                    boolean isHealthy = DiscoveryClient.this.healthCheckCallback.isHealthy();
                    DiscoveryClient.this.instanceInfo.setStatus(isHealthy ? InstanceInfo.InstanceStatus.UP : InstanceInfo.InstanceStatus.DOWN);
                }
                if (DiscoveryClient.this.instanceInfo.isDirty()) {
                    logger.info(DiscoveryClient.PREFIX + DiscoveryClient.this.appPathIdentifier + " - retransmit instance info with status " + DiscoveryClient.this.instanceInfo.getStatus().toString());
                    DiscoveryClient.this.register();
                    DiscoveryClient.this.instanceInfo.setIsDirty(false);
                }
            }
            catch (Throwable t) {
                logger.error("There was a problem with the instance info replicator :", t);
            }
        }
    }

    private class HeartbeatThread
    extends TimerTask {
        private HeartbeatThread() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ClientResponse response = null;
            try {
                response = DiscoveryClient.this.makeRemoteCall(Action.Renew);
                logger.debug(DiscoveryClient.PREFIX + DiscoveryClient.this.appPathIdentifier + " - Heartbeat status: " + (response != null ? Integer.valueOf(response.getStatus()) : "not sent"));
                if (response == null) {
                    return;
                }
                if (response.getStatus() == 404) {
                    DiscoveryClient.this.REREGISTER_COUNTER.increment();
                    logger.info(DiscoveryClient.PREFIX + DiscoveryClient.this.appPathIdentifier + " - Re-registering " + "apps/" + DiscoveryClient.this.instanceInfo.getAppName());
                    DiscoveryClient.this.register();
                }
            }
            catch (Throwable e) {
                logger.error(DiscoveryClient.PREFIX + DiscoveryClient.this.appPathIdentifier + " - was unable to send heartbeat!", e);
            }
            finally {
                if (response != null) {
                    response.close();
                }
            }
        }
    }

    public static enum DiscoveryUrlType {
        CNAME,
        A;

    }

    public static class DiscoveryClientOptionalArgs {
        @Inject(optional=true)
        private EventBus eventBus;
    }

    private static enum Action {
        Register,
        Cancel,
        Renew,
        Refresh,
        Refresh_Delta;

    }
}

