package com.netcore.android.network

import android.util.Log
import com.netcore.android.BuildConfig
import com.netcore.android.SMTConfigConstants
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.models.SMTRequest
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.nio.charset.Charset
import java.util.zip.GZIPInputStream

/**
 * @author Netcore
 * created on 26/02/2019
 * @Description: Http request client which prepares the request and makes network call.
 *
 */
internal class SMTHttpRequestClient {

    /**
     * Making http calls to get the data from the server
     * @param request it is a SMTRequest object
     * @return  NetworkResponse which has http code and response
     */
    val TAG = SMTHttpRequestClient::class.java.simpleName
    val GZIP = "gzip"

    internal fun makeNetworkCall(request: SMTRequest): NetworkResponse {
        var networkResponse = NetworkResponse()
        networkResponse.apiID = request.getAPITypeID()
        var baseUrl = if (request.getBaseUrl().isEmpty()) {
            BuildConfig.BASE_URL
        } else {
            request.getBaseUrl()
        }

        /*
        Preparing the url in case of GET request
         */
        if (request.getApiEndPoint() != null) {
            baseUrl = baseUrl.plus(request.getApiEndPoint())
        }

        var url = URL(baseUrl)

        SMTLogger.internal("URL:", baseUrl)

        var urlConnection: HttpURLConnection? = null

        try {

            urlConnection = url.openConnection() as HttpURLConnection?
            urlConnection?.requestMethod = request.getHttpMethod()?.name

            // Set connection timeout and read timeout value.
            urlConnection?.connectTimeout = SMTConfigConstants.WRITE_TIME_OUT
            urlConnection?.readTimeout = SMTConfigConstants.READ_TIME_OUT
            urlConnection?.setRequestProperty("Content-Type", "application/json")
            urlConnection?.setRequestProperty("Accept", "application/json")
            urlConnection?.setRequestProperty("Accept-Encoding", "gzip")

            /**
             * If request isn't GET then write the body params. Need to check for other method if it is required or not
             */
            if (request.getHttpMethod()?.value != SMTEnumHttpMethodType.GET.value) {
                urlConnection?.setChunkedStreamingMode(0)
                urlConnection?.doOutput = true
                var outputStream = DataOutputStream(urlConnection?.outputStream)
                val bytes = (request.getHParams().toString()).toByteArray(Charsets.UTF_8)
                outputStream.write(bytes)
                outputStream.flush()
                outputStream.close()
            }
            networkResponse.httpCode = urlConnection?.responseCode
//            Log.d("SDK3->", "Content-Encoding " + urlConnection?.contentEncoding + " Url " + urlConnection?.url)

            if (networkResponse.httpCode == HttpURLConnection.HTTP_OK) {
                if (GZIP.equals(urlConnection?.contentEncoding)) {
                    val inputStream = urlConnection?.inputStream
                    if (inputStream != null) {
                        networkResponse.response = readGzipStream(inputStream).toString()
                        SMTLogger.internal("Network", "Response : ${networkResponse.response ?: "empty"}")
                    }
                } else {
                    val inputStream = urlConnection?.inputStream
                    networkResponse.response = inputStream?.readTextAndClose()
                    SMTLogger.internal("Network", "Response : ${networkResponse.response ?: "empty"}")
                }
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.localizedMessage)
            /**
             * reading the error stream if any from server
             */
            val errorStream = urlConnection?.errorStream
            networkResponse.response = errorStream?.readTextAndClose()
            try {
                networkResponse.httpCode = urlConnection?.responseCode
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.localizedMessage)
            }
            networkResponse.isSuccess = false
            networkResponse.shouldRetry = true
            networkResponse.errorMessage = e.localizedMessage

        } finally {

            urlConnection?.disconnect()
        }

        return networkResponse
    }

    /**
     * reads the input stream and closes it
     */
    private fun InputStream.readTextAndClose(charset: Charset = Charsets.UTF_8): String {
        return this.bufferedReader(charset).use { it.readText() }
    }


    // Converting InputStream to String
    private fun readGzipStream(inputStream: InputStream): StringBuilder {
        val output = StringBuilder()
        try {
            val gStream = GZIPInputStream(inputStream)
            val reader = InputStreamReader(gStream)
            val bufferReader = BufferedReader(reader)
            var read: String?
            while (bufferReader.readLine().also { read = it } != null) {
                output.append(read).append("\n")
            }
            reader.close()
            bufferReader.close()
            gStream.close()
        } catch (e: IOException) {
            e.message
            return output.append("")
        }
        return output
    }


    /**
     * POJO class to communicate back to the caller
     */
    internal class NetworkResponse {
        var httpCode: Int? = null
        var response: String? = null
        var isSuccess: Boolean = true
        var shouldRetry: Boolean = false
        var errorMessage: String? = null
        lateinit var apiID: SMTRequest.SMTApiTypeID
    }
}