package com.netcore.android.inbox.views.fragment

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import androidx.recyclerview.widget.ItemTouchHelper
import androidx.recyclerview.widget.RecyclerView
import com.netcore.android.R
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTAppInboxTable
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.inbox.utility.*
import com.netcore.android.inbox.views.SMTViewUtils
import com.netcore.android.inbox.views.adapter.SMTInboxNotificationAdapter
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.notification.carousel.SMTCarouselSetup
import com.netcore.android.notification.models.SMTNotificationData
import java.lang.ref.WeakReference

/**
 * Fragment for App inbox
 *
 * @author Netcore
 * @version 1.0
 * @since 02-05-2019
 */
internal class SMTInboxFragment : androidx.fragment.app.Fragment(), SMTInboxContract.View {

    // Recycler view
    private lateinit var mMessageRecyclerView: RecyclerView

    // Message layout to show message which
    private lateinit var mInboxMessageLayout: LinearLayout

    // No message layout
    private lateinit var mNoMessageTextView: TextView

    // root view
    private lateinit var mView: View

    // App context
    private lateinit var mContext: Context

    // Dialog to show progress bar
    private var mAlertDialog: AlertDialog? = null

    // Presenter of the Fragment
    override var presenter: SMTInboxContract.Presenter? = null

    // variable to decide whether show back button or not
    // if fragment is displayed within App's layout then no need to show back button
    // App has to handle back button functionality
    private var isDisableBack = false

    // used to decide when to update the view status after scrolling the recycler view
    private var isUpdateViewStatus = false

    /**
     * Overriden method
     */
    override fun onAttach(context: Context) {
        super.onAttach(context)
        mContext = context
    }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        arguments?.let {
            if (it.containsKey(SMTConfigConstants.BUNDLE_KEY_INBOX_DISABLE_BACK)) {
                isDisableBack = it.getBoolean(SMTConfigConstants.BUNDLE_KEY_INBOX_DISABLE_BACK)
            }
        }
    }

    @SuppressLint("InflateParams")
    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {

        // init the views
        mView = inflater.inflate(R.layout.inbox_fragment_layout, null)
        mNoMessageTextView = mView.findViewById(R.id.no_notification_text)
        mMessageRecyclerView = mView.findViewById<RecyclerView>(R.id.notifications_list)
        mInboxMessageLayout = mView.findViewById(R.id.inbox_notifcation_layout)

        // setup recycler view properties
        setRecyclerviewProperties()

        // create the presenter and call for Data
        presenter = SMTInboxPresenter(this, mContext)
        presenter?.getInboxMessages()

        // register local brocast
        registerLocalBroadcast()

        // toggle back button visibility
        toggleBackButton()

        // return the root view
        return mView
    }

    /**
     * To set recycler view properties
     */
    private fun setRecyclerviewProperties() {

        // set layout manager to recycler view
        mMessageRecyclerView.layoutManager = androidx.recyclerview.widget.LinearLayoutManager(mContext)

        // add vertical spacing between the items
        mMessageRecyclerView.addItemDecoration(SMTVerticalSpaceItemDecoration(20))

        // create the adapter and set to recycler view
        val adapter = SMTInboxNotificationAdapter(mContext)

        mMessageRecyclerView.adapter = adapter

        // set the scroll listener to identify the views whose status needs to be changed to viewed
        mMessageRecyclerView.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                when (newState) {
                    RecyclerView.SCROLL_STATE_IDLE -> {
                        isUpdateViewStatus = false
                        updateMessageViewStatus()
                    }
                    RecyclerView.SCROLL_STATE_DRAGGING -> {
                        isUpdateViewStatus = true
                    }
                    RecyclerView.SCROLL_STATE_SETTLING -> {
                    }
                }
            }
        })

        // This is required for the first time when inbox displayed to update the status of messages to viewed
        mMessageRecyclerView.viewTreeObserver.addOnGlobalLayoutListener(object : ViewTreeObserver.OnGlobalLayoutListener {
            override fun onGlobalLayout() {
                updateMessageViewStatus()
                mMessageRecyclerView.viewTreeObserver.removeOnGlobalLayoutListener(this)
            }
        })

        // icon to be shown while swiping the recycler items
        val deleteIcon = ContextCompat.getDrawable(mContext,
                android.R.drawable.ic_menu_delete)

        // Attaching swipe to delete functionality
        val swipeController = SMTRecyclerSwipeController(object : SMTSwipeControllerActions() {
            override fun onRightSwiped(position: Int) {

                handleInboxDismissEvent(position, true)

            }
        }, deleteIcon!!)

        // Attach the touchhelper to recycler view
        val itemTouchHelper = ItemTouchHelper(swipeController)
        itemTouchHelper.attachToRecyclerView(mMessageRecyclerView)

        // set default cache size
        mMessageRecyclerView.setHasFixedSize(true)
        mMessageRecyclerView.setItemViewCacheSize(25)

    }

    /**
     * handles item dismiss like records event,
     * updates the recycler view and also updates the message count
     * @param position - item position to delete
     * @param isRecordEvent - whether to record event or not
     */
    private fun handleInboxDismissEvent(position: Int, isRecordEvent: Boolean) {

        if (isRecordEvent) {
            // update status in DB
            (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.getItemFromPosition(position)?.let {
                SMTEventRecorder.getInstance(mContext)
                        .recordInboxEvent(it.mTrid, null, SMTInboxMessageStatus.Status.DELETED)
            }
        }

        // update adapter
        (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.removeItemAtPosition(position)

        // update the view count
        (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.let {
            setMessagesCounts(it.itemCount)
        }
    }

    /**
     * updates the message status to viewed depending upon the visibility of the view
     */
    private fun updateMessageViewStatus() {
        val layoutManager = mMessageRecyclerView.layoutManager as? androidx.recyclerview.widget.LinearLayoutManager

        // fetch the first and completely visible items
        val findFirstCompletelyVisibleItemPosition = layoutManager?.findFirstCompletelyVisibleItemPosition()
                ?: 0
        val findLastCompletelyVisibleItemPosition = layoutManager?.findLastCompletelyVisibleItemPosition()
                ?: 0

        // for only completely visible items set the status to viewed
        for (i in findFirstCompletelyVisibleItemPosition..findLastCompletelyVisibleItemPosition) {
            (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.getItemFromPosition(i)?.let {
                updateInboxMessageStatusToViewed(it, i)
            }
        }
    }

    /**
     * updated the message status to viewed
     * @param notifData notification data whose status to be changed
     * @param index index of the item which needs to be updated
     */
    private fun updateInboxMessageStatusToViewed(notifData: SMTNotificationData, index: Int) {

        // if the notif item's status is less than or quals to Delivered then onyl change it
        if (getIntInboxStatusFromString(notifData.mStatus
                        ?: "") <= SMTInboxMessageStatus.Status.DELIVERED) {
            //update the adapter
            (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.updateItemStatus(index, SMTAppInboxTable.STATUS_VIEWED)

            // record the status change event
            notifData.mStatus = SMTAppInboxTable.STATUS_VIEWED
            SMTEventRecorder.getInstance(mContext)
                    .recordInboxEvent(notifData.mTrid, null, SMTInboxMessageStatus.Status.VIEWED)
        }
    }

    /**
     * method to fetch the integer status of message
     */
    private fun getIntInboxStatusFromString(status: String): Int {
        return when (status) {
            SMTAppInboxTable.STATUS_DELIVERED -> SMTInboxMessageStatus.Status.DELIVERED
            SMTAppInboxTable.STATUS_CLICKED -> SMTInboxMessageStatus.Status.READ
            SMTAppInboxTable.STATUS_VIEWED -> SMTInboxMessageStatus.Status.VIEWED
            SMTAppInboxTable.STATUS_DISMISSED -> SMTInboxMessageStatus.Status.DELETED
            else -> SMTInboxMessageStatus.Status.DELIVERED
        }
    }

    /**
     * to show / hide loading bar
     */
    override fun showLoading(active: Boolean) {
        if (active) {
            SMTViewUtils.dismissProgress(mAlertDialog)
            mAlertDialog = SMTViewUtils.showProgress(mContext, getString(R.string.loading))
        } else {
            if (mAlertDialog != null) {
                SMTViewUtils.dismissProgress(mAlertDialog)
            }
        }
    }

    override fun showError(message: String) {

    }

    /**
     * show notifications once received the data from db / server
     */
    override fun showNotifications(notifications: MutableList<SMTNotificationData>?) {
        activity?.runOnUiThread {
            showLoading(false)
            if (notifications != null && notifications.size > 0) {

                //set the views
                hideView(mNoMessageTextView)
                showView(mInboxMessageLayout)
                setMessagesCounts(notifications.size)

                // set the data to the adapter
                (mMessageRecyclerView.adapter as SMTInboxNotificationAdapter).setMessages(notifications)
                (mMessageRecyclerView.adapter as SMTInboxNotificationAdapter).notifyDataSetChanged()

            } else {
                // if there is no data then show no message view
                showView(mNoMessageTextView)
                hideView(mInboxMessageLayout)
            }
        }
    }

    /**
     * method set message count to view
     */
    private fun setMessagesCounts(size: Int) {
        mView.findViewById<TextView>(R.id.all_notiications).text = String.format(getString(R.string.all_notifications), "$size")
    }

    /**
     * Utility method to hide the view
     */
    private fun hideView(view: View) {
        view.visibility = View.GONE
    }

    /**
     * Utility method to show the view
     */
    private fun showView(view: View) {
        view.visibility = View.VISIBLE
    }

    /**
     * Register local broadcast listener
     */
    private fun registerLocalBroadcast() {

        val filterClickDismiss = IntentFilter()

        // registering for PN click / Dismiss / Inbox data refreshed
        filterClickDismiss.addAction(SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
        filterClickDismiss.addAction(SMTConfigConstants.BROADCAST_EVENT_PN_DISMISSED)
        filterClickDismiss.addAction(SMTConfigConstants.BROADCAST_EVENT_INBOX_REFRESH)

        LocalBroadcastManager.getInstance(mContext).registerReceiver(mMessageReceiver, filterClickDismiss)
    }

    // handler for received Intents. This will be called whenever an Intent
    // with an action named "custom-event- PN- DISMISS / PN click or inbox refresh" will be received.
    private val mMessageReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            // Get extra data included in the Intent
            SMTLogger.d("receiver", "Got message and action is ${intent.action}: ")
            updateInboxMessage(context, intent)
        }
    }

    /**
     * Handled broadcast recieved for PN Clicked/Dismissed or Inbox data refreshed
     * @param context app context
     * @param intent intent data received
     */
    private fun updateInboxMessage(context: Context, intent: Intent) {
        val event = intent.action
        // handles inbox data refresh case
        if (event == SMTConfigConstants.BROADCAST_EVENT_INBOX_REFRESH) {
            val notifications = SMTDataBaseService.getInstance(WeakReference(context)).getInboxMessages(SMTInboxMessageType.Type.INBOX_MESSAGE)
            if (notifications != null && notifications.size > 0) {
                (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.setMessages(notifications)
            }
        } else {
            val notifData: Any? = intent.extras?.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL)

            val trid = if (notifData is SMTNotificationData) {
                notifData.mTrid
            } else if (notifData is SMTCarouselSetup) {
                notifData.trid
            } else {
                ""
            }

            val index = getInboxItemPosition(trid)
            // handles PN click, just updates the item status in the adapter
            if (event == SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED) {
                (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.updateItemStatus(index, SMTAppInboxTable.STATUS_CLICKED)
            }
            // handles PN Dissmissed event
            else if (event == SMTConfigConstants.BROADCAST_EVENT_PN_DISMISSED) {
                if (index > -1) {
                    handleInboxDismissEvent(index, false)
                }
            }
        }
    }

    /**
     * Finds out the item position in the adapter using message trid
     */
    private fun getInboxItemPosition(trid: String?): Int {

        if (trid.isNullOrEmpty()) return -1

        val adapterNotifList = (mMessageRecyclerView.adapter as? SMTInboxNotificationAdapter)?.getNotificationList()

        adapterNotifList?.let {
            for (i in 0..(it.size - 1)) {
                val notif = it[i]
                if (trid == notif.mTrid) {
                    return i
                }
            }
        }

        return -1
    }

    override fun onDestroy() {
        // unregister the reciever
        androidx.localbroadcastmanager.content.LocalBroadcastManager.getInstance(mContext).unregisterReceiver(mMessageReceiver)
        //clear the scroll listener
        mMessageRecyclerView.clearOnScrollListeners()
        super.onDestroy()
    }

    /**
     * Toggles back button visibility
     */
    private fun toggleBackButton() {
        if (isDisableBack) {
            mView.findViewById<RelativeLayout>(R.id.ivBack).visibility = View.GONE
        } else {
            mView.findViewById<RelativeLayout>(R.id.ivBack).visibility = View.VISIBLE
        }
    }
}