@file:Suppress("PropertyName")

package com.netcore.android.db

import android.content.ContentValues
import android.database.Cursor
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.event.SMTEventSyncStatus
import com.netcore.android.event.SMTEventType
import com.netcore.android.inapp.model.SMTEvents
import com.netcore.android.logger.SMTLogger
import org.json.JSONObject

@Suppress("UNUSED_PARAMETER", "PrivatePropertyName")
/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Event DB Table, All event related operations has handled here
 *
 * @see SMTDataBaseWrapper for method definitions
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTEventTable(private val wrapper: SMTDataBaseWrapper) : SMTDBTable(wrapper) {


    private val TAG = SMTEventTable::class.java.simpleName

    // Event Table Columns names
    private val KEY_ID = "id" //(Primary Key) I
    private val KEY_EVENT_ID = "eventId"   // Column II
    private val KEY_EVENT_NAME = "eventName" //Column III
    private val KEY_EVENT_PAYLOAD = "payload" //Column IV
    private val KEY_EVENT_TIME = "time" //Column V
    private val KEY_EVENT_TYPE = "type"  // Column VI (Custom / System)

    companion object {
        const val KEY_EVENT_SYNC_STATUS = "syncStatus"  // Column VII (PENDING/INPROGRESS/DONE)
    }

    private val mTableName = "event"

    override fun createTable() {
        executeQuery(getEventTableCreateStatement())
    }

    /**
     * prepares the event table creation query statement
     */
    private fun getEventTableCreateStatement(): String {
        return ("CREATE TABLE IF NOT EXISTS " + mTableName
                + " ( "
                + KEY_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, "
                + KEY_EVENT_ID + " INTEGER," // Event ID
                + KEY_EVENT_NAME + " TEXT," // Name of the event
                + KEY_EVENT_PAYLOAD + " TEXT NOT NULL," // Event payload
                + KEY_EVENT_TIME + " LONG NOT NULL," // Time of the event on which it is fired
                + KEY_EVENT_TYPE + " TEXT," // Type of event(System / Custom)
                + KEY_EVENT_SYNC_STATUS + " INTEGER NOT NULL DEFAULT 0" // Type of event(System / Custom)
                + " ) ")
    }

    override fun upgradeTable(oldVersion: Int, newVersion: Int) {
        // if needs to do any action for updating the table

    }

    /**
     * Inserting event to event table
     * @param eventId Id of event
     * @param eventName Name of event
     * @param payload Payload of event
     * @param type Type of event
     */
    fun insertEvent(eventId: Int?, eventName: String?, payload: String, @SMTEventType.Type type: String) {

        val values = ContentValues()
        values.put(KEY_EVENT_ID, eventId ?: 0)
        values.put(KEY_EVENT_NAME, eventName ?: "")
        values.put(KEY_EVENT_PAYLOAD, payload)
        values.put(KEY_EVENT_TIME, System.currentTimeMillis())
        values.put(KEY_EVENT_TYPE, type)
        values.put(KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_PENDING)

        wrapper.insert(mTableName, null, values)
    }

    /**
     * Update event column with respective value
     */
    fun updateEvent(rowId: Int, columnName: String, columnValue: Int) {
        val values = ContentValues()
        values.put(columnName, columnValue)
        val selection = "$KEY_ID = ?"
        wrapper.update(mTableName, values, selection, arrayOf(rowId.toString()))
    }

    /**
     * Updated events sync status
     * @param rowIds event row ids which will be updated
     * @param columnName the column which needs to be updated
     * @param columnValue the value with which the column to be updated
     */
    fun updateMultipleRowsWithSameColumnValue(rowIds: Array<Int>, columnName: String, columnValue: Int) {
        val values = ContentValues()
        values.put(columnName, columnValue)
        val selection = "$KEY_ID = ?"
        wrapper.beginTransaction()
        try {
            for (id in rowIds) {
                val count = wrapper.update(mTableName, values, selection, arrayOf(id.toString()))
            }
            wrapper.setTransactionSuccessful()
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while updating multiple events $e")
        } finally {
            wrapper.endTransaction()
        }
    }

    /**
     * Updated events sync status
     * @param columnValue the value with which the column to be updated
     */
    fun updateMultipleRowsWithSameColumnValue(columnValue: Int) {
        val query = "UPDATE $mTableName SET $KEY_EVENT_SYNC_STATUS = $columnValue WHERE $KEY_EVENT_SYNC_STATUS = ${SMTEventSyncStatus.EVENT_SYNC_STATUS_INPROGRESS}"
        executeQuery(query)
    }

    /**
     * Method to update the payload of failed requests
     * it set retry : 1 for failed batch requests
     */
    fun updateFailedBatchPayload() {

        val selection = "SELECT $KEY_ID, $KEY_EVENT_PAYLOAD FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS = ${SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED}"

        val cursor = wrapper.rawQuery(selection, null)
        val hashMap = HashMap<Int, String>()

        // Move to first row
        if (cursor?.moveToFirst() == true) {
            do {
                // fetch the payload and update
                val payload = cursor.getString(cursor.getColumnIndex(KEY_EVENT_PAYLOAD))
                val id = cursor.getInt(cursor.getColumnIndex(KEY_ID))
                val jsonObject = JSONObject(payload)
                try {
                    jsonObject.put("retry", 1)
                    hashMap[id] = jsonObject.toString()
                } catch (e: Exception) {
                    SMTLogger.e(TAG, "error while updating batch payload $e")
                }
            } while (cursor.moveToNext())
        }

        // Insert the updated payload into Table
        for ((key, value) in hashMap) {
            try {
                val values = ContentValues()
                values.put("payload", value)
                val selection1 = "id = ?"
                wrapper.update(mTableName, values, selection1, arrayOf("$key"))
                // SMTLogger.v(TAG, "Failed event retry update: $key  and value $value")
            } catch (e: Exception) {
                SMTLogger.e(TAG, "exception got while updating row $e")
            }
        }
        cursor?.close()
    }

    /**
     * Delete events from event table
     * @param rowIds event row ids which will be deleted
     */
    fun deleteMultipleEventRows(rowIds: Array<Int>) {
        wrapper.beginTransaction()
        val whereClause = "$KEY_ID = ?"
        try {
            for (id in rowIds) {
                val count = wrapper.delete(mTableName, whereClause, arrayOf(id.toString()))
            }
            wrapper.setTransactionSuccessful()
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while updating multiple events $e")
        } finally {
            wrapper.endTransaction()
        }
    }

    /**
     * Delete events from event table
     */
    fun deleteEventsWithInProgress() {
        val query = "DELETE FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS = ${SMTEventSyncStatus.EVENT_SYNC_STATUS_INPROGRESS}"
        executeQuery(query)
    }


    /**
     * Retrieves Array of events from DB using batch size
     * @param batchSize size of each batch
     * @return returns arraylist of arraylist of hashmap of events
     */
    fun getEventsArrayForBatchProcessing(batchSize: Int): ArrayList<ArrayList<HashMap<String, String>>> {
        val outerlist = ArrayList<ArrayList<HashMap<String, String>>>()
        var innerList = ArrayList<HashMap<String, String>>()
        val selection = "SELECT * FROM $mTableName"
        val cursor = wrapper.rawQuery(selection, null)
        // Move to first row
        if (cursor?.moveToFirst() == false)
            return outerlist

        cursor?.let {
            if (it.count <= batchSize) {
                //read from cursor at once and return
                do {
                    val map = HashMap<String, String>()
                    map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                    innerList.add(map)
                } while (it.moveToNext())
                outerlist.add(innerList)
            } else {
                // Create different maps with batch size and add to list
                var counter = 0
                innerList = ArrayList()
                do {
                    var map = HashMap<String, String>()
                    map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                    counter++
                    innerList.add(map)
                    if (counter >= batchSize) {
                        outerlist.add(innerList)
                        innerList = ArrayList()
                        counter = 0
                    }
                } while (it.moveToNext())
                outerlist.add(innerList)
            }
        }
        cursor?.close()
        return outerlist
    }

    /**
     * Retrieves HaspMap of events from DB using batch size
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */
    fun getEventsMapWithSize(batchSize: Int, startIndex: Int): HashMap<String, String> {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS NOT ${SMTEventSyncStatus.EVENT_SYNC_STATUS_INPROGRESS}"
        val map = HashMap<String, String>()
        val cursor = wrapper.rawQuery(selection, null)
        // Move to first row
        if (cursor?.moveToFirst() == false)
            return map

        var count = 0
        cursor?.let {
            do {
                count++
                map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                if (count >= batchSize) {
                    break
                }
            } while (it.moveToNext())
        }
        cursor?.close()
        return map
    }


    /**
     * Retrieves HaspMap of In progress events from DB using batch size
     * @param batchSize size of each batch
     * @return returns hashmap of events
     */
    fun getInProgressEventsMapWithSize(batchSize: Int): HashMap<String, String> {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_INPROGRESS}"
        val map = HashMap<String, String>()
        val cursor = wrapper.rawQuery(selection, null)
        // Move to first row
        if (cursor?.moveToFirst() == false)
            return map

        var count = 0
        cursor?.let {
            do {
                count++
                map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                if (count >= batchSize) {
                    break
                }
            } while (it.moveToNext())
        }
        cursor?.close()
        return map
    }

    /**
     * Retrieves HaspMap of pending events from DB using batch size
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */
    fun getPendingEventsMapWithSize(batchSize: Int, startIndex: Int): HashMap<String, String> {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_PENDING}"
        val map = HashMap<String, String>()
        val cursor = wrapper.rawQuery(selection, null)
        // Move to first row
        if (cursor?.moveToFirst() == false)
            return map

        var count = 0
        cursor?.let {
            do {
                count++
                map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                if (count >= batchSize) {
                    break
                }
            } while (it.moveToNext())
        }
        cursor?.close()
        return map
    }


    /**
     * Retrieves HaspMap of events from DB using batch size for Pending and Failed events
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */
    fun getPendingAndFailedEventsMapWithSize(batchSize: Int, startIndex: Int): HashMap<String, String> {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_PENDING} OR $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED}"
        val map = HashMap<String, String>()
        val cursor = wrapper.rawQuery(selection, null)
        // Move to first row
        if (cursor?.moveToFirst() == false)
            return map

        var count = 0
        cursor?.let {
            do {
                count++
                map[it.getInt(it.getColumnIndex(KEY_ID)).toString()] = it.getString(it.getColumnIndex(KEY_EVENT_PAYLOAD))
                if (count >= batchSize) {
                    break
                }
            } while (it.moveToNext())
        }
        cursor?.close()
        return map
    }

    /**
     * checks for events more that batchsize present in table for batch processing
     * @param batchSize size of the batch
     * @return Boolean - returns tru or false
     */
    fun isMoreEventsPresentForProcessing(batchSize: Int): Boolean {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_PENDING} OR $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED}"
        val cursor = wrapper.rawQuery(selection, null)
        cursor?.let {
            SMTLogger.internal(TAG, "Event count is ${it.count}")
            if (it.count > 0) {
                return true
            }
        }
        cursor?.close()
        return false
    }

    /**
     * checks for Pending and Failed events more that batchsize present in table for batch processing
     * @param batchSize size of the batch
     * @return Boolean - returns tru or false
     */
    fun isMoreEventsPendingFailedPresentForProcessing(batchSize: Int): Boolean {
        val selection = "SELECT * FROM $mTableName WHERE $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_PENDING} OR $KEY_EVENT_SYNC_STATUS IS  ${SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED}"
        val cursor = wrapper.rawQuery(selection, null)
        cursor?.let {
            if (it.count > 0) {
                return true
            }
        }
        cursor?.close()
        return false
    }


    fun executeRawTestQuery(query: String): Cursor? {
        val finalQuery = query + mTableName
        return executeRawQuery(finalQuery)
    }

    internal fun getStoredEventData(payloadMap: java.util.HashMap<String, Any>): MutableList<SMTEvents> {

        val eventList: MutableList<SMTEvents> = mutableListOf()
        val eventId: String? = payloadMap[SMTEventParamKeys.SMT_EVENT_ID]?.toString() ?: ""
        val eventName: String? = payloadMap[SMTEventParamKeys.SMT_EVENT_NAME]?.toString() ?: ""
        val eventTime: String? = payloadMap[SMTEventParamKeys.SMT_EVENT_TIME]?.toString() ?: ""

        val query = if (eventId != null && eventId.trim().isNotEmpty() && eventId.toInt() > 0) {
            "select * from $mTableName where $KEY_EVENT_ID = $eventId AND $KEY_EVENT_TIME >= $eventTime ORDER BY $KEY_EVENT_TIME DESC"
        } else if (!eventName.isNullOrEmpty()) {
            "select * from $mTableName where $KEY_EVENT_NAME like '$eventName' AND $KEY_EVENT_TIME >= $eventTime ORDER BY $KEY_EVENT_TIME DESC"
        } else ""
        //println("inapp query:$query")
        val cursor = wrapper.rawQuery(query, null)

        cursor?.let {
            if (it.moveToFirst()) {
                val smtEvents = SMTEvents()
                smtEvents.id = cursor.getString(cursor.getColumnIndex(KEY_ID))
                smtEvents.eventId = cursor.getString(cursor.getColumnIndex(KEY_EVENT_ID))
                smtEvents.eventName = cursor.getString(cursor.getColumnIndex(KEY_EVENT_NAME))
                smtEvents.eventDate = cursor.getLong(cursor.getColumnIndex(KEY_EVENT_TIME)).toString()
                smtEvents.eventPayload = cursor.getString(cursor.getColumnIndex(KEY_EVENT_PAYLOAD))

                eventList.add(smtEvents)
            }
        }
        cursor?.close()
        //println("size : ${eventList.size}")

        return eventList
    }

    /**
     * Delete events from event table
     * @param rowIds event row ids which will be deleted
     */
    fun deleteMultipleEvents(limit: Int) {
        val query = "select * from $mTableName ORDER BY $KEY_EVENT_TIME ASC"
        val cursor = wrapper.rawQuery(query, null)
        // Move to first row
        cursor?.let {
            if (it.moveToFirst()) {
                var count = it.count
                do {
                    //println("count $count limit $limit")
                    if (count <= limit) {
                        break
                    }
                    val id = cursor.getString(cursor.getColumnIndex(KEY_ID))
                    //println("Event delete :$id")
                    wrapper.delete(mTableName, "$KEY_ID = ?", arrayOf(id.toString()))
                    count--
                } while (cursor.moveToNext())
            }
        }
        cursor?.close()
    }
}