package com.netcore.android.notification.gif

import android.app.Notification
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.os.Build
import android.os.Bundle
import android.view.View
import android.widget.RemoteViews
import androidx.core.app.NotificationCompat
import com.netcore.android.R
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTDownloaderUtility
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.*
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference

@Suppress("UNUSED_PARAMETER")
internal class SMTRichGifPNGenerator private constructor(private val context: WeakReference<Context>) : SMTBaseNotificationGenerator() {

    private val TAG = SMTRichGifPNGenerator::class.java.simpleName

    private var mGifImageRetriever: GifRetriever? = null
    private var mCurrentBitmapFrame: Bitmap? = null
    private var notificationBuilder: NotificationCompat.Builder? = null
    private var notifModel: SMTNotificationData? = null

    private val lock = Any()

    companion object {
        @Volatile
        private var INSTANCE: SMTRichGifPNGenerator? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: WeakReference<Context>): SMTRichGifPNGenerator =
                INSTANCE ?: synchronized(SMTRichGifPNGenerator::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: WeakReference<Context>): SMTRichGifPNGenerator {
            return SMTRichGifPNGenerator(context)
        }
    }

    fun handle(context: Context, notif: SMTNotificationData, sourceType: Int) {

        // check if any GIF is laready playing then first stop it
        // then process the new one
        synchronized(lock) {
            notifModel?.let {
                stopCurrentGifAnimation(it)
            }
        }

        notif.notificationId = SMTCommonUtility.getRandomId()

        updateNotificationId(context, notif)

        notifModel = notif

        notifModel?.let {
            fetchGif(context, it, sourceType)
        }
    }

    /**
     * Downloads the GIF file and updates the view
     * @param context App context
     * @param notifData notification data model
     */
    private fun fetchGif(context: Context, notifData: SMTNotificationData, sourceType: Int) {

        val notifOption = SMTCommonUtility.getNotificationOptions(context)
        // show scheduled local GIF notification
        if (notifModel?.mIsScheduledPN == 1) {
            notifModel = notifData
            fetchBitmapFromGif(notifData, context, notifOption)
            createNotification(context, notifData)
            notifyNotificationManger(notificationBuilder?.build()!!, notifData.notificationId)
        } else {         // show GIF notification
            SMTMediaDownloadManager().downloadMedia(context, notifData, object : SMTMediaDownloadManager.MediaDownloadListener {
                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    try {
                        if (notification.mMediaLocalPath?.equals("", ignoreCase = true) == false) {
                            // if current gif item is same as downloaded gif then show it
                            // else show the place holder icon with Gif play icon
                            if (notifModel?.mTrid == notification.mTrid) {
                                notifModel = notification
                                fetchBitmapFromGif(notification, context, notifOption)
                                createNotification(context, notification)
                                notifyNotificationManger(notificationBuilder?.build()!!, notification.notificationId)
                            } else {
                                showGifWithPlaceholderImage(context, notifOption, notification, true)
                            }
                        } else {
                            showGifWithPlaceholderImage(context, notifOption, notification, false)
                        }
                    } catch (e: Exception) {
                        SMTLogger.e(TAG, e.message.toString())
                    }
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    showGifWithPlaceholderImage(context, notifOption, notification, false)
                }
            })

        }
    }

    /**
     * Creates notification builder and notification manager
     * @param context App context
     * @param notifModel Notification model
     */
    private fun createNotification(context: Context, notifModel: SMTNotificationData) {

        notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        notificationBuilder = getNotificationBuilder(context,
                notifModel.mTitle ?: "",
                notifModel.mMessage ?: "",
                notifModel.mSubtitle ?: "",
                createPendingIntent(context, notifModel),
                notifModel)
    }

    /**
     * Shows GIF notification with placeholder image
     * @param context app context
     * @param notifOption Notificaiton options
     */
    private fun showGifWithPlaceholderImage(context: Context, notifOption: SMTNotificationOptions, notification: SMTNotificationData, showGifIcon: Boolean) {


        val remoteView = RemoteViews(context.packageName, R.layout.notification_gif_layout)

        if (showGifIcon) {
            remoteView.setImageViewBitmap(R.id.gif_image, SMTCommonUtility.getBitmapFromResId(context, notifOption.placeHolderIcon))
            remoteView.setViewVisibility(R.id.gif_icon_play, View.VISIBLE)
            remoteView.setOnClickPendingIntent(R.id.gif_icon_play, getGifPlayPendingIntent(notification))
            setRemoteViewTitles(remoteView, notification)
            notificationBuilder?.setCustomBigContentView(remoteView)
            setStyleToBuilder(remoteView)
            notifyNotificationManger(notificationBuilder?.build()!!, notification.notificationId)
        } else {
            remoteView.setViewVisibility(R.id.gif_icon_play, View.GONE)
            // show simple notification if GIF image is not available / downloaded
            SMTSimplePNGenerator().handle(context, notification)
        }

    }

    /**
     * Set the text to text views
     * @param remoteView the remote view layout which contains the Textviews
     */
    private fun setRemoteViewTitles(remoteView: RemoteViews, notification: SMTNotificationData) {
        remoteView.setTextViewText(R.id.gif_title, notification.mTitle)
        remoteView.setTextViewText(R.id.gif_message, notification.mMessage)
    }

    /**
     * Fetches Bitmap form downloaded GIF fiel
     * @param notification - Local media storage path
     * @param context App context
     * @param notifOption Notificaiton options
     */
    private fun fetchBitmapFromGif(notification: SMTNotificationData, context: Context, notifOption: SMTNotificationOptions) {

        mGifImageRetriever = GifRetriever()

        mGifImageRetriever?.onFrameAvailable = GifRetriever.OnFrameAvailable { bitmap ->
            bitmap?.let {
                synchronized(lock) {
                    var showGifIcon = false

                    // If GIF looped once then stop it
                    // If user wants to see it again then have to click on GIF play button

                    if ((mGifImageRetriever?.currentFrameIndex == (mGifImageRetriever?.frameCount!! - 1))) {
                        showGifIcon = true

                        stopCurrentGifAnimation(notification)
                    }
                    mCurrentBitmapFrame = it

                    // update the notification tray
                    refreshRemoteView(it, context, showGifIcon, notifOption, notification)
                }
            }
        }
        mGifImageRetriever?.setBytes(SMTDownloaderUtility.getGifBytes(notification.mMediaLocalPath))
        mGifImageRetriever?.startAnimation()
    }

    private fun stopCurrentGifAnimation(notification: SMTNotificationData) {
        // Dont stop anitmation of other GIF
        // use case cancelling one gif stops other gif animation
        if (notifModel?.mTrid != notification.mTrid) return

        val isAnimating = mGifImageRetriever?.isAnimating ?: false
        mGifImageRetriever?.clear()

        if (isAnimating) {
            mCurrentBitmapFrame?.let {
                refreshRemoteView(it, context.get()!!, true, SMTCommonUtility.getNotificationOptions(context.get()!!), notification)
            }
        }

        mGifImageRetriever = null
        mCurrentBitmapFrame = null
    }

    /**
     * Update the remoteview with the New Bitmap frame
     * @param bitmap - next fetched frame
     * @param context App context
     * @param showGifIcon - whether to show GIF icon or not
     * @param notifOption Notificaiton options
     */
    private fun refreshRemoteView(bitmap: Bitmap, context: Context, showGifIcon: Boolean, notifOption: SMTNotificationOptions, notification: SMTNotificationData) {
        // notification's layout
        val remoteView = RemoteViews(context.packageName, R.layout.notification_gif_layout)

        remoteView.setImageViewBitmap(R.id.gif_image, bitmap)

        if (showGifIcon) {
            remoteView.setViewVisibility(R.id.gif_icon_play, View.VISIBLE)
            remoteView.setOnClickPendingIntent(R.id.gif_icon_play, getGifPlayPendingIntent(notification))
        } else {
            remoteView.setViewVisibility(R.id.gif_icon_play, View.GONE)
        }
        remoteView.setImageViewBitmap(R.id.gif_large_icon, SMTCommonUtility.getBitmapFromResId(context, notifOption.largeIconId))

        setRemoteViewTitles(remoteView, notification)

        createNotification(context, notification)

        notificationBuilder?.setCustomBigContentView(remoteView)
        setStyleToBuilder(remoteView)
        // if GIF has been clicked then dont again show it
        // It has to be cleared
        notifyNotificationManger(notificationBuilder?.build()!!, notification.notificationId)
    }

    /**
     * Show notificaiton on system tray
     * @param notification - Notificaiton reference
     * @param randomNotifyId - Notification ID
     */
    private fun notifyNotificationManger(notification: Notification, randomNotifyId: Int) {
        synchronized(lock) {
            notificationManager?.notify(randomNotifyId, notification)
        }
    }

    /**
     * Notification Action receiver Pending Intent created
     * @param context - App context
     * @param notifModel - Parsed Notification data model
     * @return PendingIntent - receiver pending intent
     */
    private fun createPendingIntent(context: Context, notifModel: SMTNotificationData): PendingIntent {
        val rand = System.currentTimeMillis().toInt()

        val bundle = Bundle()
        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, notifModel.mNotificationType)
        bundle.putInt(SMTNotificationConstants.GIF_ITEM_CLICKED_KEY, SMTNotificationConstants.GIF_OTHER_REGION_CLICKED)
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notifModel)

        val launchIntent = Intent(context, SMTPNActionReceiver::class.java)

        launchIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context, rand, launchIntent, PendingIntent.FLAG_UPDATE_CURRENT)
    }

    /**
     * cleares the file from local storage
     * on notification dismiss
     */
    override fun handleNotificationDismiss(context: Context?, extras: Bundle) {
        if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            val parcel = extras.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL) as SMTNotificationData
            // clear the cache
            clearGifNotification(parcel)
        }
    }

    /**
     * cleares the file from local storage
     * on notification dismiss and also clears the bitmap from GIF decoder
     */
    private fun clearGifNotification(parcel: SMTNotificationData) {
        // Stop the animation
        stopCurrentGifAnimation(parcel)

        // delete the file from file system
        if (parcel.mMediaLocalPath?.isNotEmpty() == true) {
            SMTCommonUtility.deleteFile(parcel.mMediaLocalPath!!)
        }
    }

    /**
     * Handles the image click action
     * @param context App context
     * @param extras - Extras passed through the pending intent
     */
    fun handleContentClick(context: Context, extras: Bundle) {
        if (extras.containsKey(SMTNotificationConstants.GIF_ITEM_CLICKED_KEY)) {
            val parcel = extras.getParcelable<SMTNotificationData>(SMTNotificationConstants.NOTIFICATION_PARCEL)
            when (extras.getInt(SMTNotificationConstants.GIF_ITEM_CLICKED_KEY)) {
                SMTNotificationConstants.GIF_PLAY_BUTTON_CLICKED -> {
                    parcel?.let {
                        synchronized(lock) {
                            // stop current playing gif first
                            // then play the second one
                            notifModel?.let { currentNotif ->
                                stopCurrentGifAnimation(currentNotif)
                            }

                            // Create Notification builder
                            createNotification(context, it)

                            notifModel = it

                            // Start Animation
                            fetchBitmapFromGif(it, context, SMTCommonUtility.getNotificationOptions(context))
                        }
                    }
                }
                //
                SMTNotificationConstants.GIF_OTHER_REGION_CLICKED -> {
                    parcel?.let {
                        synchronized(lock) {

                            clearGifNotification(it)

                            //Update DB record whether it has been clicked
                            SMTEventRecorder.getInstance(context).recordNotificationClick(it.mTrid, it.mPNMeta,
                                    it.mDeepLinkPath ?: "", it.mSource, it.mSmtAttributePayload
                                    ?: HashMap(), it.mIsScheduledPN)

                            // Handle Deeplink
                            SMTCommonUtility.handleNotificationClick(context, it.mDeepLinkPath
                                    ?: "", it.mCustomPayload)

                            // Dismiss the notification from tray
                            val mNotifyManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                            mNotifyManager.cancel(it.notificationId)
                        }
                    }
                }
            }
        }
    }


    /**
     * Creates actionable pending intent as per event id
     */
    private fun getGifPlayPendingIntent(notification: SMTNotificationData): PendingIntent? {
        var carouselIntent: Intent
        context.get()?.let {
            carouselIntent = Intent(it, SMTPNActionReceiver::class.java)
            val bundle = Bundle()
            bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, notification.mNotificationType!!)
            bundle.putInt(SMTNotificationConstants.GIF_ITEM_CLICKED_KEY, SMTNotificationConstants.GIF_PLAY_BUTTON_CLICKED)
            bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notification)
            carouselIntent.putExtras(bundle)
            return PendingIntent.getBroadcast(it, SMTCommonUtility.getRandomId(), carouselIntent, PendingIntent.FLAG_ONE_SHOT)
        }

        return null
    }

    private fun setStyleToBuilder(remoteView: RemoteViews) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            remoteView.setViewVisibility(R.id.gif_large_icon, View.GONE)
            notificationBuilder?.setStyle(NotificationCompat.DecoratedCustomViewStyle())
            notificationBuilder?.setLargeIcon(null)
        }
    }
}