package com.netcore.android.mediadownloader

import android.content.Context
import android.content.ContextWrapper
import android.graphics.Bitmap
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationType
import java.io.*
import java.lang.ref.WeakReference
import java.security.MessageDigest

internal object SMTDownloaderUtility {
    private val fileDirectory: String = "SmarTechDirectory"
    private val mInboxDirectory: String = "SmtInboxDirectory"
    private val filePrefix = "SmtFile"
    val TAG: String = SMTDownloaderUtility::class.java.simpleName

    /**
     * Calculate the sample size for image scaling
     *
     */
    fun calculateInSampleSize(
            width: Int, height: Int, reqWidth: Int, reqHeight: Int): Int {
        SMTLogger.d(TAG, "Calculating In sample size")
        var inSampleSize = 1

        if (height > reqHeight || width > reqWidth) {

            val halfHeight = height / 2
            val halfWidth = width / 2

            while (halfHeight / inSampleSize >= reqHeight && halfWidth / inSampleSize >= reqWidth) {
                inSampleSize *= 2
            }
        }
        SMTLogger.d(TAG, "Calulated In sample size $inSampleSize")
        return inSampleSize
    }

    /**
     * Method to store bitmap into the internal storage
     * @param context
     * @param bitmapImage downloaded image bitmap
     */

    fun saveBitmapToInternalStorage(bitmapImage: Bitmap, file: File): String? {
        SMTLogger.d(TAG, "Saving Bitmap Image")
        var fileSaved = false
        var fos: FileOutputStream? = null
        try {
            fos = FileOutputStream(file)
            bitmapImage.compress(Bitmap.CompressFormat.PNG, 100, fos)
            fileSaved = true
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Image bitmap saving error ${e.localizedMessage}")
        } finally {
            try {
                fos!!.close()
            } catch (e: IOException) {
                SMTLogger.e(TAG, e.message.toString())
            }

        }
        SMTLogger.d(TAG, "Is Image file saved $fileSaved")
        return if (fileSaved)
            file.absolutePath
        else
            null
    }

    fun saveFileToInternalStorage(ins: InputStream, file: File): String? {
        SMTLogger.d(TAG, "Saving Media file")
        var fileSaved = false
        var output: FileOutputStream? = null
        try {
            // Output stream to write file
            output = FileOutputStream(file)

            val data = ByteArray(1024)
            var count: Int

            do {
                count = ins.read(data)
                if (count == -1) {
                    break
                }
                output.write(data, 0, count)


            } while (true)
            output.flush()
            fileSaved = true
            SMTLogger.d(TAG, "Is file saved $fileSaved")
        } catch (e: Exception) {
            SMTLogger.d(TAG, "Saving media file save failed ${e.localizedMessage}")
        } finally {
            try {
                output!!.close()
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }

        }
        if (fileSaved) {
            SMTLogger.d(TAG, "File saved ${file.absolutePath}")
            return file.absolutePath
        } else {
            return null
        }
    }

    /**
     * Supported algorithms on Android:
     *
     * Algorithm	Supported API Levels
     * MD5          1+
     * SHA-1	    1+
     * SHA-224	    1-8,22+
     * SHA-256	    1+
     * SHA-384	    1+
     * SHA-512	    1+
     */
    private fun hashSha1String(input: String): String {
        val HEX_CHARS = "0123456789ABCDEF"
        val bytes = MessageDigest
                .getInstance("SHA-1")
                .digest(input.toByteArray())
        val result = StringBuilder(bytes.size * 2)

        bytes.forEach {
            val i = it.toInt()
            result.append(HEX_CHARS[i shr 4 and 0x0f])
            result.append(HEX_CHARS[i and 0x0f])
        }
        return result.toString()
    }

    fun deleteInboxDirectory(context: WeakReference<Context>) {
        SMTLogger.d("SMTDownloaderUtility", "deleting inbox directory")
        if (context.get() != null) {
            val dir = getDirectory(context.get()!!, true)
            if (dir.isDirectory) {
                val children = dir.list()
                for (i in children.indices) {
                    File(dir, children[i]).delete()
                }
            }
        }
    }

    /**
     * Method to create file
     */
    fun getDownloadFile(context: Context, url: String, type: String, isInbox: Boolean = false): File {

        SMTLogger.d(TAG, "Creating file")
        return File(getDirectory(context, isInbox), getFileName(url, type))
    }

    private fun getDirectory(context: Context, isInbox: Boolean): File {
        val cw = ContextWrapper(context.applicationContext)
        return if (isInbox) {
            cw.getDir(mInboxDirectory, Context.MODE_PRIVATE)
        } else {
            cw.getDir(fileDirectory, Context.MODE_PRIVATE)
        }
    }

    private fun getFileName(url: String, type: String): String {
        return filePrefix.plus("_").plus(System.currentTimeMillis()).plus(getFileExtensionFromUrl(url, type))

    }

    private fun getFileExtensionFromUrl(url: String, type: String): String? {
        var extension: String? = null
        if (url.contains(".")) {
            extension = url.substring(url.lastIndexOf("."))
        }
        if (extension != null)
            if (isExtValid(extension)) {
                SMTLogger.d(TAG, "File extension from url $extension")
                return extension
            }


        if (type == SMTNotificationType.BIG_IMAGE.type
                || type == SMTNotificationType.CAROUSEL_PORTRAIT.type
                || type == SMTNotificationType.CAROUSEL_LANDSCAPE.type
                || type == SMTNotificationType.SIMPLE.type) {
            extension = ".jpg"
        } else if (type == SMTNotificationType.GIF.type) {
            extension = ".gif"
        } else if (type == SMTNotificationType.VIDEO.type) {
            extension = ".mp4"
        } else if (type == SMTNotificationType.AUDIO.type) {
            extension = ".mp3"
        }
        SMTLogger.d(TAG, "File extension from notification type $extension")
        return extension

    }

    /**
     * method to check if the file extension is valid
     */
    private fun isExtValid(extension: String): Boolean {
        val strings = arrayOf(".jpg", ".JPG", ".JPEG", ".jpeg", ".png", ".PNG", ".gif", ".GIF", ".mp3", ".MP3", ".mp4", ".MP4", ".3gp", ".3GP", ".wma", ".WMA", ".wav", ".WAV")
        val isValid = strings.contains(extension)
        SMTLogger.d(TAG, "Validating the url extension $isValid")
        return isValid
    }

    fun getGifBytes(mMediaLocalPath: String?): ByteArray {
        val file = File(mMediaLocalPath)
        val size = file.length().toInt()
        val bytes = ByteArray(size)
        try {
            val buf = BufferedInputStream(FileInputStream(file))
            buf.read(bytes, 0, bytes.size)
            buf.close()
        } catch (e: FileNotFoundException) {
            SMTLogger.e(TAG, e.message.toString())
        } catch (e: IOException) {
            SMTLogger.e(TAG, e.message.toString())
        }
        return bytes
    }
}